package IO::Iron::IronMQ::Api;

## no critic (Documentation::RequirePodAtEnd)
## no critic (Documentation::RequirePodSections)

use 5.010_000;
use strict;
use warnings;

# Global Creator
BEGIN {
	# No exports.
}

# Global Destructor
END {
}


# ABSTRACT: IronMQ API reference for Perl Client Libraries!

our $VERSION = '0.12_01'; # TRIAL VERSION: generated by DZP::OurPkgVersion




sub IRONMQ_CREATE_QUEUE {
	return {
			'action_name'  => 'IRONMQ_CREATE_QUEUE',
			'href'         => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}',
			'action'       => 'PUT',
			'return'       => 'NONE',
			'retry'        => 1,
			'require_body' => 1,
			'request_fields' => { 'message_timeout' => 1, 'message_expiration' => 1, 'type' => 1, 'push' => 1, 'dead_letter' => 1, },
			'url_escape'   => { '{Project ID}' => 1, '{Queue Name}' => 1 },
			'log_message'  => '(project={Project ID}, queue={Queue Name}). Created a message queue.',
		};
}


sub IRONMQ_GET_QUEUE_INFO {
	return {
			'action_name'  => 'IRONMQ_GET_QUEUE_INFO',
			'href'         => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}',
			'action'       => 'GET',
			'return'       => 'HASH',
			'retry'        => 1,
			'require_body' => 0,
			'url_escape'   => { '{Project ID}' => 1, '{Queue Name}' => 1 },
			'log_message'  => '(project={Project ID}, queue={Queue Name}). Got info about a message queue.',
		};
}


sub IRONMQ_UPDATE_QUEUE {
	return {
			'action_name'  => 'IRONMQ_UPDATE_QUEUE',
			'href'         => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}',
			'action'       => 'PATCH',
			'return'       => 'NONE',
			'retry'        => 1,
			'require_body' => 1,
			'request_fields' => { 'subscribers' => 1, 'push_type' => 1, 'retries' => 1, 'retries_delay' => 1 },
			'url_escape'   => { '{Project ID}' => 1, '{Queue Name}' => 1 },
			'log_message'  => '(project={Project ID}, queue={Queue Name}). Updated a message queue.',
		};
}


sub IRONMQ_DELETE_QUEUE {
	return {
			'action_name'  => 'IRONMQ_DELETE_QUEUE',
			'href'         => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}',
			'action'       => 'DELETE',
			'return'       => 'MESSAGE',
			'retry'        => 1,
			'require_body' => 0,
			'url_escape'   => { '{Project ID}' => 1, '{Queue Name}' => 1 },
			'log_message'  => '(project={Project ID}, queue={Queue Name}). Deleted message queue.',
		};
}


sub IRONMQ_LIST_QUEUES {
	return {
			'action_name'  => 'IRONMQ_LIST_QUEUES',
			'href'         => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues',
			'action'       => 'GET',
			'return'       => 'HASH',
			'retry'        => 0,
			'require_body' => 0,
			'url_params'   => { 'per_page' => 1, 'previous' => 1, 'prefix' => 1, },
			'url_escape'   => { '{Project ID}' => 1 },
			'log_message'  => '(project={Project ID}). Listed queues.',
		};
}


sub IRONMQ_ADD_SUBSCRIBERS_TO_A_MESSAGE_QUEUE {
	return {
			'action_name'  => 'IRONMQ_ADD_SUBSCRIBERS_TO_A_MESSAGE_QUEUE',
			'href'         => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/subscribers',
			'action'       => 'POST',
			'return'       => 'MESSAGE',
			'retry'        => 1,
			'require_body' => 1,
			'request_fields' => { 'subscribers' => 1 },
			'url_escape'   => { '{Project ID}' => 1, '{Queue Name}' => 1 },
			'log_message'  => '(project={Project ID}, queue={Queue Name}). Updated a message queue by adding subscribers.',
		};
}


sub IRONMQ_DELETE_SUBSCRIBERS_FROM_A_MESSAGE_QUEUE {
	return {
			'action_name'  => 'IRONMQ_DELETE_SUBSCRIBERS_FROM_A_MESSAGE_QUEUE',
			'href'         => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/subscribers',
			'action'       => 'DELETE',
			'return'       => 'MESSAGE',
			'retry'        => 1,
			'require_body' => 1,
			'request_fields' => { 'subscribers' => 1 },
			'url_escape'   => { '{Project ID}' => 1, '{Queue Name}' => 1 },
			'log_message'  => '(project={Project ID}, queue={Queue Name}). Updated a message queue by deleting subscribers.',
		};
}


# TODO Bug in documentation: href last '/' not needed.

sub IRONMQ_ADD_ALERTS_TO_A_QUEUE {
	return {
			'action_name'  => 'IRONMQ_ADD_ALERTS_TO_A_QUEUE',
			'href'         => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/alerts',
			'action'       => 'POST',
			'return'       => 'MESSAGE',
			'retry'        => 1,
			'require_body' => 1,
			'request_fields' => { 'alerts' => 1 },
			'url_escape'   => { '{Project ID}' => 1, '{Queue Name}' => 1 },
			'log_message'  => '(project={Project ID}, queue={Queue Name}). Set alerts to the queue.',
		};
}


# TODO Bug in documentation: href last '/' not needed.

sub IRONMQ_REPLACE_ALERTS_ON_A_QUEUE {
	return {
			'action_name'  => 'IRONMQ_REPLACE_ALERTS_ON_A_QUEUE',
			'href'         => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/alerts',
			'action'       => 'PUT',
			'return'       => 'MESSAGE',
			'retry'        => 1,
			'require_body' => 1,
			'request_fields' => { 'alerts' => 1 },
			'url_escape'   => { '{Project ID}' => 1, '{Queue Name}' => 1 },
			'log_message'  => '(project={Project ID}, queue={Queue Name}). Replaced alerts on the queue.',
		};
}


# TODO Bug in documentation: href last '/' not needed.

sub IRONMQ_REMOVE_ALERTS_FROM_A_QUEUE {
	return {
			'action_name'  => 'IRONMQ_REMOVE_ALERTS_FROM_A_QUEUE',
			'href'         => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/alerts',
			'action'       => 'DELETE',
			'return'       => 'MESSAGE',
			'retry'        => 1,
			'require_body' => 1,
			'request_fields' => { 'alerts' => 1 },
			'url_escape'   => { '{Project ID}' => 1, '{Queue Name}' => 1 },
			'log_message'  => '(project={Project ID}, queue={Queue Name}). Removed alerts from the queue.',
		};
}


sub IRONMQ_REMOVE_ALERTS_FROM_A_QUEUE_BY_ID {
	return {
			'action_name'  => 'IRONMQ_REMOVE_ALERTS_FROM_A_QUEUE_BY_ID',
			'href'         => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/alerts/{Alert ID}',
			'action'       => 'DELETE',
			'return'       => 'MESSAGE',
			'retry'        => 1,
			'require_body' => 0,
			'request_fields' => { 'alerts' => 1 },
			'url_escape'   => { '{Project ID}' => 1, '{Queue Name}' => 1, '{Alert ID}' => 1, },
			'log_message'  => '(project={Project ID}, queue={Queue Name}, alert_id={Alert ID}). Removed one alert from the queue.',
		};
}


sub IRONMQ_POST_MESSAGES {
	return {
			'action_name'    => 'IRONMQ_POST_MESSAGES',
			'href'           => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/messages',
			'action'         => 'POST',
			'return'         => 'HASH',
			'retry'          => 1,
			'require_body'   => 1,
			'request_fields' => { 'messages' => 1 },
			'url_escape'     => { '{Project ID}' => 1, '{Queue Name}' => 1 },
			'log_message'    => '(project={Project ID}, queue={Queue Name}). Pushed messages to the queue.',
		};
}


sub IRONMQ_RESERVE_MESSAGES {
	return {
			'action_name'    => 'IRONMQ_RESERVE_MESSAGES',
			'href'           => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/reservations',
			'action'         => 'POST',
			'return'         => 'HASH',
			'retry'          => 1,
			'require_body'   => 1,
			'request_fields' => { 'n' => 1, 'timeout' => 1, 'wait' => 1, 'delete' => 1, },
			'url_escape'     => { '{Project ID}' => 1, '{Queue Name}' => 1 },
			'log_message'    => '(project={Project ID}, queue={Queue Name}). Reserved messages from the queue.',
		};
}


sub IRONMQ_GET_MESSAGE_BY_ID {
	return {
			'action_name'    => 'IRONMQ_GET_MESSAGE_BY_ID',
			'href'           => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/messages/{Message ID}',
			'action'         => 'GET',
			'return'         => 'HASH',
			'retry'          => 1,
			'require_body'   => 0,
			'url_escape'     => { '{Project ID}' => 1, '{Queue Name}' => 1, '{Message ID}' => 1, },
			'log_message'    => '(project={Project ID}, queue={Queue Name}, message_id={Message ID}). Got a message from the queue.',
		};
}


sub IRONMQ_PEEK_MESSAGES {
	return {
			'action_name'    => 'IRONMQ_PEEK_MESSAGES',
			'href'           => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/messages',
			'action'         => 'GET',
			'return'         => 'HASH',
			'retry'          => 1,
			'require_body'   => 0,
			'url_params'     => { 'n' => 1, },
			'url_escape'     => { '{Project ID}' => 1, '{Queue Name}' => 1, },
			'log_message'    => '(project={Project ID}, queue={Queue Name}). Peeked at messages on the queue.',
		};
}


sub IRONMQ_DELETE_MESSAGE {
	return {
			'action_name'    => 'IRONMQ_DELETE_MESSAGE',
			'href'           => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/messages/{Message ID}',
			'action'         => 'DELETE',
			'return'         => 'MESSAGE',
			'retry'          => 1,
			'require_body'   => 1,
			'request_fields' => { 'reservation_id' => 1, },
			'url_escape'     => { '{Project ID}' => 1, '{Queue Name}' => 1, '{Message ID}' => 1, },
			'log_message'    => '(project={Project ID}, queue={Queue Name}, message_id={Message ID}). Deleted a message from the queue.',
		};
}


sub IRONMQ_DELETE_MESSAGES {
	return {
			'action_name'    => 'IRONMQ_DELETE_MESSAGES',
			'href'           => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/messages',
			'action'         => 'DELETE',
			'return'         => 'MESSAGE',
			'retry'          => 1,
			'require_body'   => 1,
			'request_fields' => { 'ids' => 1, },
			'url_escape'     => { '{Project ID}' => 1, '{Queue Name}' => 1, },
			'log_message'    => '(project={Project ID}, queue={Queue Name}). Deleted messages from the queue.',
		};
}


sub IRONMQ_TOUCH_MESSAGE {
	return {
			'action_name'    => 'IRONMQ_TOUCH_MESSAGE',
			'href'           => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/messages/{Message ID}/touch',
			'action'         => 'POST',
			'return'         => 'HASH',
			'retry'          => 1,
			'require_body'   => 1,
			'request_fields' => { 'reservation_id' => 1, 'timeout' => 1, },
			'url_escape'     => { '{Project ID}' => 1, '{Queue Name}' => 1, '{Message ID}' => 1, },
			'log_message'    => '(project={Project ID}, queue={Queue Name}, message_id={Message ID}). Touched a message on the queue.',
		};
}


sub IRONMQ_RELEASE_MESSAGE {
	return {
			'action_name'    => 'IRONMQ_RELEASE_MESSAGE',
			'href'           => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/messages/{Message ID}/release',
			'action'         => 'POST',
			'return'         => 'MESSAGE',
			'retry'          => 1,
			'require_body'   => 1,
			'request_fields' => { 'reservation_id' => 1, 'delay' => 1, },
			'url_escape'     => { '{Project ID}' => 1, '{Queue Name}' => 1, '{Message ID}' => 1, },
			'log_message'    => '(project={Project ID}, queue={Queue Name}, message_id={Message ID}). Released a message on the queue.',
		};
}


sub IRONMQ_CLEAR_MESSAGES {
	return {
			'action_name'    => 'IRONMQ_CLEAR_MESSAGES',
			'href'           => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/messages',
			'action'         => 'DELETE',
			'return'         => 'MESSAGE',
			'retry'          => 1,
			'require_body'   => 1,
			'request_fields' => {},
			'url_escape'     => { '{Project ID}' => 1, '{Queue Name}' => 1, },
			'log_message'    => '(project={Project ID}, queue={Queue Name}). Cleared all messages from the queue.',
		};
}


sub IRONMQ_GET_PUSH_STATUSES_FOR_A_MESSAGE {
	return {
			'action_name'  => 'IRONMQ_GET_PUSH_STATUSES_FOR_A_MESSAGE',
			'href'         => '{Protocol}://{Host}:{Port}/{API Version}/projects/{Project ID}/queues/{Queue Name}/messages/{Message ID}/subscribers',
			'action'       => 'GET',
			'return'       => 'HASH',
			'retry'        => 1,
			'require_body' => 0,
			'url_escape'   => { '{Project ID}' => 1, '{Queue Name}' => 1 },
			'log_message'  => '(project={Project ID}, queue={Queue Name}, message_id={Message ID}). Got push status about a message.',
		};
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

IO::Iron::IronMQ::Api - IronMQ API reference for Perl Client Libraries!

=head1 VERSION

version 0.12_01

=head1 SYNOPSIS

This package is for internal use of IO::Iron::IronMQ::Client/Queue packages.
Only IronMQ v3. Older v1/v2 no longer supported.

=head1 DESCRIPTION

=for stopwords IronMQ API Mikko Koivunalho perldoc CPAN AnnoCPAN ACKNOWLEDGMENTS TODO tradename licensable MERCHANTABILITY

=head1 SUBROUTINES/METHODS

=head2 Operate message queues

=head3 IRONMQ_CREATE_QUEUE

/projects/{Project ID}/queues/{Queue Name}

=head3 IRONMQ_GET_QUEUE_INFO

/projects/{Project ID}/queues/{Queue Name}

=head3 IRONMQ_UPDATE_QUEUE

/projects/{Project ID}/queues/{Queue Name}

=head3 IRONMQ_DELETE_QUEUE

/projects/{Project ID}/queues/{Queue Name}

=head3 IRONMQ_LIST_QUEUES

/projects/{Project ID}/queues

=head3 IRONMQ_ADD_SUBSCRIBERS_TO_A_MESSAGE_QUEUE

/projects/{Project ID}/queues/{Queue Name}/subscribers

=head3 IRONMQ_DELETE_SUBSCRIBERS_FROM_A_MESSAGE_QUEUE

/projects/{Project ID}/queues/{Queue Name}/subscribers

=head2 Define queue alerts

=head3 IRONMQ_ADD_ALERTS_TO_A_QUEUE

/projects/{Project ID}/queues/{Queue Name}/alerts

=head3 IRONMQ_REPLACE_ALERTS_ON_A_QUEUE

/projects/{Project ID}/queues/{Queue Name}/alerts

=head3 IRONMQ_REMOVE_ALERTS_FROM_A_QUEUE

/projects/{Project ID}/queues/{Queue Name}/alerts

=head3 IRONMQ_REMOVE_ALERTS_FROM_A_QUEUE_BY_ID

/projects/{Project ID}/queues/{Queue Name}/alerts/{Alert ID}

=head2 Operate messages

=head3 IRONMQ_POST_MESSAGES

/projects/{Project ID}/queues/{Queue Name}/messages

=head3 IRONMQ_RESERVE_MESSAGES

/projects/{Project ID}/queues/{Queue Name}/reservations

=head3 IRONMQ_GET_MESSAGE_BY_ID

/projects/{Project ID}/queues/{Queue Name}/messages/{Message ID}

=head3 IRONMQ_PEEK_MESSAGES

/projects/{Project ID}/queues/{Queue Name}/messages/peek

=head3 IRONMQ_DELETE_MESSAGE

/projects/{Project ID}/queues/{Queue Name}/messages/{Message ID}

=head3 IRONMQ_DELETE_MESSAGES

/projects/{Project ID}/queues/{Queue Name}/messages

=head3 IRONMQ_TOUCH_MESSAGE

/projects/{Project ID}/queues/{Queue Name}/messages/{Message ID}/touch

=head3 IRONMQ_RELEASE_MESSAGE

/projects/{Project ID}/queues/{Queue Name}/messages/{Message ID}/release

=head3 IRONMQ_CLEAR_MESSAGES

/projects/{Project ID}/queues/{Queue Name}/clear

=head3 IRONMQ_GET_PUSH_STATUSES_FOR_A_MESSAGE

/projects/{Project ID}/queues/{Queue Name}/messages/{Message ID}/subscribers

=head1 AUTHOR

Mikko Koivunalho <mikko.koivunalho@iki.fi>

=head1 BUGS

Please report any bugs or feature requests to bug-io-iron@rt.cpan.org or through the web interface at:
 http://rt.cpan.org/Public/Dist/Display.html?Name=IO-Iron

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Mikko Koivunalho.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

The full text of the license can be found in the
F<LICENSE> file included with this distribution.

=cut
