# NAME

`Getopt::Auto::Long::Usage` - generate usage strings from Getopt::Long specs

# VERSION

Version 0.01

# SYNOPSIS

This is a pure perl module that generates simple usage / help messages by parsing [Getopt::Long](https://metacpan.org/pod/Getopt%3A%3ALong) argument specs (and optionally using provided descriptions).

      print getoptlong2usage( Getopt_Long => \@conf [, ...] )

# DESCRIPTION

`Getopt::Auto::Long::Usage` strives to be compatible with [Getopt::LongUsage](https://metacpan.org/pod/Getopt%3A%3ALongUsage). In particular, it does not require supplementing existing arglist specs with additional data (e.g. descriptions are optional). However, the goal is to provide maximum maintainability with the least amount of code, not to achieve complete [Getopt::Long](https://metacpan.org/pod/Getopt%3A%3ALong) coverage. So, there are some differences:

- the generated usage clearly distinguishes boolean flags from arguments requiring an option, and prints type information for the latter. For negatable boolean options (`longopt|s!`), it will print the corresponding `--no-longopt` flag (but not `--no-s`).
- there are no dependencies; the main function can be copied directly into your source code, if necessary
- it does not attempt to parse `GetOptions()` abbreviated / case-insensitive options, and in fact recommends that you disable those when using `Getopt::Long` for maintainability and predictability. One shortopt + one (or several) longopts, explicitly specified, will avoid nasty surprises (plus, suppose you decide to rewrite the code in some other language...)

The following example should print the generated help message either to stdout, if requested (`--help`) or to stderr, if argument parsing fails.

    use Getopt::Auto::Long::Usage;
    use Getopt::Long;
    my @getoptargs = qw{ help
                         delim:s
                         eval|e!
                       };
    my %_O_; my @getoptconf = (\%_O_, @getoptargs);

    sub usage {
      my ($rc) = @_;
      my @dsc = ( delim => 'instead of newline' );
      print getoptlong2usage(
        Getopt_Long => \@getoptconf, # all others optional
        cli_use => "Arguments: [OPTION]...\nOptions:";
        footer => "No other arguments may be supplied"
        descriptions => \@dsc
      );
      exit $rc if defined( $rc );
    }

    Getopt::Long::Configure( qw(
      no_ignore_case no_auto_abbrev no_getopt_compat
      gnu_compat bundling
    ));
    unless( GetOptions( @getoptconf ) ) {
      local *STDOUT = *STDERR; usage 1;
    }
    usage 0 if( $_O_{ help } );

# EXPORT

- `getoptlong2usage`

# FUNCTIONS

## getoptlong2usage

    $usage = getoptlong2usage( Getopt_Long => \@getoptconf [,
      descriptions => \@dsc,
      cli_use => $cli_use,
      indent => $n,
      pfx => $prefix ] )

`@getoptconf` is an arrayref containing all the arguments you would supply to `GetOptions()`, including the initial hashref in which `GetOptions()` stores results (and which is ignored). It's easiest to define `@getoptconf` separately and reuse it for both calls. See the synopsis for an example.

# SOURCE

The source code repository can be found at [https://gitlab.com/kstr0k/perl-getopt-auto-long-usage.git](https://gitlab.com/kstr0k/perl-getopt-auto-long-usage.git).

# SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Getopt::Auto::Long::Usage

# AUTHOR

Alin Mr., `<almr.oss at outlook.com>`

# LICENSE AND COPYRIGHT

This software is Copyright (c) 2021 by Alin Mr.

This is free software, licensed under:

    The MIT (X11) License
