#!perl

use strict;
use warnings;
use File::Temp;
use File::Slurp qw(write_file);
use Test::More tests => 8;

BEGIN {
    use_ok( 'Parse::AccessLog' ) || print "Bail out!\n";
}

my $p = new_ok('Parse::AccessLog');
my $data = do { local $/; <DATA> };

my $fh = File::Temp->new;
my $fname = $fh->filename;
if ( ! write_file($fname, $data) ) {
    croak("Can't write file [$fname]: $!");
}

my @recs = $p->parse($fname);
is(@recs, 665, 'Parsed 665 entries (lines) from file' );

my %temp;
my @uniq_ips = grep { !$temp{$_->{remote_addr}}++ } @recs;
is(@uniq_ips, 128, 'Got 128 unique IPs');

# get all entries with a referer string
my @referers = grep { $_->{referer} ne '-' } @recs;
is( @referers, 111, 'Got 111 lines with a referer string' );

my $most_hits = (sort { $temp{$b} <=> $temp{$a} } keys %temp )[0]; 
is($most_hits, '216.96.86.26', 'Most frequent IP = 216.96.86.26');
is($temp{$most_hits}, 261, 'Most frequent IP has 261 entries.');

# possibly ipv6 (use a valid regex to make sure, not within the scope of this
# module)
my @ipv6 = grep { (length($_->{remote_addr}) > 15) ||
                  (length($_->{remote_addr}) < 4)
           } @recs;
is(@ipv6, 19, 'Got 19 IPv6 records.');
#say $_->{remote_addr} for @ipv6;


__END__
::1 - - [21/Jan/2013:17:55:52 -0600] "POST / HTTP/1.1" 405 173 "-" "PHP-SOAP/5.4.10"
216.96.86.26 - - [17/Jan/2013:13:20:44 -0600] "HEAD / HTTP/1.1" 302 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:13:22:05 -0600] "HEAD / HTTP/1.0" 302 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:13:22:16 -0600] "HEAD / HTTP/1.0" 302 0 "-" "Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.8.1.18) Gecko/20081112 Fedora/2.0.0.18-1.fc8 Firefox/2.0.0.18"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:13:22:55 -0600] "POST /wp-cron.php?doing_wp_cron=1358450575.7623460292816162109375 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
216.96.86.26 - - [17/Jan/2013:13:22:55 -0600] "GET /contact-us/ HTTP/1.1" 200 6289 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:13:22:56 -0600] "GET /wp-content/plugins/contact-form-7/includes/css/styles.css?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:13:22:56 -0600] "GET /wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:13:22:56 -0600] "GET /wp-includes/js/jquery/jquery.js?ver=1.8.3 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:13:22:56 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/jquery.form.min.js?ver=3.23 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:13:22:56 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/scripts.js?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:13:23:01 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:13:23:02 -0600] "-" 400 0 "-" "-"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:13:24:59 -0600] "POST /wp-cron.php?doing_wp_cron=1358450699.2754919528961181640625 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
216.96.86.26 - - [17/Jan/2013:13:24:59 -0600] "GET /contact-us/ HTTP/1.1" 200 6289 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0 FirePHP/0.7.1"
216.96.86.26 - - [17/Jan/2013:13:24:59 -0600] "GET /wp-content/plugins/contact-form-7/includes/css/styles.css?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0 FirePHP/0.7.1"
216.96.86.26 - - [17/Jan/2013:13:24:59 -0600] "GET /wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0 FirePHP/0.7.1"
216.96.86.26 - - [17/Jan/2013:13:24:59 -0600] "GET /wp-includes/js/jquery/jquery.js?ver=1.8.3 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0 FirePHP/0.7.1"
216.96.86.26 - - [17/Jan/2013:13:24:59 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/jquery.form.min.js?ver=3.23 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0 FirePHP/0.7.1"
216.96.86.26 - - [17/Jan/2013:13:24:59 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/scripts.js?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0 FirePHP/0.7.1"
216.96.86.26 - - [17/Jan/2013:13:25:05 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:13:25:05 -0600] "-" 400 0 "-" "-"
178.32.77.81 - - [17/Jan/2013:13:28:10 -0600] "GET / HTTP/1.0" 302 0 "-" "Mozilla/5.0 (Windows NT 5.1; U; en-US; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 Opera 10.70"
208.115.113.86 - - [17/Jan/2013:13:29:50 -0600] "GET /blog/disable-speed-dial-feature-in-firefox-13/ HTTP/1.1" 200 11568 "-" "Mozilla/5.0 (compatible; Ezooms/1.0; ezooms.bot@gmail.com)"
208.115.113.86 - - [17/Jan/2013:13:29:54 -0600] "GET /blog/life-update-castles-belgium/ HTTP/1.1" 200 9964 "-" "Mozilla/5.0 (compatible; Ezooms/1.0; ezooms.bot@gmail.com)"
216.96.86.26 - - [17/Jan/2013:13:30:07 -0600] "HEAD / HTTP/1.0" 302 0 "-" "Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.8.1.18) Gecko/20081112 Fedora/2.0.0.18-1.fc8 Firefox/2.0.0.18"
216.96.86.26 - - [17/Jan/2013:13:30:12 -0600] "HEAD /blog/ HTTP/1.0" 200 0 "-" "Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.8.1.18) Gecko/20081112 Fedora/2.0.0.18-1.fc8 Firefox/2.0.0.18"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:13:39:32 -0600] "POST /wp-cron.php?doing_wp_cron=1358451572.0317718982696533203125 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
123.125.71.108 - - [17/Jan/2013:13:39:32 -0600] "GET / HTTP/1.1" 200 13346 "-" "Mozilla/5.0 (compatible; Baiduspider/2.0; +http://www.baidu.com/search/spider.html)"
220.181.108.183 - - [17/Jan/2013:13:39:42 -0600] "GET / HTTP/1.1" 200 13346 "-" "Mozilla/5.0 (compatible; Baiduspider/2.0; +http://www.baidu.com/search/spider.html)"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:13:45:02 -0600] "POST /wp-cron.php?doing_wp_cron=1358451902.1631639003753662109375 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
64.60.192.82 - - [17/Jan/2013:13:45:02 -0600] "GET /about/ HTTP/1.1" 200 5094 "http://search.yahoo.com/search;_ylt=A0oGdV1nVPhQjC8AojZXNyoA?p=JASMIN+DRAGON+TEA&fr2=sb-top&fr=yfp-t-701-14&type_param=" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.1; WOW64; Trident/5.0; SLCC2; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729; Media Center PC 6.0; .NET4.0C; .NET4.0E; BRI/2)"
64.60.192.82 - - [17/Jan/2013:13:45:02 -0600] "GET /wp-content/themes/thesis_182/style.css HTTP/1.1" 200 15003 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:02 -0600] "GET /wp-content/themes/thesis_182/custom/layout.css HTTP/1.1" 200 17434 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:02 -0600] "GET /wp-content/themes/thesis_182/custom/custom.css HTTP/1.1" 200 132 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:02 -0600] "GET /wp-content/plugins/contact-form-7/includes/css/styles.css?ver=3.3.2 HTTP/1.1" 200 887 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:02 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/jquery.form.min.js?ver=3.23 HTTP/1.1" 200 15159 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:02 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/scripts.js?ver=3.3.2 HTTP/1.1" 200 6859 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:02 -0600] "GET /wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 200 786 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:03 -0600] "GET /wp-includes/js/jquery/jquery.js?ver=1.8.3 HTTP/1.1" 200 93658 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:03 -0600] "GET /wp-content/uploads/2011/12/jdt51.png HTTP/1.1" 200 28188 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:03 -0600] "GET /wp-content/themes/thesis_182/lib/images/dot-ddd.gif HTTP/1.1" 200 49 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:03 -0600] "GET /wp-content/themes/thesis_182/lib/images/icon-rss.gif HTTP/1.1" 200 967 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:04 -0600] "GET /wp-content/themes/thesis_182/custom/rotator/54810053_4787fa7f3a.jpg HTTP/1.1" 200 141127 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:05 -0600] "GET /favicon.ico HTTP/1.1" 200 0 "-" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:26 -0600] "GET / HTTP/1.1" 200 13346 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:26 -0600] "GET /wp-includes/images/smilies/icon_wink.gif HTTP/1.1" 200 170 "http://localhost/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:27 -0600] "GET /wp-content/themes/thesis_182/custom/rotator/54676138_2a43bbdf2a.jpg HTTP/1.1" 200 177854 "http://localhost/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:28 -0600] "GET /favicon.ico HTTP/1.1" 200 0 "-" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
61.135.249.98 - - [17/Jan/2013:13:45:41 -0600] "GET / HTTP/1.1" 302 5 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.249.98 - - [17/Jan/2013:13:45:41 -0600] "GET /robots.txt HTTP/1.1" 200 15 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.249.98 - - [17/Jan/2013:13:45:42 -0600] "GET /blog/ HTTP/1.1" 200 40332 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
64.60.192.82 - - [17/Jan/2013:13:45:47 -0600] "GET /contact-us/ HTTP/1.1" 200 6289 "http://localhost/about/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
64.60.192.82 - - [17/Jan/2013:13:45:48 -0600] "GET /wp-content/plugins/contact-form-7/images/ajax-loader.gif HTTP/1.1" 200 847 "http://localhost/contact-us/" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0)"
173.199.119.59 - - [17/Jan/2013:13:50:21 -0600] "GET /blog/tag/terminal/ HTTP/1.1" 200 8216 "-" "Mozilla/5.0 (compatible; AhrefsBot/4.0; +http://ahrefs.com/robot/)"
216.96.86.26 - - [17/Jan/2013:14:07:20 -0600] "GET / HTTP/1.1" 302 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:21 -0600] "GET /blog/ HTTP/1.1" 200 12993 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:22 -0600] "GET /blog/wp-content/themes/koi-improved/yellow.css HTTP/1.1" 404 13019 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:26 -0600] "GET /blog/about/ HTTP/1.1" 200 2573 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:26 -0600] "GET /blog/wp-content/plugins/codecolorer/codecolorer.css?ver=0.9.9 HTTP/1.1" 304 0 "http://mysite.com/blog/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:26 -0600] "GET /blog/wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 200 786 "http://mysite.com/blog/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:26 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:07:27 -0600] "GET /blog/wp-content/themes/koi-improved/yellow.css HTTP/1.1" 404 13019 "http://mysite.com/blog/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:32 -0600] "GET /blog/life-list/ HTTP/1.1" 200 2120 "http://mysite.com/blog/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:32 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:07:32 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:07:32 -0600] "GET /blog/wp-content/plugins/codecolorer/codecolorer.css?ver=0.9.9 HTTP/1.1" 304 0 "http://mysite.com/blog/life-list/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:32 -0600] "GET /blog/wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 304 0 "http://mysite.com/blog/life-list/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:33 -0600] "GET /blog/wp-content/themes/koi-improved/yellow.css HTTP/1.1" 404 13019 "http://mysite.com/blog/life-list/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:34 -0600] "GET /blog/about/ HTTP/1.1" 200 2573 "http://mysite.com/blog/life-list/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:37 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:07:40 -0600] "GET /blog/about/ HTTP/1.1" 200 2573 "http://mysite.com/blog/life-list/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:40 -0600] "GET /blog/wp-content/plugins/codecolorer/codecolorer.css?ver=0.9.9 HTTP/1.1" 304 0 "http://mysite.com/blog/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:40 -0600] "GET /blog/wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 304 0 "http://mysite.com/blog/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:41 -0600] "GET /blog/wp-content/themes/koi-improved/yellow.css HTTP/1.1" 404 13019 "http://mysite.com/blog/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:07:46 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:07:47 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:07:47 -0600] "GET /blog/feed/ HTTP/1.1" 200 38223 "http://mysite.com/blog/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:14:08:06 -0600] "POST /wp-cron.php?doing_wp_cron=1358453286.8141279220581054687500 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
216.96.86.26 - - [17/Jan/2013:14:08:06 -0600] "GET /contact-us/ HTTP/1.1" 200 2178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:07 -0600] "GET /wp-content/plugins/contact-form-7/includes/css/styles.css?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:07 -0600] "GET /wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:07 -0600] "GET /wp-includes/js/jquery/jquery.js?ver=1.8.3 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:08 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/jquery.form.min.js?ver=3.23 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:08 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/scripts.js?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:09 -0600] "GET /about/ HTTP/1.1" 200 1845 "http://localhost/contact-us/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:09 -0600] "GET /wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:09 -0600] "GET /wp-content/plugins/contact-form-7/includes/css/styles.css?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:09 -0600] "GET /wp-includes/js/jquery/jquery.js?ver=1.8.3 HTTP/1.1" 304 0 "http://localhost/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:09 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/jquery.form.min.js?ver=3.23 HTTP/1.1" 304 0 "http://localhost/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:10 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/scripts.js?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:13 -0600] "GET /feed/ HTTP/1.1" 200 11319 "http://localhost/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:15 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:08:15 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:08:17 -0600] "GET / HTTP/1.1" 200 4926 "http://localhost/about/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:17 -0600] "GET /wp-includes/js/jquery/jquery.js?ver=1.8.3 HTTP/1.1" 304 0 "http://localhost/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:17 -0600] "GET /wp-content/plugins/contact-form-7/includes/css/styles.css?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:17 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/jquery.form.min.js?ver=3.23 HTTP/1.1" 304 0 "http://localhost/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:17 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/scripts.js?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:21 -0600] "GET /camellia-sinensis-the-tea-plant/ HTTP/1.1" 200 2666 "http://localhost/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:21 -0600] "GET /wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/camellia-sinensis-the-tea-plant/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:21 -0600] "GET /wp-content/plugins/contact-form-7/includes/css/styles.css?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/camellia-sinensis-the-tea-plant/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:21 -0600] "GET /wp-includes/js/jquery/jquery.js?ver=1.8.3 HTTP/1.1" 304 0 "http://localhost/camellia-sinensis-the-tea-plant/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:21 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/jquery.form.min.js?ver=3.23 HTTP/1.1" 304 0 "http://localhost/camellia-sinensis-the-tea-plant/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:22 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/scripts.js?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/camellia-sinensis-the-tea-plant/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:27 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:08:27 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:08:39 -0600] "GET /files HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:40 -0600] "GET /files/ HTTP/1.1" 403 135 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:46 -0600] "GET /xfiles/ HTTP/1.1" 404 136 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:50 -0600] "GET / HTTP/1.1" 302 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:51 -0600] "GET /blog/ HTTP/1.1" 200 12993 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:51 -0600] "GET /blog/wp-content/plugins/codecolorer/codecolorer.css?ver=0.9.9 HTTP/1.1" 304 0 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:52 -0600] "GET /blog/wp-content/themes/koi-improved/yellow.css HTTP/1.1" 404 13019 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:08:56 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:09:07 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:09:09 -0600] "GET / HTTP/1.1" 200 31 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:09:09 -0600] "GET /favicon.ico HTTP/1.1" 404 136 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:09:10 -0600] "GET / HTTP/1.1" 302 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:09:11 -0600] "GET /blog/ HTTP/1.1" 200 12993 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:09:11 -0600] "GET /blog/wp-content/plugins/codecolorer/codecolorer.css?ver=0.9.9 HTTP/1.1" 304 0 "http://blackcarrot.be/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:09:12 -0600] "GET /blog/wp-content/themes/koi-improved/yellow.css HTTP/1.1" 404 13019 "http://blackcarrot.be/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:09:13 -0600] "GET /favicon.ico HTTP/1.1" 404 136 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:09:17 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:09:17 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:12:54 -0600] "GET / HTTP/1.1" 302 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:12:55 -0600] "GET /blog/ HTTP/1.1" 200 12993 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:12:55 -0600] "GET /blog/wp-content/plugins/codecolorer/codecolorer.css?ver=0.9.9 HTTP/1.1" 304 0 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:12:56 -0600] "GET /blog/wp-content/themes/koi-improved/yellow.css HTTP/1.1" 404 13019 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:13:56 -0600] "GET / HTTP/1.1" 302 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:13:57 -0600] "GET /blog/ HTTP/1.1" 200 12993 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:13:57 -0600] "GET /blog/wp-content/plugins/codecolorer/codecolorer.css?ver=0.9.9 HTTP/1.1" 304 0 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:13:58 -0600] "GET /blog/wp-content/themes/koi-improved/yellow.css HTTP/1.1" 404 13019 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:13:58 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:13:58 -0600] "GET / HTTP/1.1" 302 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:13:59 -0600] "GET /blog/ HTTP/1.1" 200 12993 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:14:00 -0600] "GET /blog/wp-content/plugins/codecolorer/codecolorer.css?ver=0.9.9 HTTP/1.1" 304 0 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:14:00 -0600] "GET /blog/wp-content/themes/koi-improved/yellow.css HTTP/1.1" 404 13019 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:14:03 -0600] "GET /blog/ HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:14:04 -0600] "GET /blog/ HTTP/1.1" 200 12993 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:14:04 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:14:04 -0600] "GET /blog/wp-content/plugins/codecolorer/codecolorer.css?ver=0.9.9 HTTP/1.1" 304 0 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:14:05 -0600] "GET /blog/wp-content/themes/koi-improved/yellow.css HTTP/1.1" 404 13019 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:14:07 -0600] "GET / HTTP/1.1" 302 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:14:08 -0600] "GET /blog/ HTTP/1.1" 200 12993 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:14:08 -0600] "GET /blog/wp-content/plugins/codecolorer/codecolorer.css?ver=0.9.9 HTTP/1.1" 304 0 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:14:09 -0600] "GET /blog/wp-content/themes/koi-improved/yellow.css HTTP/1.1" 404 13019 "http://mysite.com/blog/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:14:27 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:14:14:32 -0600] "HEAD /blog/ HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
178.154.163.29 - - [17/Jan/2013:14:17:10 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots)"
70.75.177.99 - - [17/Jan/2013:14:24:44 -0600] "GET /scripts/twitter-who2follow.user.js HTTP/1.1" 404 136 "-" "Mozilla/5.0 (Windows NT 6.1; WOW64; rv:18.0) Gecko/20100101 Firefox/18.0"
70.75.177.99 - - [17/Jan/2013:14:24:49 -0600] "-" 400 0 "-" "-"
70.75.177.99 - - [17/Jan/2013:14:24:49 -0600] "-" 400 0 "-" "-"
70.75.177.99 - - [17/Jan/2013:14:24:49 -0600] "-" 400 0 "-" "-"
180.76.5.51 - - [17/Jan/2013:14:30:56 -0600] "GET /blog/category/food/ HTTP/1.1" 200 6258 "-" "Mozilla/5.0 (compatible; Baiduspider/2.0; +http://www.baidu.com/search/spider.html)"
178.255.215.67 - - [17/Jan/2013:14:33:56 -0600] "GET /robots.txt HTTP/1.1" 200 15 "-" "Mozilla/5.0 (compatible; Exabot/3.0; +http://www.exabot.com/go/robot)"
178.255.215.67 - - [17/Jan/2013:14:33:57 -0600] "GET /blog/costa-rica/ HTTP/1.1" 200 3023 "-" "Mozilla/5.0 (compatible; Exabot/3.0; +http://www.exabot.com/go/robot)"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:14:39:12 -0600] "POST /wp-cron.php?doing_wp_cron=1358455152.6517388820648193359375 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
123.125.71.101 - - [17/Jan/2013:14:39:12 -0600] "GET / HTTP/1.1" 200 13346 "-" "Mozilla/5.0 (compatible; Baiduspider/2.0; +http://www.baidu.com/search/spider.html)"
220.181.108.161 - - [17/Jan/2013:14:39:54 -0600] "GET / HTTP/1.1" 200 13346 "-" "Mozilla/5.0 (compatible; Baiduspider/2.0; +http://www.baidu.com/search/spider.html)"
119.63.193.195 - - [17/Jan/2013:14:40:46 -0600] "GET / HTTP/1.1" 302 5 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
216.96.86.26 - - [17/Jan/2013:14:48:44 -0600] "GET /blog/feed/ HTTP/1.1" 304 0 "-" "Vienna 3.0b.2821 rv:2821 (Macintosh; Mac OS X 10.7.4; en_US)"
173.199.119.59 - - [17/Jan/2013:14:49:25 -0600] "GET /blog/almost-christmas/ HTTP/1.1" 200 5828 "-" "Mozilla/5.0 (compatible; AhrefsBot/4.0; +http://ahrefs.com/robot/)"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:14:50:03 -0600] "POST /wp-cron.php?doing_wp_cron=1358455803.3613009452819824218750 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
216.96.86.26 - - [17/Jan/2013:14:50:03 -0600] "GET / HTTP/1.1" 200 4926 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:50:03 -0600] "GET /wp-content/plugins/contact-form-7/includes/css/styles.css?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:50:03 -0600] "GET /wp-includes/js/jquery/jquery.js?ver=1.8.3 HTTP/1.1" 304 0 "http://localhost/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:50:04 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/jquery.form.min.js?ver=3.23 HTTP/1.1" 304 0 "http://localhost/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:50:04 -0600] "GET /wp-content/plugins/contact-form-7/includes/js/scripts.js?ver=3.3.2 HTTP/1.1" 304 0 "http://localhost/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:14:50:10 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:14:50:14 -0600] "HEAD / HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:14:50:21 -0600] "HEAD /about HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:14:50:23 -0600] "HEAD /about/ HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:14:50:26 -0600] "GET /about/ HTTP/1.1" 200 5094 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
157.55.33.50 - - [17/Jan/2013:14:52:14 -0600] "GET /robots.txt HTTP/1.1" 200 15 "-" "Mozilla/5.0 (compatible; bingbot/2.0; +http://www.bing.com/bingbot.htm)"
157.55.33.50 - - [17/Jan/2013:14:54:15 -0600] "GET / HTTP/1.1" 302 5 "-" "Mozilla/5.0 (compatible; bingbot/2.0; +http://www.bing.com/bingbot.htm)"
157.55.33.50 - - [17/Jan/2013:14:54:18 -0600] "GET /blog/ HTTP/1.1" 200 12993 "-" "Mozilla/5.0 (compatible; bingbot/2.0; +http://www.bing.com/bingbot.htm)"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:14:54:34 -0600] "POST /wp-cron.php?doing_wp_cron=1358456074.4810259342193603515625 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
178.154.163.29 - - [17/Jan/2013:14:54:34 -0600] "GET /robots.txt HTTP/1.1" 200 123 "-" "Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots)"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:14:58:41 -0600] "POST /wp-cron.php?doing_wp_cron=1358456321.1747479438781738281250 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
157.55.32.87 - - [17/Jan/2013:14:58:41 -0600] "GET /robots.txt HTTP/1.1" 200 123 "-" "Mozilla/5.0 (compatible; bingbot/2.0; +http://www.bing.com/bingbot.htm)"
216.96.86.26 - - [17/Jan/2013:14:59:53 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:15:00:00 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:15:00:03 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:15:00:06 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:15:00:10 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:15:02:06 -0600] "HEAD /files HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:15:02:08 -0600] "HEAD /files/ HTTP/1.1" 403 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:15:02:32 -0600] "HEAD /blog/ HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:15:05:21 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:05:22 -0600] "HEAD /blog/ HTTP/1.1" 200 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:06:20 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
66.249.74.38 - - [17/Jan/2013:15:06:48 -0600] "GET /blog/page/2/ HTTP/1.1" 200 18157 "-" "Mozilla/5.0 (iPhone; U; CPU iPhone OS 4_1 like Mac OS X; en-us) AppleWebKit/532.9 (KHTML, like Gecko) Version/4.0.5 Mobile/8B117 Safari/6531.22.7 (compatible; Googlebot-Mobile/2.1; +http://www.google.com/bot.html)"
216.96.86.26 - - [17/Jan/2013:15:10:26 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:10:27 -0600] "HEAD /files HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:10:28 -0600] "HEAD /files/ HTTP/1.1" 403 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:10:29 -0600] "HEAD /blog/ HTTP/1.1" 200 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:10:30 -0600] "HEAD / HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:10:31 -0600] "HEAD /  HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:10:32 -0600] "HEAD /  HTTP/1.1" 200 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:10:33 -0600] "HEAD /  HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:15:10:33 -0600] "POST /wp-cron.php?doing_wp_cron=1358457033.8215329647064208984375 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
216.96.86.26 - - [17/Jan/2013:15:10:34 -0600] "HEAD / HTTP/1.1" 200 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:10:35 -0600] "HEAD /about/ HTTP/1.1" 200 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:10:36 -0600] "HEAD / HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
173.199.119.59 - - [17/Jan/2013:15:13:19 -0600] "GET /blog/remove-who-to-follow/ HTTP/1.1" 200 3007 "-" "Mozilla/5.0 (compatible; AhrefsBot/4.0; +http://ahrefs.com/robot/)"
216.96.86.26 - - [17/Jan/2013:15:14:09 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:15:07 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:15:13 -0600] "HEAD / HTTP/1.1" 499 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:18:11 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:18:12 -0600] "HEAD /files HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:18:13 -0600] "HEAD /files/ HTTP/1.1" 403 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:18:14 -0600] "HEAD /blog/ HTTP/1.1" 200 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:18:15 -0600] "HEAD / HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:18:15 -0600] "HEAD /  HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:18:16 -0600] "HEAD /  HTTP/1.1" 200 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:18:17 -0600] "HEAD /  HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:15:18:18 -0600] "POST /wp-cron.php?doing_wp_cron=1358457498.6058139801025390625000 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
216.96.86.26 - - [17/Jan/2013:15:18:18 -0600] "HEAD / HTTP/1.1" 200 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:18:19 -0600] "HEAD /about/ HTTP/1.1" 200 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:18:20 -0600] "HEAD / HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:18:23 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
65.55.212.197 - - [17/Jan/2013:15:18:30 -0600] "GET /robots.txt HTTP/1.1" 200 15 "-" "msnbot-media/1.1 (+http://search.msn.com/msnbot.htm)"
65.55.212.197 - - [17/Jan/2013:15:18:31 -0600] "GET /blog/wp-content/uploads/2010/11/chateau_frontenac-s.jpg HTTP/1.1" 200 19999 "-" "msnbot-media/1.1 (+http://search.msn.com/msnbot.htm)"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:15:22:08 -0600] "POST /wp-cron.php?doing_wp_cron=1358457728.7418529987335205078125 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
173.45.73.18 - - [17/Jan/2013:15:22:08 -0600] "GET / HTTP/1.1" 200 4926 "-" "Mozilla/5.0 (Windows NT 6.1; rv:16.0) Gecko/20100101 Firefox/16.0"
216.96.86.26 - - [17/Jan/2013:15:23:22 -0600] "HEAD / HTTP/1.1" 302 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:15:25:24 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:26:18 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:26:39 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:28:31 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:28:40 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:28:44 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:31:34 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
178.154.163.29 - - [17/Jan/2013:15:31:58 -0600] "GET /blog/tag/tokyo/ HTTP/1.1" 200 3258 "-" "Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots)"
216.96.86.26 - - [17/Jan/2013:15:32:07 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:32:41 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:32:53 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:33:23 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
173.199.119.59 - - [17/Jan/2013:15:33:29 -0600] "GET /blog/black-carrots/ HTTP/1.1" 200 3140 "-" "Mozilla/5.0 (compatible; AhrefsBot/4.0; +http://ahrefs.com/robot/)"
216.96.86.26 - - [17/Jan/2013:15:34:19 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:34:22 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:34:25 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:34:30 -0600] "HEAD / HTTP/1.1" 302 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
173.199.119.59 - - [17/Jan/2013:15:35:29 -0600] "GET /blog/tag/superfluous-whitespace/ HTTP/1.1" 200 4524 "-" "Mozilla/5.0 (compatible; AhrefsBot/4.0; +http://ahrefs.com/robot/)"
31.202.231.5 - - [17/Jan/2013:15:36:22 -0600] "POST /wp-login.php HTTP/1.0" 404 564 "http:///wp-login.php" "Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1; EmbeddedWB 14.52 from: http://www.bsalsa.com/ EmbeddedWB 14,52; .NET CLR 1.1.4322; .NET CLR 2.0.50727; .NET CLR 3.0.04506.648; .NET CLR 3.5.21022)"
216.96.86.26 - - [17/Jan/2013:15:36:27 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:36:28 -0600] "HEAD /files HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:36:29 -0600] "HEAD /files/ HTTP/1.1" 403 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:36:30 -0600] "HEAD /blog/ HTTP/1.1" 200 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:36:31 -0600] "HEAD / HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:36:32 -0600] "HEAD /  HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:36:33 -0600] "HEAD /  HTTP/1.1" 200 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:36:34 -0600] "HEAD /  HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:15:36:35 -0600] "POST /wp-cron.php?doing_wp_cron=1358458595.4214270114898681640625 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
216.96.86.26 - - [17/Jan/2013:15:36:35 -0600] "HEAD / HTTP/1.1" 200 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:36:36 -0600] "HEAD /about/ HTTP/1.1" 200 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:36:37 -0600] "HEAD / HTTP/1.1" 301 0 "-" "HTTP-Tiny/0.022"
216.96.86.26 - - [17/Jan/2013:15:36:39 -0600] "HEAD / HTTP/1.1" 302 0 "-" "HTTP-Tiny/0.022"
123.125.71.114 - - [17/Jan/2013:15:39:41 -0600] "GET / HTTP/1.1" 302 5 "-" "Mozilla/5.0 (compatible; Baiduspider/2.0; +http://www.baidu.com/search/spider.html)"
220.181.108.168 - - [17/Jan/2013:15:39:46 -0600] "GET / HTTP/1.1" 302 5 "-" "Mozilla/5.0 (compatible; Baiduspider/2.0; +http://www.baidu.com/search/spider.html)"
163.125.171.81 - - [17/Jan/2013:15:53:11 -0600] "-" 400 0 "-" "-"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:15:53:32 -0600] "POST /wp-cron.php?doing_wp_cron=1358459612.2744119167327880859375 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
157.55.32.87 - - [17/Jan/2013:15:53:32 -0600] "GET / HTTP/1.1" 200 4927 "-" "Mozilla/5.0 (compatible; bingbot/2.0; +http://www.bing.com/bingbot.htm)"
61.135.249.188 - - [17/Jan/2013:15:57:38 -0600] "GET / HTTP/1.1" 302 5 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.249.188 - - [17/Jan/2013:15:57:38 -0600] "GET /robots.txt HTTP/1.1" 200 15 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.249.188 - - [17/Jan/2013:15:57:39 -0600] "GET /blog/ HTTP/1.1" 200 12993 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
173.199.119.59 - - [17/Jan/2013:16:00:35 -0600] "GET /blog/purple-carrot/ HTTP/1.1" 200 2983 "-" "Mozilla/5.0 (compatible; AhrefsBot/4.0; +http://ahrefs.com/robot/)"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:16:01:51 -0600] "POST /wp-cron.php?doing_wp_cron=1358460111.3835120201110839843750 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
119.247.227.238 - - [17/Jan/2013:16:01:59 -0600] "GET /mydci/entries/edit/504 HTTP/1.1" 404 564 "-" "Mozilla/4.0 (compatible; MSIE 5.5; Windows NT)"
178.154.163.29 - - [17/Jan/2013:16:09:21 -0600] "GET /blog/tag/belgium/ HTTP/1.1" 200 3544 "-" "Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots)"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:16:12:58 -0600] "POST /wp-cron.php?doing_wp_cron=1358460778.9857649803161621093750 HTTP/1.0" 404 162 "-" "WordPress/3.5; http://localhost"
65.55.213.66 - - [17/Jan/2013:16:17:03 -0600] "GET /robots.txt HTTP/1.1" 200 15 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:17:03 -0600] "GET /blog HTTP/1.1" 301 178 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:17:05 -0600] "GET /blog/ HTTP/1.1" 200 12993 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:17:17 -0600] "GET /blog/about/ HTTP/1.1" 200 2573 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:17:20 -0600] "GET /blog/category/tech/ HTTP/1.1" 200 29999 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:17:22 -0600] "GET /blog/category/adventuring/ HTTP/1.1" 200 8193 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:17:24 -0600] "GET /blog/tag/carrot-soup/ HTTP/1.1" 200 2758 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:17:27 -0600] "GET /blog/spicnspan/ HTTP/1.1" 200 5216 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:17:29 -0600] "GET /blog/page/2/ HTTP/1.1" 200 18157 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:17:32 -0600] "GET /blog/black-carrots/ HTTP/1.1" 200 3140 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:17:34 -0600] "GET /blog/category/tech/information-security/ HTTP/1.1" 200 3892 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:17:35 -0600] "GET /blog/tag/purple-potatoes/ HTTP/1.1" 200 2625 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:17:38 -0600] "GET /blog/tag/beet-soup/ HTTP/1.1" 200 2619 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:17:40 -0600] "GET /blog/rare-ish-foods/ HTTP/1.1" 200 2887 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:17:43 -0600] "GET /blog/tag/yellow-carrots/ HTTP/1.1" 200 2760 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:17:47 -0600] "GET /blog/purple-carrot/ HTTP/1.1" 200 2983 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:17:49 -0600] "GET /blog/tag/purple-carrots/ HTTP/1.1" 200 2759 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:17:51 -0600] "GET /blog/tag/terminal/ HTTP/1.1" 200 2671 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:17:53 -0600] "GET /blog/tag/osx-terminal/ HTTP/1.1" 200 2672 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:17:55 -0600] "GET /blog/tag/osx-lion/ HTTP/1.1" 200 2669 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:17:57 -0600] "GET /blog/tag/osx-command-line/ HTTP/1.1" 200 2677 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:18:00 -0600] "GET /blog/tag/osx-2/ HTTP/1.1" 200 6770 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:18:02 -0600] "GET /blog/category/tech/osx/ HTTP/1.1" 200 7984 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:18:04 -0600] "GET /blog/osx-change-root-shell/ HTTP/1.1" 200 3462 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:18:06 -0600] "GET /blog/tag/firefox-speed-dial/ HTTP/1.1" 200 2562 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:18:07 -0600] "GET /blog/tag/firefox-13-speed-dial/ HTTP/1.1" 200 2560 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:18:10 -0600] "GET /blog/tag/firefox-13/ HTTP/1.1" 200 2555 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:18:12 -0600] "GET /blog/tag/ff13/ HTTP/1.1" 200 2556 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:18:14 -0600] "GET /blog/tag/disable-firefox-speed-dial/ HTTP/1.1" 200 2564 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:18:16 -0600] "GET /blog/disable-speed-dial-feature-in-firefox-13/ HTTP/1.1" 200 3841 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:18:18 -0600] "GET /blog/sailing/ HTTP/1.1" 200 3747 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:18:20 -0600] "GET /blog/category/freedom/ HTTP/1.1" 200 21993 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:18:22 -0600] "GET /blog/thailand-september-2012/ HTTP/1.1" 200 4621 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:18:24 -0600] "GET /blog/category/travel/ HTTP/1.1" 200 16723 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:18:26 -0600] "GET /blog/category/horticulture/ HTTP/1.1" 200 6714 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:18:28 -0600] "GET /blog/category/food/ HTTP/1.1" 200 6258 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:18:30 -0600] "GET /blog/almost-christmas/ HTTP/1.1" 200 5828 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:18:31 -0600] "GET /blog/category/tech/linux/ HTTP/1.1" 200 5394 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:18:33 -0600] "GET /blog/tag/ubuntu-sshd/ HTTP/1.1" 200 2567 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:18:35 -0600] "GET /blog/remove-ubuntudebian-version-from-sshd-server-string/ HTTP/1.1" 200 3300 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.67 - - [17/Jan/2013:16:18:37 -0600] "GET /blog/feed/ HTTP/1.1" 200 38223 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
65.55.213.66 - - [17/Jan/2013:16:18:39 -0600] "GET /blog/life-list/ HTTP/1.1" 200 2120 "-" "msnbot/2.0b (+http://search.msn.com/msnbot.htm)"
216.96.86.26 - - [17/Jan/2013:16:27:18 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:16:34:00 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:16:34:00 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:16:34:01 -0600] "-" 400 0 "-" "-"
65.111.177.188 - - [17/Jan/2013:16:50:49 -0600] "-" 400 0 "-" "-"
65.111.177.188 - - [17/Jan/2013:16:50:49 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [17/Jan/2013:16:50:52 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [17/Jan/2013:16:50:53 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [17/Jan/2013:16:50:54 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
216.96.86.26 - - [17/Jan/2013:16:53:13 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:16:53:13 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:06:38 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:06:38 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:27:56 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:27:56 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:27:56 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:30:02 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:30:09 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:30:09 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:31:38 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:31:38 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:31:38 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:31:38 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:36:03 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:36:05 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:36:21 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:36:29 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:36:30 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:56:23 -0600] "GET /login HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:17:56:25 -0600] "GET /css/style.css HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:17:56:33 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:56:34 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:56:34 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:17:56:41 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:17:56:42 -0600] "GET /list HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:19:07:19 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:19:07:19 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:19:12:44 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:19:12:44 -0600] "-" 400 0 "-" "-"
119.63.193.131 - - [17/Jan/2013:19:17:04 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
106.187.97.211 - - [17/Jan/2013:21:19:43 -0600] "GET / HTTP/1.1" 301 178 "-" "curl/7.27.0"
106.187.97.211 - - [17/Jan/2013:21:19:57 -0600] "GET / HTTP/1.1" 301 178 "-" "curl/7.27.0"
106.187.97.211 - - [17/Jan/2013:21:20:02 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.27.0"
106.187.97.211 - - [17/Jan/2013:21:21:11 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.27.0"
216.96.86.26 - - [17/Jan/2013:21:43:48 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:21:44:38 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:21:46:33 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
61.135.248.210 - - [17/Jan/2013:21:47:08 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:08 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:08 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:08 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:08 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:09 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:09 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:09 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:09 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:09 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:09 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:09 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:09 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:09 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:09 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:10 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:10 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:10 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:10 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
61.135.248.210 - - [17/Jan/2013:21:47:10 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )"
178.154.163.29 - - [17/Jan/2013:21:58:48 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots)"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:22:05:14 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.27.0"
106.187.97.211 - - [17/Jan/2013:22:05:45 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.27.0"
106.187.97.211 - - [17/Jan/2013:22:07:25 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.27.0"
2400:8900::f03c:91ff:feae:3de2 - - [17/Jan/2013:22:08:50 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.27.0"
106.187.97.211 - - [17/Jan/2013:22:08:54 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.27.0"
106.187.97.211 - - [17/Jan/2013:22:08:59 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.27.0"
106.187.97.211 - - [17/Jan/2013:22:09:36 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.27.0"
216.96.86.26 - - [17/Jan/2013:22:09:53 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:22:14:08 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:22:16:59 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
66.249.74.63 - - [17/Jan/2013:22:19:05 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)"
66.249.74.63 - - [17/Jan/2013:22:19:05 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)"
216.96.86.26 - - [17/Jan/2013:22:26:35 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:22:26:40 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:22:26:47 -0600] "HEAD // HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:22:27:12 -0600] "HEAD /files/ HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:22:27:20 -0600] "HEAD /files/ HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:22:27:41 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:22:28:16 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:22:28:20 -0600] "HEAD / HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:22:30:48 -0600] "HEAD /homesite HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:22:43:11 -0600] "GET / HTTP/1.1" 301 178 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [17/Jan/2013:22:43:27 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:22:43:29 -0600] "GET /list HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:22:43:31 -0600] "GET /css/style.css HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:22:43:33 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [17/Jan/2013:22:44:11 -0600] "GET /list/34 HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [17/Jan/2013:22:50:56 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.6; rv:17.0) Gecko/20100101 Firefox/17.0"
202.46.50.74 - - [17/Jan/2013:22:55:02 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
119.63.193.195 - - [17/Jan/2013:22:55:38 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
77.67.63.234 - - [17/Jan/2013:23:13:58 -0600] "GET / HTTP/1.1" 301 178 "-" "agent"
178.154.163.29 - - [17/Jan/2013:23:16:58 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots)"
178.154.163.29 - - [18/Jan/2013:00:35:32 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots)"
216.96.86.26 - - [18/Jan/2013:00:42:26 -0600] "GET /login HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [18/Jan/2013:00:42:54 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
218.30.116.99 - - [18/Jan/2013:01:38:55 -0600] "GET /blog/ HTTP/1.1" 301 178 "http://mysite.com/" "Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 6.1; Trident/4.0; SLCC2; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729; Media Center PC 6.0; Tablet PC 2.0)"
5.135.182.148 - - [18/Jan/2013:02:00:36 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
5.39.86.195 - - [18/Jan/2013:02:00:36 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
217.23.131.8 - - [18/Jan/2013:02:37:55 -0600] "-" 400 0 "-" "-"
65.111.177.188 - - [18/Jan/2013:03:40:22 -0600] "-" 400 0 "-" "-"
65.111.177.188 - - [18/Jan/2013:03:40:22 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [18/Jan/2013:03:40:25 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [18/Jan/2013:03:40:27 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [18/Jan/2013:03:40:28 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
5.135.186.103 - - [18/Jan/2013:06:32:37 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
115.164.1.116 - - [18/Jan/2013:07:52:36 -0600] "-" 400 0 "-" "-"
86.124.164.72 - - [18/Jan/2013:08:23:54 -0600] "-" 400 0 "-" "-"
86.124.164.72 - - [18/Jan/2013:08:24:34 -0600] "-" 400 0 "-" "-"
86.124.164.72 - - [18/Jan/2013:08:24:34 -0600] "-" 400 0 "-" "-"
86.124.164.72 - - [18/Jan/2013:08:24:34 -0600] "-" 400 0 "-" "-"
86.124.164.72 - - [18/Jan/2013:08:24:34 -0600] "-" 400 0 "-" "-"
78.129.149.17 - - [18/Jan/2013:10:12:00 -0600] "GET http://www.intel.com/ HTTP/1.1" 301 178 "-" "Internet Explorer 4.01"
216.96.86.26 - - [18/Jan/2013:11:02:58 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [18/Jan/2013:11:03:09 -0600] "GET /wp-admin HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [18/Jan/2013:11:04:41 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [18/Jan/2013:11:04:41 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [18/Jan/2013:11:04:46 -0600] "GET /wp-admin HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
216.96.86.26 - - [18/Jan/2013:11:04:58 -0600] "HEAD /wp-admin HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
216.96.86.26 - - [18/Jan/2013:11:05:00 -0600] "HEAD /wp-admin/ HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
124.109.48.196 - - [18/Jan/2013:11:05:19 -0600] "-" 400 0 "-" "-"
124.109.48.196 - - [18/Jan/2013:11:05:19 -0600] "-" 400 0 "-" "-"
112.207.207.110 - - [18/Jan/2013:12:26:47 -0600] "-" 400 0 "-" "-"
112.207.207.110 - - [18/Jan/2013:12:26:51 -0600] "-" 400 0 "-" "-"
124.115.6.15 - - [18/Jan/2013:13:36:18 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0(compatible; Sosospider/2.0; +http://help.soso.com/webspider.htm)"
66.249.74.103 - - [18/Jan/2013:14:17:04 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)"
66.249.77.19 - - [18/Jan/2013:14:17:04 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)"
119.63.193.196 - - [18/Jan/2013:14:39:16 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
78.129.149.17 - - [18/Jan/2013:14:57:45 -0600] "GET http://37.28.156.211/sprawdza.php HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; Trident/5.0)"
216.96.86.26 - - [18/Jan/2013:15:14:15 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [18/Jan/2013:15:27:42 -0600] "-" 400 0 "-" "-"
65.111.177.188 - - [18/Jan/2013:15:28:47 -0600] "-" 400 0 "-" "-"
65.111.177.188 - - [18/Jan/2013:15:28:47 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [18/Jan/2013:15:28:49 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [18/Jan/2013:15:28:50 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [18/Jan/2013:15:28:52 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
216.96.86.26 - - [18/Jan/2013:15:31:38 -0600] "-" 400 0 "-" "-"
125.39.66.149 - - [18/Jan/2013:17:23:34 -0600] "POST /wp-login.php HTTP/1.1" 301 178 "-" "Mozilla/3.0 (compatible; Indy Library)"
219.157.200.18 - - [18/Jan/2013:17:23:53 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "-" "Mozilla/3.0 (compatible; Indy Library)"
122.72.80.101 - - [18/Jan/2013:17:24:03 -0600] "POST /wp-login.php HTTP/1.1" 301 178 "-" "Mozilla/3.0 (compatible; Indy Library)"
190.128.170.18 - - [18/Jan/2013:17:24:11 -0600] "-" 400 0 "-" "-"
190.128.170.18 - - [18/Jan/2013:17:24:13 -0600] "-" 400 0 "-" "-"
211.142.236.132 - - [18/Jan/2013:17:24:19 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "-" "Mozilla/3.0 (compatible; Indy Library)"
122.72.80.101 - - [18/Jan/2013:17:24:27 -0600] "POST /wp-login.php HTTP/1.1" 301 178 "-" "Mozilla/3.0 (compatible; Indy Library)"
175.25.243.27 - - [18/Jan/2013:17:24:34 -0600] "POST /wp-login.php HTTP/1.1" 301 178 "-" "Mozilla/3.0 (compatible; Indy Library)"
177.87.193.134 - - [18/Jan/2013:17:24:42 -0600] "POST /wp-login.php HTTP/1.1" 301 178 "-" "Mozilla/3.0 (compatible; Indy Library)"
122.72.80.101 - - [18/Jan/2013:17:24:51 -0600] "POST /wp-login.php HTTP/1.1" 301 178 "-" "Mozilla/3.0 (compatible; Indy Library)"
216.96.86.26 - - [18/Jan/2013:17:29:23 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [18/Jan/2013:17:29:24 -0600] "-" 400 0 "-" "-"
216.96.86.26 - - [18/Jan/2013:17:29:24 -0600] "-" 400 0 "-" "-"
86.144.193.62 - - [18/Jan/2013:17:42:44 -0600] "-" 400 0 "-" "-"
121.210.89.193 - - [18/Jan/2013:23:35:14 -0600] "-" 400 0 "-" "-"
94.23.14.147 - - [18/Jan/2013:23:35:33 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
5.135.186.47 - - [18/Jan/2013:23:35:33 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
94.23.4.106 - - [18/Jan/2013:23:35:33 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
91.121.26.97 - - [18/Jan/2013:23:35:34 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
91.121.198.168 - - [18/Jan/2013:23:35:34 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
91.121.204.98 - - [18/Jan/2013:23:35:34 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
94.23.62.47 - - [18/Jan/2013:23:35:34 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
5.135.182.150 - - [18/Jan/2013:23:35:34 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
5.135.186.48 - - [18/Jan/2013:23:35:34 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
94.23.250.149 - - [18/Jan/2013:23:35:37 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
202.46.56.175 - - [19/Jan/2013:00:39:18 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
119.63.193.195 - - [19/Jan/2013:00:39:23 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
202.46.56.124 - - [19/Jan/2013:04:41:26 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
119.63.193.132 - - [19/Jan/2013:04:41:34 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
65.111.177.188 - - [19/Jan/2013:06:02:37 -0600] "-" 400 0 "-" "-"
65.111.177.188 - - [19/Jan/2013:06:02:37 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [19/Jan/2013:06:02:39 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [19/Jan/2013:06:02:40 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [19/Jan/2013:06:02:41 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
66.249.77.133 - - [19/Jan/2013:06:28:13 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)"
66.249.77.133 - - [19/Jan/2013:06:28:14 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)"
91.121.9.21 - - [19/Jan/2013:08:05:53 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
5.135.185.89 - - [19/Jan/2013:08:05:53 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
91.121.156.62 - - [19/Jan/2013:08:05:54 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
5.39.86.162 - - [19/Jan/2013:08:26:10 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
5.135.182.147 - - [19/Jan/2013:08:26:11 -0600] "POST /wp-login.php HTTP/1.0" 301 178 "http://ninjas.com/wp-login.php" "Opera/9.80 (Windows NT 6.1; U; ru) Presto/2.8.131 Version/11.10"
180.76.5.167 - - [19/Jan/2013:08:40:26 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1; rv:6.0.2) Gecko/20100101 Firefox/6.0.2"
70.45.211.154 - - [19/Jan/2013:13:09:27 -0600] "-" 400 0 "-" "-"
70.45.211.154 - - [19/Jan/2013:13:09:27 -0600] "-" 400 0 "-" "-"
70.45.211.154 - - [19/Jan/2013:13:09:27 -0600] "-" 400 0 "-" "-"
89.242.100.55 - - [19/Jan/2013:14:14:31 -0600] "-" 400 0 "-" "-"
119.63.193.131 - - [19/Jan/2013:14:38:11 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
2.222.254.34 - - [19/Jan/2013:15:56:27 -0600] "-" 400 0 "-" "-"
2.222.254.34 - - [19/Jan/2013:16:00:21 -0600] "-" 400 0 "-" "-"
2.222.254.34 - - [19/Jan/2013:16:00:21 -0600] "-" 400 0 "-" "-"
2.222.254.34 - - [19/Jan/2013:16:00:21 -0600] "-" 400 0 "-" "-"
2.222.254.34 - - [19/Jan/2013:16:00:21 -0600] "-" 400 0 "-" "-"
176.61.139.128 - - [19/Jan/2013:16:34:07 -0600] "GET http://www.yahoo.com/ HTTP/1.1" 301 178 "-" "Mozilla/4.06 [en] (WinNT; I)"
98.26.161.88 - - [19/Jan/2013:18:00:27 -0600] "-" 400 0 "-" "-"
79.42.241.210 - - [19/Jan/2013:19:03:41 -0600] "-" 400 0 "-" "-"
176.61.139.128 - - [19/Jan/2013:19:06:34 -0600] "GET http://37.28.156.211/sprawdza.php HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; Trident/5.0)"
176.61.139.128 - - [19/Jan/2013:19:06:41 -0600] "GET http://37.28.156.211/sprawdza.php HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; Trident/5.0)"
124.115.6.15 - - [19/Jan/2013:21:29:34 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0(compatible; Sosospider/2.0; +http://help.soso.com/webspider.htm)"
65.111.177.188 - - [19/Jan/2013:21:49:16 -0600] "-" 400 0 "-" "-"
65.111.177.188 - - [19/Jan/2013:21:49:16 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [19/Jan/2013:21:49:18 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [19/Jan/2013:21:49:19 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [19/Jan/2013:21:49:20 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
66.249.74.103 - - [19/Jan/2013:22:16:05 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)"
66.249.74.103 - - [19/Jan/2013:22:16:05 -0600] "GET /xmlrpc.php HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)"
119.63.193.132 - - [20/Jan/2013:00:39:19 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
180.76.5.179 - - [20/Jan/2013:03:24:40 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1; rv:6.0.2) Gecko/20100101 Firefox/6.0.2"
146.0.74.67 - - [20/Jan/2013:09:01:17 -0600] "GET http://www.yahoo.com/ HTTP/1.1" 301 178 "-" "Internet Explorer 4.01"
65.111.177.188 - - [20/Jan/2013:09:42:12 -0600] "-" 400 0 "-" "-"
65.111.177.188 - - [20/Jan/2013:09:42:12 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [20/Jan/2013:09:42:14 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [20/Jan/2013:09:42:15 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [20/Jan/2013:09:42:16 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
180.76.6.233 - - [20/Jan/2013:10:18:19 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1; rv:6.0.2) Gecko/20100101 Firefox/6.0.2"
193.43.252.252 - - [20/Jan/2013:10:57:48 -0600] "HEAD / HTTP/1.1" 301 0 "http://torrentland.biz" "-"
98.23.232.73 - - [20/Jan/2013:12:08:52 -0600] "GET /login HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.232.73 - - [20/Jan/2013:12:08:52 -0600] "GET /css/style.css HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.232.73 - - [20/Jan/2013:12:08:55 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.232.73 - - [20/Jan/2013:12:08:56 -0600] "GET /list HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.232.73 - - [20/Jan/2013:12:14:06 -0600] "GET /css/style.css HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.232.73 - - [20/Jan/2013:12:14:11 -0600] "-" 400 0 "-" "-"
95.108.151.244 - - [20/Jan/2013:13:02:17 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots)"
95.108.151.244 - - [20/Jan/2013:13:02:17 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots)"
70.45.211.154 - - [20/Jan/2013:13:09:30 -0600] "-" 400 0 "-" "-"
70.45.211.154 - - [20/Jan/2013:13:09:30 -0600] "-" 400 0 "-" "-"
202.46.59.157 - - [20/Jan/2013:14:38:35 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
119.63.193.196 - - [20/Jan/2013:14:38:38 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
146.0.74.67 - - [20/Jan/2013:15:00:40 -0600] "GET http://37.28.156.211/sprawdza.php HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; Trident/5.0)"
139.52.219.176 - - [20/Jan/2013:15:21:02 -0600] "-" 400 0 "-" "-"
64.246.165.190 - - [20/Jan/2013:15:48:43 -0600] "GET /robots.txt HTTP/1.0" 301 178 "-" "Mozilla/5.0 (Windows; U; Windows NT 5.1; en; rv:1.9.0.13) Gecko/2009073022 Firefox/3.5.2 (.NET CLR 3.5.30729) SurveyBot/2.3 (DomainTools)"
64.246.165.190 - - [20/Jan/2013:15:48:44 -0600] "GET / HTTP/1.1" 301 178 "http://whois.domaintools.com/ninjas.com" "Mozilla/5.0 (Windows; U; Windows NT 5.1; en; rv:1.9.0.13) Gecko/2009073022 Firefox/3.5.2 (.NET CLR 3.5.30729) SurveyBot/2.3 (DomainTools)"
64.246.165.190 - - [20/Jan/2013:15:48:44 -0600] "GET /blog/ HTTP/1.1" 301 178 "http://whois.domaintools.com/ninjas.com" "Mozilla/5.0 (Windows; U; Windows NT 5.1; en; rv:1.9.0.13) Gecko/2009073022 Firefox/3.5.2 (.NET CLR 3.5.30729) SurveyBot/2.3 (DomainTools)"
64.246.165.190 - - [20/Jan/2013:15:48:45 -0600] "GET /blog/ HTTP/1.1" 301 178 "http://whois.domaintools.com/ninjas.com" "Mozilla/5.0 (Windows; U; Windows NT 5.1; en; rv:1.9.0.13) Gecko/2009073022 Firefox/3.5.2 (.NET CLR 3.5.30729) SurveyBot/2.3 (DomainTools)"
178.154.163.29 - - [20/Jan/2013:16:10:30 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots)"
98.23.232.73 - - [20/Jan/2013:16:17:01 -0600] "GET /login HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.232.73 - - [20/Jan/2013:16:17:02 -0600] "GET /css/style.css HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.232.73 - - [20/Jan/2013:16:18:45 -0600] "-" 400 0 "-" "-"
98.23.232.73 - - [20/Jan/2013:16:59:42 -0600] "-" 400 0 "-" "-"
98.23.232.73 - - [20/Jan/2013:16:59:43 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.232.73 - - [20/Jan/2013:16:59:44 -0600] "GET /list HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.232.73 - - [20/Jan/2013:16:59:49 -0600] "-" 400 0 "-" "-"
178.154.163.29 - - [20/Jan/2013:20:42:21 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots)"
108.34.225.90 - - [20/Jan/2013:20:42:21 -0600] "-" 400 0 "-" "-"
67.19.79.218 - - [20/Jan/2013:22:57:02 -0600] "GET /robots.txt HTTP/1.0" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 5.01; Windows NT)"
65.111.177.188 - - [20/Jan/2013:23:40:14 -0600] "-" 400 0 "-" "-"
65.111.177.188 - - [20/Jan/2013:23:40:15 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [20/Jan/2013:23:40:17 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [20/Jan/2013:23:40:18 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [20/Jan/2013:23:40:20 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
119.63.193.195 - - [21/Jan/2013:00:40:08 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
96.254.171.2 - - [21/Jan/2013:02:56:37 -0600] "\x00" 400 166 "-" "-"
96.254.171.2 - - [21/Jan/2013:02:56:38 -0600] "GET http://server6.cyberpods.net/azenv.php HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.2.28) Gecko/20120306 Firefox/3.6.28 (.NET CLR 3.5.30729)"
96.254.171.2 - - [21/Jan/2013:02:56:39 -0600] "GET http://server6.cyberpods.net/azenv.php HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.2.28) Gecko/20120306 Firefox/3.6.28 (.NET CLR 3.5.30729)"
96.254.171.2 - - [21/Jan/2013:02:56:41 -0600] "\x04\x01\x00P`/\xE7U0\x00" 400 166 "-" "-"
96.254.171.2 - - [21/Jan/2013:02:56:41 -0600] "\x04\x01\x00P`/\xE7U0\x00" 400 166 "-" "-"
96.254.171.2 - - [21/Jan/2013:02:56:41 -0600] "\x05\x01\x00" 400 166 "-" "-"
183.164.155.30 - - [21/Jan/2013:03:08:07 -0600] "e" 400 166 "-" "-"
65.111.177.188 - - [21/Jan/2013:04:54:31 -0600] "-" 400 0 "-" "-"
65.111.177.188 - - [21/Jan/2013:04:54:31 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [21/Jan/2013:04:54:33 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [21/Jan/2013:04:54:35 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
65.111.177.188 - - [21/Jan/2013:04:54:37 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.56 Safari/535.11"
124.115.6.15 - - [21/Jan/2013:05:24:48 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0(compatible; Sosospider/2.0; +http://help.soso.com/webspider.htm)"
178.154.163.29 - - [21/Jan/2013:06:57:45 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots)"
178.154.163.29 - - [21/Jan/2013:08:15:14 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots)"
202.46.53.31 - - [21/Jan/2013:09:12:23 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
119.63.193.131 - - [21/Jan/2013:09:13:39 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
188.165.109.82 - - [21/Jan/2013:10:49:55 -0600] "-" 400 0 "-" "-"
188.165.109.82 - - [21/Jan/2013:10:49:55 -0600] "-" 400 0 "-" "-"
98.23.232.73 - - [21/Jan/2013:11:01:14 -0600] "-" 400 0 "-" "-"
24.12.3.89 - - [21/Jan/2013:11:36:11 -0600] "-" 400 0 "-" "-"
24.12.3.89 - - [21/Jan/2013:11:36:51 -0600] "-" 400 0 "-" "-"
24.12.3.89 - - [21/Jan/2013:11:36:51 -0600] "-" 400 0 "-" "-"
66.162.125.100 - - [21/Jan/2013:12:33:30 -0600] "GET /vicidial/project_auth_entries.txt HTTP/1.1" 301 178 "-" "-"
70.75.177.99 - - [21/Jan/2013:12:49:41 -0600] "-" 400 0 "-" "-"
180.76.5.156 - - [21/Jan/2013:14:44:04 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1; rv:6.0.2) Gecko/20100101 Firefox/6.0.2"
180.76.6.37 - - [21/Jan/2013:14:44:04 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1; rv:6.0.2) Gecko/20100101 Firefox/6.0.2"
119.63.193.195 - - [21/Jan/2013:15:14:49 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
78.129.149.17 - - [21/Jan/2013:15:24:33 -0600] "GET http://www.intel.com/ HTTP/1.1" 301 178 "-" "Internet Explorer 4.01"
98.193.197.124 - - [21/Jan/2013:15:28:02 -0600] "-" 400 0 "-" "-"
66.249.75.69 - - [21/Jan/2013:17:55:38 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)"
66.249.75.69 - - [21/Jan/2013:17:55:38 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)"
78.129.149.17 - - [21/Jan/2013:18:17:45 -0600] "GET http://37.28.156.211/sprawdza.php HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; Trident/5.0)"
70.45.211.154 - - [21/Jan/2013:19:50:05 -0600] "-" 400 0 "-" "-"
70.45.211.154 - - [21/Jan/2013:19:50:06 -0600] "-" 400 0 "-" "-"
70.45.211.154 - - [21/Jan/2013:19:50:06 -0600] "-" 400 0 "-" "-"
98.23.232.73 - - [21/Jan/2013:20:16:38 -0600] "-" 400 0 "-" "-"
98.23.232.73 - - [21/Jan/2013:20:16:51 -0600] "-" 400 0 "-" "-"
98.23.232.73 - - [21/Jan/2013:20:20:38 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (X11; Linux i686; rv:6.0.2) Gecko/20100101 Firefox/6.0.2"
98.23.232.73 - - [21/Jan/2013:20:20:43 -0600] "-" 400 0 "-" "-"
98.23.232.73 - - [21/Jan/2013:20:20:47 -0600] "-" 400 0 "-" "-"
98.23.232.73 - - [21/Jan/2013:20:20:52 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (X11; Linux i686; rv:6.0.2) Gecko/20100101 Firefox/6.0.2"
98.23.232.73 - - [21/Jan/2013:20:20:57 -0600] "-" 400 0 "-" "-"
98.23.232.73 - - [21/Jan/2013:20:21:06 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (X11; Linux i686; rv:6.0.2) Gecko/20100101 Firefox/6.0.2"
98.23.232.73 - - [21/Jan/2013:20:21:11 -0600] "-" 400 0 "-" "-"
202.46.53.99 - - [21/Jan/2013:21:13:10 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
119.63.193.132 - - [21/Jan/2013:21:14:25 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
96.3.101.170 - - [21/Jan/2013:23:26:14 -0600] "-" 400 0 "-" "-"
180.76.5.137 - - [22/Jan/2013:01:22:24 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; Baiduspider/2.0; +http://www.baidu.com/search/spider.html)"
194.176.39.150 - - [22/Jan/2013:02:35:36 -0600] "-" 400 0 "-" "-"
194.176.39.150 - - [22/Jan/2013:02:35:36 -0600] "-" 400 0 "-" "-"
210.83.204.186 - - [22/Jan/2013:03:34:29 -0600] "HEAD / HTTP/1.0" 301 0 "-" "-"
41.93.33.251 - - [22/Jan/2013:03:55:35 -0600] "-" 400 0 "-" "-"
129.125.154.210 - - [22/Jan/2013:04:22:16 -0600] "-" 400 0 "-" "-"
119.63.193.132 - - [22/Jan/2013:05:15:29 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
193.142.62.64 - - [22/Jan/2013:06:33:32 -0600] "HEAD / HTTP/1.1" 301 0 "-" "-"
66.249.75.70 - - [22/Jan/2013:06:56:45 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)"
66.249.75.70 - - [22/Jan/2013:06:56:46 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)"
218.30.116.99 - - [22/Jan/2013:11:13:26 -0600] "GET /blog/ HTTP/1.1" 301 178 "http://mysite.com/" "Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 6.1; Trident/4.0; SLCC2; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729; Media Center PC 6.0; Tablet PC 2.0)"
70.75.177.99 - - [22/Jan/2013:12:51:40 -0600] "-" 400 0 "-" "-"
124.115.6.15 - - [22/Jan/2013:13:22:13 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0(compatible; Sosospider/2.0; +http://help.soso.com/webspider.htm)"
119.63.193.194 - - [22/Jan/2013:14:38:06 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
78.108.63.44 - - [22/Jan/2013:14:47:00 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 6.1; rv:10.0) Gecko/20100101 Firefox/10.0"
78.108.63.44 - - [22/Jan/2013:14:47:07 -0600] "-" 400 0 "-" "-"
78.108.63.44 - - [22/Jan/2013:14:47:12 -0600] "GET /login HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 6.1; rv:10.0) Gecko/20100101 Firefox/10.0"
78.108.63.44 - - [22/Jan/2013:14:47:15 -0600] "GET /css/style.css HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 6.1; rv:10.0) Gecko/20100101 Firefox/10.0"
96.254.171.2 - - [22/Jan/2013:15:04:31 -0600] "\x00" 400 166 "-" "-"
96.254.171.2 - - [22/Jan/2013:15:04:32 -0600] "GET http://server6.cyberpods.net/azenv.php HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.2.28) Gecko/20120306 Firefox/3.6.28 (.NET CLR 3.5.30729)"
96.254.171.2 - - [22/Jan/2013:15:04:33 -0600] "GET http://server6.cyberpods.net/azenv.php HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.2.28) Gecko/20120306 Firefox/3.6.28 (.NET CLR 3.5.30729)"
96.254.171.2 - - [22/Jan/2013:15:04:35 -0600] "\x04\x01\x00P`/\xE7U0\x00" 400 166 "-" "-"
96.254.171.2 - - [22/Jan/2013:15:04:35 -0600] "\x04\x01\x00P`/\xE7U0\x00" 400 166 "-" "-"
96.254.171.2 - - [22/Jan/2013:15:04:35 -0600] "\x05\x01\x00" 400 166 "-" "-"
98.23.232.73 - - [22/Jan/2013:15:10:34 -0600] "GET /login HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.232.73 - - [22/Jan/2013:15:10:34 -0600] "GET /css/style.css HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.232.73 - - [22/Jan/2013:15:10:41 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.232.73 - - [22/Jan/2013:15:10:43 -0600] "GET /list HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
202.46.53.83 - - [22/Jan/2013:19:14:15 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
119.63.193.194 - - [22/Jan/2013:19:15:34 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
176.61.139.128 - - [22/Jan/2013:23:06:44 -0600] "GET http://www.yahoo.com/ HTTP/1.1" 301 178 "-" "Mozilla/4.06 [en] (WinNT; I)"
91.194.77.16 - - [23/Jan/2013:00:14:32 -0600] "GET /phpmyadmin/scripts/setup.php HTTP/1.1" 301 178 "http://ninjas.COM/phpmyadmin/scripts/setup.php" "Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 5.1; SV1; .NET CLR 2.0.50727)"
176.61.139.128 - - [23/Jan/2013:02:11:30 -0600] "GET http://37.28.156.211/sprawdza.php HTTP/1.1" 301 178 "-" "Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; Trident/5.0)"
81.236.48.97 - - [23/Jan/2013:03:06:25 -0600] "-" 400 0 "-" "-"
81.236.48.97 - - [23/Jan/2013:03:06:25 -0600] "-" 400 0 "-" "-"
119.63.193.131 - - [23/Jan/2013:05:15:19 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
176.123.0.32 - - [23/Jan/2013:09:01:39 -0600] "GET /wp-login.php HTTP/1.1" 301 178 "http://ninjas.com/wp-login.php" "Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1)"
202.46.58.39 - - [23/Jan/2013:09:16:01 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
119.63.193.196 - - [23/Jan/2013:09:17:21 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
98.23.236.241 - - [23/Jan/2013:09:26:21 -0600] "GET /login HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.236.241 - - [23/Jan/2013:09:26:21 -0600] "GET /list HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
98.23.236.241 - - [23/Jan/2013:09:26:23 -0600] "GET /css/style.css HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
180.76.5.51 - - [23/Jan/2013:10:37:02 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1; rv:6.0.2) Gecko/20100101 Firefox/6.0.2"
180.76.6.222 - - [23/Jan/2013:10:37:02 -0600] "GET /robots.txt HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows NT 5.1; rv:6.0.2) Gecko/20100101 Firefox/6.0.2"
70.75.177.99 - - [23/Jan/2013:12:51:39 -0600] "-" 400 0 "-" "-"
176.9.111.204 - - [23/Jan/2013:12:58:14 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 5.1)"
202.46.57.62 - - [23/Jan/2013:14:38:43 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
119.63.193.194 - - [23/Jan/2013:14:38:46 -0600] "GET / HTTP/1.1" 301 178 "-" "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)"
94.153.127.247 - - [23/Jan/2013:15:13:57 -0600] "GET //administrator/components/com_jce/jce.xml HTTP/1.1" 301 178 "-" "Mozilla/5.0 (Windows; U; Windows NT 6.1; da) AppleWebKit/522.15.5 (KHTML, like Gecko) Version/3.0.3 Safari/522.15.5"
