package Module::Format::ModuleList;
$Module::Format::ModuleList::VERSION = '0.2.0';
use warnings;
use strict;


use Module::Format::Module;


sub new
{
    my $class = shift;
    my $self = bless {}, $class;
    $self->_init(@_);
    return $self;
}

sub _modules
{
    my $self = shift;

    if (@_)
    {
        $self->{_modules} = shift;
    }

    return $self->{_modules};
}

sub _add_module
{
    my ( $self, $module ) = @_;

    if ( not $module->isa('Module::Format::Module') )
    {
        die "Module is " . ref($module) . " instead of Module::Format::Module.";
    }

    push @{ $self->_modules() }, $module;

    return;
}

sub _init
{
    my ( $self, $args ) = @_;

    $self->_modules( [] );

    foreach my $module ( @{ $args->{modules} } )
    {
        $self->_add_module($module);
    }

    return;
}


sub get_modules_list_copy
{
    my ($self) = @_;

    return [ @{ $self->_modules() } ];
}


sub format_as
{
    my ( $self, $format ) = @_;

    return [ map { $_->format_as($format) } @{ $self->_modules() } ];
}


sub sane_from_guesses
{
    my ( $class, $args ) = @_;

    my @modules;

    foreach my $string ( @{ $args->{'values'} } )
    {
        my $module = Module::Format::Module->from_guess( { value => $string } );

        if ( !$module->is_sane() )
        {
            die "Value '$string' does not evaluate as a sane module.";
        }

        push @modules, $module;
    }

    return $class->new( { modules => \@modules } );
}


1;    # End of Module::Format::ModuleList

__END__

=pod

=encoding UTF-8

=head1 NAME

Module::Format::ModuleList - an ordered list of L<Module::Format::Module>.

=head1 VERSION

version 0.2.0

=head1 SYNOPSIS

    use Module::Format::ModuleList;

    my $list = Module::Format::ModuleList->new(
        {
            modules =>
            [
                Module::Format::Module->from_guess({ value => 'XML::RSS'}),
                Module::Format::Module->from_guess({ value => 'Data-Dumper'}),
            ],
        }
    );

    foreach my $name (@{$list->format_as('rpm_colon')})
    {
        print "$name\n";
    }

    my $list = Module::Format::ModuleList->sane_from_guesses(
        {
            values =>
            [qw/
                Algorithm::Permutations
                rpm(Foo::Bar::Baz)
                perl-HTML-TreeBuilder-LibXML
            /],
        },
    )

=head1 VERSION

version 0.2.0

=head1 FUNCTIONS

=head2 my $list = Module::Format::ModuleList->new({ modules => \@list})

The generic constructor. Initialises a new module list from a @list which
must be an array of L<Module::Format::Module> modules.

=head2 my $array_ref = $self->get_modules_list_copy()

Returns a shallow copy of the modules list contained in the object. Useful
for debugging.

=head2 my $array_ref_of_strings = $list->format_as($format)

Returns a list containing all the modules formatted in the $format . See
L<Module::Format::Module> for the available formats.

=head2 my $list_obj = Module::Format::ModuleList->sane_from_guesses({values => \@list_of_strings});

Initialises a module list object from a list of strings by using
L<Module::Format::Module>'s from_guess on each string and while checking the
guesses for sanity. See the synposis for an example.

=head1 AUTHOR

Shlomi Fish, L<http://www.shlomifish.org/>

=head1 BUGS

Please report any bugs or feature requests to C<bug-module-format at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Module-Format>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Module::Format::ModuleList

You can also look for information at:

=over 4

=item * MetaCPAN

L<https://metacpan.org/release/Module-Format>

=item * RT: CPAN's request tracker

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Module-Format>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Module-Format>

=back

=head1 ACKNOWLEDGEMENTS

=head1 COPYRIGHT & LICENSE

Copyright 2010 Shlomi Fish.

This program is distributed under the MIT (X11) License:
L<http://www.opensource.org/licenses/mit-license.php>

Permission is hereby granted, free of charge, to any person
obtaining a copy of this software and associated documentation
files (the "Software"), to deal in the Software without
restriction, including without limitation the rights to use,
copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following
conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

=head1 AUTHOR

Shlomi Fish <shlomif@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2018 by Shlomi Fish.

This is free software, licensed under:

  The MIT (X11) License

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website
L<https://github.com/shlomif/perl-Module-Format/issues>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=for :stopwords cpan testmatrix url annocpan anno bugtracker rt cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 SUPPORT

=head2 Perldoc

You can find documentation for this module with the perldoc command.

  perldoc Module::Format::ModuleList

=head2 Websites

The following websites have more information about this module, and may be of help to you. As always,
in addition to those websites please use your favorite search engine to discover more resources.

=over 4

=item *

MetaCPAN

A modern, open-source CPAN search engine, useful to view POD in HTML format.

L<https://metacpan.org/release/Module-Format>

=item *

Search CPAN

The default CPAN search engine, useful to view POD in HTML format.

L<http://search.cpan.org/dist/Module-Format>

=item *

RT: CPAN's Bug Tracker

The RT ( Request Tracker ) website is the default bug/issue tracking system for CPAN.

L<https://rt.cpan.org/Public/Dist/Display.html?Name=Module-Format>

=item *

AnnoCPAN

The AnnoCPAN is a website that allows community annotations of Perl module documentation.

L<http://annocpan.org/dist/Module-Format>

=item *

CPAN Ratings

The CPAN Ratings is a website that allows community ratings and reviews of Perl modules.

L<http://cpanratings.perl.org/d/Module-Format>

=item *

CPANTS

The CPANTS is a website that analyzes the Kwalitee ( code metrics ) of a distribution.

L<http://cpants.cpanauthors.org/dist/Module-Format>

=item *

CPAN Testers

The CPAN Testers is a network of smoke testers who run automated tests on uploaded CPAN distributions.

L<http://www.cpantesters.org/distro/M/Module-Format>

=item *

CPAN Testers Matrix

The CPAN Testers Matrix is a website that provides a visual overview of the test results for a distribution on various Perls/platforms.

L<http://matrix.cpantesters.org/?dist=Module-Format>

=item *

CPAN Testers Dependencies

The CPAN Testers Dependencies is a website that shows a chart of the test results of all dependencies for a distribution.

L<http://deps.cpantesters.org/?module=Module::Format>

=back

=head2 Bugs / Feature Requests

Please report any bugs or feature requests by email to C<bug-module-format at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/Public/Bug/Report.html?Queue=Module-Format>. You will be automatically notified of any
progress on the request by the system.

=head2 Source Code

The code is open to the world, and available for you to hack on. Please feel free to browse it and play
with it, or whatever. If you want to contribute patches, please send me a diff or prod me to pull
from your repository :)

L<https://github.com/shlomif/perl-Module-Format>

  git clone git://github.com/shlomif/perl-Module-Format.git

=cut
