# ABSTRACT: Driver for the Czech tagset of the CoNLL 2006 and 2007 Shared Tasks.
# Skript, kterým Zdeněk konvertoval značky PDT do CoNLL, je v /home/zabokrtsky/projects/CoNLL/for_Jens.
# Pozor, Zdeněk připravoval pouze data pro CoNLL 2007! Ta pro rok 2006 si chystali organizátoři sami,
# zřejmě z PDT 1.0 z formátu CSTS.
# Copyright © 2008, 2014 Dan Zeman <zeman@ufal.mff.cuni.cz>

package Lingua::Interset::Tagset::CS::Conll;
use strict;
use warnings;
our $VERSION = '2.016'; # VERSION: generated by DZP::OurPkgVersion

use utf8;
use open ':utf8';
use namespace::autoclean;
use Moose;
extends 'Lingua::Interset::Tagset::CS::Pdt';



#------------------------------------------------------------------------------
# Creates atomic drivers for 12 surface features (11 inherited from Pdt).
#------------------------------------------------------------------------------
sub _create_atoms
{
    my $self = shift;
    my $atoms = $self->SUPER::_create_atoms();
    # SEMANTIC CLASS ####################
    # Sem contains the character after ";" in lemma suffix. It encodes types of proper nouns (named entities):
    $atoms->{nametype} = $self->create_simple_atom
    (
        'intfeature' => 'nametype',
        'simple_decode_map' =>
        {
            'G' => 'geo',
            'Y' => 'giv',
            'S' => 'sur',
            'E' => 'nat',
            'R' => 'pro',
            'K' => 'com',
            'm' => 'oth',
            'o' => 'col',
            'H' => 'che',
            'U' => 'med',
            'L' => 'sci',
            'j' => 'jus',
            'g' => 'tec',
            'c' => 'cel',
            'y' => 'hob',
            'b' => 'fin',
            'u' => 'cul',
            'w' => 'spo',
            'p' => 'gov',
            'z' => 'env'
        }
    );
    return $atoms;
}



#------------------------------------------------------------------------------
# Decodes a physical tag (string) and returns the corresponding feature
# structure.
#------------------------------------------------------------------------------
sub decode
{
    my $self = shift;
    my $tag = shift;
    my %conll_features;
    my $pdt = _conll_to_pdt($tag, \%conll_features);
    my $fs = $self->SUPER::decode($pdt);
    if($conll_features{Sem})
    {
        my $atoms = $self->atoms();
        $atoms->{nametype}->decode_and_merge_hard($conll_features{Sem}, $fs);
        $fs->set_nountype('prop');
    }
    # Here we could set $fs->set_tagset('cs::conll') but we will not so that all
    # the descendants of cs::pdt can share the same feature structures.
    # (The cs::conll feature structures are not identical to cs::pdt but they do
    # not add their own 'other' values and that matters.)
    return $fs;
}



#------------------------------------------------------------------------------
# Takes feature structure and returns the corresponding physical tag (string).
#------------------------------------------------------------------------------
sub encode
{
    my $self = shift;
    my $fs = shift; # Lingua::Interset::FeatureStructure
    my $pdt = $self->SUPER::encode($fs);
    my $conll = _pdt_to_conll($pdt);
    if($fs->is_proper_noun() || $fs->nametype() ne '')
    {
        my $atoms = $self->atoms();
        my $sem = $atoms->{nametype}->encode($fs);
        $sem = 'm' if($sem eq '');
        if($conll =~ s/_$//)
        {
            $conll .= "Sem=$sem";
        }
        else
        {
            $conll .= "|Sem=$sem";
        }
    }
    return $conll;
}



#------------------------------------------------------------------------------
# CoNLL tagsets in Interset are traditionally three values separated by tabs.
# The values come from the CoNLL columns CPOS, POS and FEAT. For Czech, these
# values are derived from the tagset of the Prague Dependency Treebank.
# However, there is an additional feature called Sem that is derived from PDT
# lemmas and that must be decoded and encoded separately.
# This function converts a PDT tag to CoNLL (minus Sem).
#------------------------------------------------------------------------------
sub _pdt_to_conll
{
    my $tag = shift;
    $tag =~ s/^(.)(.)//;
    my ($pos, $subpos) = ($1, $2);
    my @features_conll;
    foreach my $name (qw(Gen Num Cas PGe PNu Per Ten Gra Neg Voi Rs1 Rs2 Var))
    {
        $tag =~ s/^(.)//;
        if($1 ne '-')
        {
            push(@features_conll, "$name=$1");
        }
    }
    my $features = (@features_conll) ? join('|', @features_conll) : '_';
    $tag = "$pos\t$subpos\t$features";
    return $tag;
}



#------------------------------------------------------------------------------
# This function converts a CoNLL tag (except the Sem feature) to PDT.
#------------------------------------------------------------------------------
sub _conll_to_pdt
{
    my $tag = shift;
    my $features_conll = shift; # output parameter: hash reference; the caller will want to access the Sem feature!
    my %storage;
    if(ref($features_conll) ne 'HASH')
    {
        $features_conll = \%storage;
    }
    # three components: coarse-grained pos, fine-grained pos, features
    # example: N\tN\tGen=M|Num=S|Cas=1|Neg=A
    my ($pos, $subpos, $features) = split(/\s+/, $tag);
    # The CoNLL tagset is derived from the PDT tagset.
    # Coarse-grained POS is the first character of the PDT tag.
    # Fine-grained POS is the second character of the PDT tag.
    # Features are the rest: Gen Num Cas PGe PNu Per Ten Gra Neg Voi Rs1 Rs2 Var
    # The Sem feature comes from PDT lemma, not tag.
    my $pdttag;
    if($features eq '_')
    {
        $pdttag = $pos.$subpos.'-------------';
    }
    else
    {
        $pdttag = $pos.$subpos;
        my @features_conll = split(/\|/, $features);
        foreach my $f (@features_conll)
        {
            if($f =~ m/^(\w+)=(.)$/)
            {
                $features_conll->{$1} = $2;
            }
        }
        foreach my $name (qw(Gen Num Cas PGe PNu Per Ten Gra Neg Voi Rs1 Rs2 Var))
        {
            if(defined($features_conll->{$name}) && $features_conll->{$name} ne '')
            {
                $pdttag .= $features_conll->{$name};
            }
            else
            {
                $pdttag .= '-';
            }
        }
    }
    return $pdttag;
}



#------------------------------------------------------------------------------
# Returns reference to list of known tags.
#------------------------------------------------------------------------------
sub list
{
    my $self = shift;
    my $list = <<end_of_list
A	.	_
A	2	Neg=A
A	2	Neg=A|Sem=G
A	2	Neg=A|Sem=K
A	2	Neg=A|Sem=S
A	2	Neg=A|Sem=Y
A	2	Neg=A|Sem=m
A	2	Neg=A|Var=1
A	2	Neg=N
A	A	Gen=F|Num=D|Cas=7|Gra=1|Neg=A
A	A	Gen=F|Num=D|Cas=7|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=D|Cas=7|Gra=1|Neg=N
A	A	Gen=F|Num=D|Cas=7|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=D|Cas=7|Gra=2|Neg=A
A	A	Gen=F|Num=D|Cas=7|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=D|Cas=7|Gra=2|Neg=N
A	A	Gen=F|Num=D|Cas=7|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=D|Cas=7|Gra=3|Neg=A
A	A	Gen=F|Num=D|Cas=7|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=D|Cas=7|Gra=3|Neg=N
A	A	Gen=F|Num=D|Cas=7|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=1|Gra=1|Neg=A
A	A	Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Sem=G
A	A	Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Sem=K
A	A	Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Sem=R
A	A	Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Sem=U
A	A	Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Sem=m
A	A	Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Sem=o
A	A	Gen=F|Num=P|Cas=1|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=1|Gra=1|Neg=N
A	A	Gen=F|Num=P|Cas=1|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=1|Gra=2|Neg=A
A	A	Gen=F|Num=P|Cas=1|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=1|Gra=2|Neg=N
A	A	Gen=F|Num=P|Cas=1|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=1|Gra=3|Neg=A
A	A	Gen=F|Num=P|Cas=1|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=1|Gra=3|Neg=N
A	A	Gen=F|Num=P|Cas=1|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=2|Gra=1|Neg=A
A	A	Gen=F|Num=P|Cas=2|Gra=1|Neg=A|Sem=G
A	A	Gen=F|Num=P|Cas=2|Gra=1|Neg=A|Sem=R
A	A	Gen=F|Num=P|Cas=2|Gra=1|Neg=A|Sem=S
A	A	Gen=F|Num=P|Cas=2|Gra=1|Neg=A|Sem=m
A	A	Gen=F|Num=P|Cas=2|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=2|Gra=1|Neg=N
A	A	Gen=F|Num=P|Cas=2|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=2|Gra=2|Neg=A
A	A	Gen=F|Num=P|Cas=2|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=2|Gra=2|Neg=N
A	A	Gen=F|Num=P|Cas=2|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=2|Gra=3|Neg=A
A	A	Gen=F|Num=P|Cas=2|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=2|Gra=3|Neg=N
A	A	Gen=F|Num=P|Cas=2|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=3|Gra=1|Neg=A
A	A	Gen=F|Num=P|Cas=3|Gra=1|Neg=A|Sem=G
A	A	Gen=F|Num=P|Cas=3|Gra=1|Neg=A|Sem=m
A	A	Gen=F|Num=P|Cas=3|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=3|Gra=1|Neg=N
A	A	Gen=F|Num=P|Cas=3|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=3|Gra=2|Neg=A
A	A	Gen=F|Num=P|Cas=3|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=3|Gra=2|Neg=N
A	A	Gen=F|Num=P|Cas=3|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=3|Gra=3|Neg=A
A	A	Gen=F|Num=P|Cas=3|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=3|Gra=3|Neg=N
A	A	Gen=F|Num=P|Cas=3|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=4|Gra=1|Neg=A
A	A	Gen=F|Num=P|Cas=4|Gra=1|Neg=A|Sem=G
A	A	Gen=F|Num=P|Cas=4|Gra=1|Neg=A|Sem=m
A	A	Gen=F|Num=P|Cas=4|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=4|Gra=1|Neg=N
A	A	Gen=F|Num=P|Cas=4|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=4|Gra=2|Neg=A
A	A	Gen=F|Num=P|Cas=4|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=4|Gra=2|Neg=N
A	A	Gen=F|Num=P|Cas=4|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=4|Gra=3|Neg=A
A	A	Gen=F|Num=P|Cas=4|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=4|Gra=3|Neg=N
A	A	Gen=F|Num=P|Cas=4|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=5|Gra=1|Neg=A
A	A	Gen=F|Num=P|Cas=5|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=5|Gra=1|Neg=N
A	A	Gen=F|Num=P|Cas=5|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=5|Gra=2|Neg=A
A	A	Gen=F|Num=P|Cas=5|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=5|Gra=2|Neg=N
A	A	Gen=F|Num=P|Cas=5|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=5|Gra=3|Neg=A
A	A	Gen=F|Num=P|Cas=5|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=5|Gra=3|Neg=N
A	A	Gen=F|Num=P|Cas=5|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=6|Gra=1|Neg=A
A	A	Gen=F|Num=P|Cas=6|Gra=1|Neg=A|Sem=G
A	A	Gen=F|Num=P|Cas=6|Gra=1|Neg=A|Sem=S
A	A	Gen=F|Num=P|Cas=6|Gra=1|Neg=A|Sem=m
A	A	Gen=F|Num=P|Cas=6|Gra=1|Neg=A|Sem=o
A	A	Gen=F|Num=P|Cas=6|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=6|Gra=1|Neg=N
A	A	Gen=F|Num=P|Cas=6|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=6|Gra=2|Neg=A
A	A	Gen=F|Num=P|Cas=6|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=6|Gra=2|Neg=N
A	A	Gen=F|Num=P|Cas=6|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=6|Gra=3|Neg=A
A	A	Gen=F|Num=P|Cas=6|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=6|Gra=3|Neg=N
A	A	Gen=F|Num=P|Cas=6|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=7|Gra=1|Neg=A
A	A	Gen=F|Num=P|Cas=7|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=7|Gra=1|Neg=A|Var=7
A	A	Gen=F|Num=P|Cas=7|Gra=1|Neg=N
A	A	Gen=F|Num=P|Cas=7|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=7|Gra=1|Neg=N|Var=7
A	A	Gen=F|Num=P|Cas=7|Gra=2|Neg=A
A	A	Gen=F|Num=P|Cas=7|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=7|Gra=2|Neg=A|Var=7
A	A	Gen=F|Num=P|Cas=7|Gra=2|Neg=N
A	A	Gen=F|Num=P|Cas=7|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=7|Gra=2|Neg=N|Var=7
A	A	Gen=F|Num=P|Cas=7|Gra=3|Neg=A
A	A	Gen=F|Num=P|Cas=7|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=P|Cas=7|Gra=3|Neg=A|Var=7
A	A	Gen=F|Num=P|Cas=7|Gra=3|Neg=N
A	A	Gen=F|Num=P|Cas=7|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=P|Cas=7|Gra=3|Neg=N|Var=7
A	A	Gen=F|Num=S|Cas=1|Gra=1|Neg=A
A	A	Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=G
A	A	Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=K
A	A	Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=L
A	A	Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=R
A	A	Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=S
A	A	Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=m
A	A	Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Sem=o
A	A	Gen=F|Num=S|Cas=1|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=1|Gra=1|Neg=N
A	A	Gen=F|Num=S|Cas=1|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=1|Gra=2|Neg=A
A	A	Gen=F|Num=S|Cas=1|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=1|Gra=2|Neg=N
A	A	Gen=F|Num=S|Cas=1|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=1|Gra=3|Neg=A
A	A	Gen=F|Num=S|Cas=1|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=1|Gra=3|Neg=N
A	A	Gen=F|Num=S|Cas=1|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=2|Gra=1|Neg=A
A	A	Gen=F|Num=S|Cas=2|Gra=1|Neg=A|Sem=G
A	A	Gen=F|Num=S|Cas=2|Gra=1|Neg=A|Sem=K
A	A	Gen=F|Num=S|Cas=2|Gra=1|Neg=A|Sem=R
A	A	Gen=F|Num=S|Cas=2|Gra=1|Neg=A|Sem=m
A	A	Gen=F|Num=S|Cas=2|Gra=1|Neg=A|Sem=o
A	A	Gen=F|Num=S|Cas=2|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=2|Gra=1|Neg=N
A	A	Gen=F|Num=S|Cas=2|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=2|Gra=2|Neg=A
A	A	Gen=F|Num=S|Cas=2|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=2|Gra=2|Neg=N
A	A	Gen=F|Num=S|Cas=2|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=2|Gra=3|Neg=A
A	A	Gen=F|Num=S|Cas=2|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=2|Gra=3|Neg=N
A	A	Gen=F|Num=S|Cas=2|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=3|Gra=1|Neg=A
A	A	Gen=F|Num=S|Cas=3|Gra=1|Neg=A|Sem=G
A	A	Gen=F|Num=S|Cas=3|Gra=1|Neg=A|Sem=m
A	A	Gen=F|Num=S|Cas=3|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=3|Gra=1|Neg=N
A	A	Gen=F|Num=S|Cas=3|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=3|Gra=2|Neg=A
A	A	Gen=F|Num=S|Cas=3|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=3|Gra=2|Neg=N
A	A	Gen=F|Num=S|Cas=3|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=3|Gra=3|Neg=A
A	A	Gen=F|Num=S|Cas=3|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=3|Gra=3|Neg=N
A	A	Gen=F|Num=S|Cas=3|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=4|Gra=1|Neg=A
A	A	Gen=F|Num=S|Cas=4|Gra=1|Neg=A|Sem=G
A	A	Gen=F|Num=S|Cas=4|Gra=1|Neg=A|Sem=K
A	A	Gen=F|Num=S|Cas=4|Gra=1|Neg=A|Sem=S
A	A	Gen=F|Num=S|Cas=4|Gra=1|Neg=A|Sem=m
A	A	Gen=F|Num=S|Cas=4|Gra=1|Neg=A|Sem=o
A	A	Gen=F|Num=S|Cas=4|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=4|Gra=1|Neg=N
A	A	Gen=F|Num=S|Cas=4|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=4|Gra=2|Neg=A
A	A	Gen=F|Num=S|Cas=4|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=4|Gra=2|Neg=N
A	A	Gen=F|Num=S|Cas=4|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=4|Gra=3|Neg=A
A	A	Gen=F|Num=S|Cas=4|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=4|Gra=3|Neg=N
A	A	Gen=F|Num=S|Cas=4|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=5|Gra=1|Neg=A
A	A	Gen=F|Num=S|Cas=5|Gra=1|Neg=A|Sem=R
A	A	Gen=F|Num=S|Cas=5|Gra=1|Neg=A|Sem=m
A	A	Gen=F|Num=S|Cas=5|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=5|Gra=1|Neg=N
A	A	Gen=F|Num=S|Cas=5|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=5|Gra=2|Neg=A
A	A	Gen=F|Num=S|Cas=5|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=5|Gra=2|Neg=N
A	A	Gen=F|Num=S|Cas=5|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=5|Gra=3|Neg=A
A	A	Gen=F|Num=S|Cas=5|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=5|Gra=3|Neg=N
A	A	Gen=F|Num=S|Cas=5|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=6|Gra=1|Neg=A
A	A	Gen=F|Num=S|Cas=6|Gra=1|Neg=A|Sem=G
A	A	Gen=F|Num=S|Cas=6|Gra=1|Neg=A|Sem=K
A	A	Gen=F|Num=S|Cas=6|Gra=1|Neg=A|Sem=m
A	A	Gen=F|Num=S|Cas=6|Gra=1|Neg=A|Sem=o
A	A	Gen=F|Num=S|Cas=6|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=6|Gra=1|Neg=N
A	A	Gen=F|Num=S|Cas=6|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=6|Gra=2|Neg=A
A	A	Gen=F|Num=S|Cas=6|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=6|Gra=2|Neg=N
A	A	Gen=F|Num=S|Cas=6|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=6|Gra=3|Neg=A
A	A	Gen=F|Num=S|Cas=6|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=6|Gra=3|Neg=N
A	A	Gen=F|Num=S|Cas=6|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=7|Gra=1|Neg=A
A	A	Gen=F|Num=S|Cas=7|Gra=1|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=7|Gra=1|Neg=N
A	A	Gen=F|Num=S|Cas=7|Gra=1|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=7|Gra=2|Neg=A
A	A	Gen=F|Num=S|Cas=7|Gra=2|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=7|Gra=2|Neg=N
A	A	Gen=F|Num=S|Cas=7|Gra=2|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=7|Gra=3|Neg=A
A	A	Gen=F|Num=S|Cas=7|Gra=3|Neg=A|Var=6
A	A	Gen=F|Num=S|Cas=7|Gra=3|Neg=N
A	A	Gen=F|Num=S|Cas=7|Gra=3|Neg=N|Var=6
A	A	Gen=F|Num=S|Cas=X|Gra=1|Neg=A
A	A	Gen=F|Num=S|Cas=X|Gra=1|Neg=A|Sem=G
A	A	Gen=F|Num=S|Cas=X|Gra=1|Neg=A|Sem=K
A	A	Gen=F|Num=S|Cas=X|Gra=1|Neg=A|Sem=m
A	A	Gen=F|Num=X|Cas=X|Gra=1|Neg=A
A	A	Gen=F|Num=X|Cas=X|Gra=1|Neg=A|Sem=G
A	A	Gen=F|Num=X|Cas=X|Gra=1|Neg=A|Sem=m
A	A	Gen=F|Num=X|Cas=X|Gra=1|Neg=A|Var=8
A	A	Gen=F|Num=X|Cas=X|Gra=1|Neg=N|Var=8
A	A	Gen=I|Num=P|Cas=1|Gra=1|Neg=A
A	A	Gen=I|Num=P|Cas=1|Gra=1|Neg=A|Sem=G
A	A	Gen=I|Num=P|Cas=1|Gra=1|Neg=A|Sem=R
A	A	Gen=I|Num=P|Cas=1|Gra=1|Neg=A|Sem=m
A	A	Gen=I|Num=P|Cas=1|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=1|Gra=1|Neg=N
A	A	Gen=I|Num=P|Cas=1|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=1|Gra=2|Neg=A
A	A	Gen=I|Num=P|Cas=1|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=1|Gra=2|Neg=N
A	A	Gen=I|Num=P|Cas=1|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=1|Gra=3|Neg=A
A	A	Gen=I|Num=P|Cas=1|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=1|Gra=3|Neg=N
A	A	Gen=I|Num=P|Cas=1|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=2|Gra=1|Neg=A
A	A	Gen=I|Num=P|Cas=2|Gra=1|Neg=A|Sem=G
A	A	Gen=I|Num=P|Cas=2|Gra=1|Neg=A|Sem=S
A	A	Gen=I|Num=P|Cas=2|Gra=1|Neg=A|Sem=m
A	A	Gen=I|Num=P|Cas=2|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=2|Gra=1|Neg=N
A	A	Gen=I|Num=P|Cas=2|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=2|Gra=2|Neg=A
A	A	Gen=I|Num=P|Cas=2|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=2|Gra=2|Neg=N
A	A	Gen=I|Num=P|Cas=2|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=2|Gra=3|Neg=A
A	A	Gen=I|Num=P|Cas=2|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=2|Gra=3|Neg=N
A	A	Gen=I|Num=P|Cas=2|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=3|Gra=1|Neg=A
A	A	Gen=I|Num=P|Cas=3|Gra=1|Neg=A|Sem=G
A	A	Gen=I|Num=P|Cas=3|Gra=1|Neg=A|Sem=m
A	A	Gen=I|Num=P|Cas=3|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=3|Gra=1|Neg=N
A	A	Gen=I|Num=P|Cas=3|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=3|Gra=2|Neg=A
A	A	Gen=I|Num=P|Cas=3|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=3|Gra=2|Neg=N
A	A	Gen=I|Num=P|Cas=3|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=3|Gra=3|Neg=A
A	A	Gen=I|Num=P|Cas=3|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=3|Gra=3|Neg=N
A	A	Gen=I|Num=P|Cas=3|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=4|Gra=1|Neg=A
A	A	Gen=I|Num=P|Cas=4|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=4|Gra=1|Neg=N
A	A	Gen=I|Num=P|Cas=4|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=4|Gra=2|Neg=A
A	A	Gen=I|Num=P|Cas=4|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=4|Gra=2|Neg=N
A	A	Gen=I|Num=P|Cas=4|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=4|Gra=3|Neg=A
A	A	Gen=I|Num=P|Cas=4|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=4|Gra=3|Neg=N
A	A	Gen=I|Num=P|Cas=4|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=5|Gra=1|Neg=A
A	A	Gen=I|Num=P|Cas=5|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=5|Gra=1|Neg=N
A	A	Gen=I|Num=P|Cas=5|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=5|Gra=2|Neg=A
A	A	Gen=I|Num=P|Cas=5|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=5|Gra=2|Neg=N
A	A	Gen=I|Num=P|Cas=5|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=5|Gra=3|Neg=A
A	A	Gen=I|Num=P|Cas=5|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=5|Gra=3|Neg=N
A	A	Gen=I|Num=P|Cas=5|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=6|Gra=1|Neg=A
A	A	Gen=I|Num=P|Cas=6|Gra=1|Neg=A|Sem=R
A	A	Gen=I|Num=P|Cas=6|Gra=1|Neg=A|Sem=m
A	A	Gen=I|Num=P|Cas=6|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=6|Gra=1|Neg=N
A	A	Gen=I|Num=P|Cas=6|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=6|Gra=2|Neg=A
A	A	Gen=I|Num=P|Cas=6|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=6|Gra=2|Neg=N
A	A	Gen=I|Num=P|Cas=6|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=6|Gra=3|Neg=A
A	A	Gen=I|Num=P|Cas=6|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=6|Gra=3|Neg=N
A	A	Gen=I|Num=P|Cas=6|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=7|Gra=1|Neg=A
A	A	Gen=I|Num=P|Cas=7|Gra=1|Neg=A|Sem=G
A	A	Gen=I|Num=P|Cas=7|Gra=1|Neg=A|Sem=m
A	A	Gen=I|Num=P|Cas=7|Gra=1|Neg=A|Sem=o
A	A	Gen=I|Num=P|Cas=7|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=7|Gra=1|Neg=A|Var=7
A	A	Gen=I|Num=P|Cas=7|Gra=1|Neg=N
A	A	Gen=I|Num=P|Cas=7|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=7|Gra=1|Neg=N|Var=7
A	A	Gen=I|Num=P|Cas=7|Gra=2|Neg=A
A	A	Gen=I|Num=P|Cas=7|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=7|Gra=2|Neg=A|Var=7
A	A	Gen=I|Num=P|Cas=7|Gra=2|Neg=N
A	A	Gen=I|Num=P|Cas=7|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=7|Gra=2|Neg=N|Var=7
A	A	Gen=I|Num=P|Cas=7|Gra=3|Neg=A
A	A	Gen=I|Num=P|Cas=7|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=P|Cas=7|Gra=3|Neg=A|Var=7
A	A	Gen=I|Num=P|Cas=7|Gra=3|Neg=N
A	A	Gen=I|Num=P|Cas=7|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=P|Cas=7|Gra=3|Neg=N|Var=7
A	A	Gen=I|Num=P|Cas=X|Gra=1|Neg=A
A	A	Gen=I|Num=P|Cas=X|Gra=1|Neg=A|Sem=G
A	A	Gen=I|Num=P|Cas=X|Gra=1|Neg=A|Sem=m
A	A	Gen=I|Num=P|Cas=X|Gra=1|Neg=N
A	A	Gen=I|Num=P|Cas=X|Gra=2|Neg=A
A	A	Gen=I|Num=P|Cas=X|Gra=2|Neg=N
A	A	Gen=I|Num=P|Cas=X|Gra=3|Neg=A
A	A	Gen=I|Num=P|Cas=X|Gra=3|Neg=N
A	A	Gen=I|Num=S|Cas=1|Gra=1|Neg=A
A	A	Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Sem=G
A	A	Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Sem=H
A	A	Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Sem=K
A	A	Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Sem=S
A	A	Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Sem=m
A	A	Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Sem=o
A	A	Gen=I|Num=S|Cas=1|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=1|Gra=1|Neg=N
A	A	Gen=I|Num=S|Cas=1|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=1|Gra=2|Neg=A
A	A	Gen=I|Num=S|Cas=1|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=1|Gra=2|Neg=N
A	A	Gen=I|Num=S|Cas=1|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=1|Gra=3|Neg=A
A	A	Gen=I|Num=S|Cas=1|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=1|Gra=3|Neg=N
A	A	Gen=I|Num=S|Cas=1|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=2|Gra=1|Neg=A
A	A	Gen=I|Num=S|Cas=2|Gra=1|Neg=A|Sem=G
A	A	Gen=I|Num=S|Cas=2|Gra=1|Neg=A|Sem=H
A	A	Gen=I|Num=S|Cas=2|Gra=1|Neg=A|Sem=S
A	A	Gen=I|Num=S|Cas=2|Gra=1|Neg=A|Sem=m
A	A	Gen=I|Num=S|Cas=2|Gra=1|Neg=A|Sem=o
A	A	Gen=I|Num=S|Cas=2|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=2|Gra=1|Neg=N
A	A	Gen=I|Num=S|Cas=2|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=2|Gra=2|Neg=A
A	A	Gen=I|Num=S|Cas=2|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=2|Gra=2|Neg=N
A	A	Gen=I|Num=S|Cas=2|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=2|Gra=3|Neg=A
A	A	Gen=I|Num=S|Cas=2|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=2|Gra=3|Neg=N
A	A	Gen=I|Num=S|Cas=2|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=3|Gra=1|Neg=A
A	A	Gen=I|Num=S|Cas=3|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=3|Gra=1|Neg=N
A	A	Gen=I|Num=S|Cas=3|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=3|Gra=2|Neg=A
A	A	Gen=I|Num=S|Cas=3|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=3|Gra=2|Neg=N
A	A	Gen=I|Num=S|Cas=3|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=3|Gra=3|Neg=A
A	A	Gen=I|Num=S|Cas=3|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=3|Gra=3|Neg=N
A	A	Gen=I|Num=S|Cas=3|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=4|Gra=1|Neg=A
A	A	Gen=I|Num=S|Cas=4|Gra=1|Neg=A|Sem=G
A	A	Gen=I|Num=S|Cas=4|Gra=1|Neg=A|Sem=S
A	A	Gen=I|Num=S|Cas=4|Gra=1|Neg=A|Sem=m
A	A	Gen=I|Num=S|Cas=4|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=4|Gra=1|Neg=N
A	A	Gen=I|Num=S|Cas=4|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=4|Gra=2|Neg=A
A	A	Gen=I|Num=S|Cas=4|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=4|Gra=2|Neg=N
A	A	Gen=I|Num=S|Cas=4|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=4|Gra=3|Neg=A
A	A	Gen=I|Num=S|Cas=4|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=4|Gra=3|Neg=N
A	A	Gen=I|Num=S|Cas=4|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=5|Gra=1|Neg=A
A	A	Gen=I|Num=S|Cas=5|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=5|Gra=1|Neg=N
A	A	Gen=I|Num=S|Cas=5|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=5|Gra=2|Neg=A
A	A	Gen=I|Num=S|Cas=5|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=5|Gra=2|Neg=N
A	A	Gen=I|Num=S|Cas=5|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=5|Gra=3|Neg=A
A	A	Gen=I|Num=S|Cas=5|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=5|Gra=3|Neg=N
A	A	Gen=I|Num=S|Cas=5|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=6|Gra=1|Neg=A
A	A	Gen=I|Num=S|Cas=6|Gra=1|Neg=A|Sem=G
A	A	Gen=I|Num=S|Cas=6|Gra=1|Neg=A|Sem=S
A	A	Gen=I|Num=S|Cas=6|Gra=1|Neg=A|Sem=m
A	A	Gen=I|Num=S|Cas=6|Gra=1|Neg=A|Sem=o
A	A	Gen=I|Num=S|Cas=6|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=6|Gra=1|Neg=A|Var=7
A	A	Gen=I|Num=S|Cas=6|Gra=1|Neg=N
A	A	Gen=I|Num=S|Cas=6|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=6|Gra=1|Neg=N|Var=7
A	A	Gen=I|Num=S|Cas=6|Gra=2|Neg=A
A	A	Gen=I|Num=S|Cas=6|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=6|Gra=2|Neg=N
A	A	Gen=I|Num=S|Cas=6|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=6|Gra=3|Neg=A
A	A	Gen=I|Num=S|Cas=6|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=6|Gra=3|Neg=N
A	A	Gen=I|Num=S|Cas=6|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=7|Gra=1|Neg=A
A	A	Gen=I|Num=S|Cas=7|Gra=1|Neg=A|Sem=H
A	A	Gen=I|Num=S|Cas=7|Gra=1|Neg=A|Sem=m
A	A	Gen=I|Num=S|Cas=7|Gra=1|Neg=A|Sem=o
A	A	Gen=I|Num=S|Cas=7|Gra=1|Neg=A|Var=2
A	A	Gen=I|Num=S|Cas=7|Gra=1|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=7|Gra=1|Neg=N
A	A	Gen=I|Num=S|Cas=7|Gra=1|Neg=N|Var=2
A	A	Gen=I|Num=S|Cas=7|Gra=1|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=7|Gra=2|Neg=A
A	A	Gen=I|Num=S|Cas=7|Gra=2|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=7|Gra=2|Neg=N
A	A	Gen=I|Num=S|Cas=7|Gra=2|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=7|Gra=3|Neg=A
A	A	Gen=I|Num=S|Cas=7|Gra=3|Neg=A|Var=6
A	A	Gen=I|Num=S|Cas=7|Gra=3|Neg=N
A	A	Gen=I|Num=S|Cas=7|Gra=3|Neg=N|Var=6
A	A	Gen=I|Num=S|Cas=X|Gra=1|Neg=A
A	A	Gen=I|Num=S|Cas=X|Gra=1|Neg=A|Sem=K
A	A	Gen=I|Num=S|Cas=X|Gra=1|Neg=A|Sem=R
A	A	Gen=I|Num=S|Cas=X|Gra=1|Neg=A|Sem=m
A	A	Gen=I|Num=X|Cas=X|Gra=1|Neg=A
A	A	Gen=I|Num=X|Cas=X|Gra=1|Neg=A|Sem=m
A	A	Gen=M|Num=P|Cas=1|Gra=1|Neg=A
A	A	Gen=M|Num=P|Cas=1|Gra=1|Neg=A|Sem=G
A	A	Gen=M|Num=P|Cas=1|Gra=1|Neg=A|Sem=K
A	A	Gen=M|Num=P|Cas=1|Gra=1|Neg=A|Sem=m
A	A	Gen=M|Num=P|Cas=1|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=1|Gra=1|Neg=N
A	A	Gen=M|Num=P|Cas=1|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=1|Gra=2|Neg=A
A	A	Gen=M|Num=P|Cas=1|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=1|Gra=2|Neg=N
A	A	Gen=M|Num=P|Cas=1|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=1|Gra=3|Neg=A
A	A	Gen=M|Num=P|Cas=1|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=1|Gra=3|Neg=N
A	A	Gen=M|Num=P|Cas=1|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=2|Gra=1|Neg=A
A	A	Gen=M|Num=P|Cas=2|Gra=1|Neg=A|Sem=G
A	A	Gen=M|Num=P|Cas=2|Gra=1|Neg=A|Sem=S
A	A	Gen=M|Num=P|Cas=2|Gra=1|Neg=A|Sem=m
A	A	Gen=M|Num=P|Cas=2|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=2|Gra=1|Neg=N
A	A	Gen=M|Num=P|Cas=2|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=2|Gra=2|Neg=A
A	A	Gen=M|Num=P|Cas=2|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=2|Gra=2|Neg=N
A	A	Gen=M|Num=P|Cas=2|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=2|Gra=3|Neg=A
A	A	Gen=M|Num=P|Cas=2|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=2|Gra=3|Neg=N
A	A	Gen=M|Num=P|Cas=2|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=3|Gra=1|Neg=A
A	A	Gen=M|Num=P|Cas=3|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=3|Gra=1|Neg=N
A	A	Gen=M|Num=P|Cas=3|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=3|Gra=2|Neg=A
A	A	Gen=M|Num=P|Cas=3|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=3|Gra=2|Neg=N
A	A	Gen=M|Num=P|Cas=3|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=3|Gra=3|Neg=A
A	A	Gen=M|Num=P|Cas=3|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=3|Gra=3|Neg=N
A	A	Gen=M|Num=P|Cas=3|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=4|Gra=1|Neg=A
A	A	Gen=M|Num=P|Cas=4|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=4|Gra=1|Neg=N
A	A	Gen=M|Num=P|Cas=4|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=4|Gra=2|Neg=A
A	A	Gen=M|Num=P|Cas=4|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=4|Gra=2|Neg=N
A	A	Gen=M|Num=P|Cas=4|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=4|Gra=3|Neg=A
A	A	Gen=M|Num=P|Cas=4|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=4|Gra=3|Neg=N
A	A	Gen=M|Num=P|Cas=4|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=5|Gra=1|Neg=A
A	A	Gen=M|Num=P|Cas=5|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=5|Gra=1|Neg=N
A	A	Gen=M|Num=P|Cas=5|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=5|Gra=2|Neg=A
A	A	Gen=M|Num=P|Cas=5|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=5|Gra=2|Neg=N
A	A	Gen=M|Num=P|Cas=5|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=5|Gra=3|Neg=A
A	A	Gen=M|Num=P|Cas=5|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=5|Gra=3|Neg=N
A	A	Gen=M|Num=P|Cas=5|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=6|Gra=1|Neg=A
A	A	Gen=M|Num=P|Cas=6|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=6|Gra=1|Neg=N
A	A	Gen=M|Num=P|Cas=6|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=6|Gra=2|Neg=A
A	A	Gen=M|Num=P|Cas=6|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=6|Gra=2|Neg=N
A	A	Gen=M|Num=P|Cas=6|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=6|Gra=3|Neg=A
A	A	Gen=M|Num=P|Cas=6|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=6|Gra=3|Neg=N
A	A	Gen=M|Num=P|Cas=6|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=7|Gra=1|Neg=A
A	A	Gen=M|Num=P|Cas=7|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=7|Gra=1|Neg=A|Var=7
A	A	Gen=M|Num=P|Cas=7|Gra=1|Neg=N
A	A	Gen=M|Num=P|Cas=7|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=7|Gra=1|Neg=N|Var=7
A	A	Gen=M|Num=P|Cas=7|Gra=2|Neg=A
A	A	Gen=M|Num=P|Cas=7|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=7|Gra=2|Neg=A|Var=7
A	A	Gen=M|Num=P|Cas=7|Gra=2|Neg=N
A	A	Gen=M|Num=P|Cas=7|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=7|Gra=2|Neg=N|Var=7
A	A	Gen=M|Num=P|Cas=7|Gra=3|Neg=A
A	A	Gen=M|Num=P|Cas=7|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=P|Cas=7|Gra=3|Neg=A|Var=7
A	A	Gen=M|Num=P|Cas=7|Gra=3|Neg=N
A	A	Gen=M|Num=P|Cas=7|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=P|Cas=7|Gra=3|Neg=N|Var=7
A	A	Gen=M|Num=S|Cas=1|Gra=1|Neg=A
A	A	Gen=M|Num=S|Cas=1|Gra=1|Neg=A|Sem=G
A	A	Gen=M|Num=S|Cas=1|Gra=1|Neg=A|Sem=S
A	A	Gen=M|Num=S|Cas=1|Gra=1|Neg=A|Sem=U
A	A	Gen=M|Num=S|Cas=1|Gra=1|Neg=A|Sem=m
A	A	Gen=M|Num=S|Cas=1|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=1|Gra=1|Neg=N
A	A	Gen=M|Num=S|Cas=1|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=1|Gra=2|Neg=A
A	A	Gen=M|Num=S|Cas=1|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=1|Gra=2|Neg=N
A	A	Gen=M|Num=S|Cas=1|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=1|Gra=3|Neg=A
A	A	Gen=M|Num=S|Cas=1|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=1|Gra=3|Neg=N
A	A	Gen=M|Num=S|Cas=1|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=2|Gra=1|Neg=A
A	A	Gen=M|Num=S|Cas=2|Gra=1|Neg=A|Sem=S
A	A	Gen=M|Num=S|Cas=2|Gra=1|Neg=A|Sem=m
A	A	Gen=M|Num=S|Cas=2|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=2|Gra=1|Neg=N
A	A	Gen=M|Num=S|Cas=2|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=2|Gra=2|Neg=A
A	A	Gen=M|Num=S|Cas=2|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=2|Gra=2|Neg=N
A	A	Gen=M|Num=S|Cas=2|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=2|Gra=3|Neg=A
A	A	Gen=M|Num=S|Cas=2|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=2|Gra=3|Neg=N
A	A	Gen=M|Num=S|Cas=2|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=3|Gra=1|Neg=A
A	A	Gen=M|Num=S|Cas=3|Gra=1|Neg=A|Sem=S
A	A	Gen=M|Num=S|Cas=3|Gra=1|Neg=A|Sem=m
A	A	Gen=M|Num=S|Cas=3|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=3|Gra=1|Neg=N
A	A	Gen=M|Num=S|Cas=3|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=3|Gra=2|Neg=A
A	A	Gen=M|Num=S|Cas=3|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=3|Gra=2|Neg=N
A	A	Gen=M|Num=S|Cas=3|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=3|Gra=3|Neg=A
A	A	Gen=M|Num=S|Cas=3|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=3|Gra=3|Neg=N
A	A	Gen=M|Num=S|Cas=3|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=4|Gra=1|Neg=A
A	A	Gen=M|Num=S|Cas=4|Gra=1|Neg=A|Sem=S
A	A	Gen=M|Num=S|Cas=4|Gra=1|Neg=A|Sem=m
A	A	Gen=M|Num=S|Cas=4|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=4|Gra=1|Neg=N
A	A	Gen=M|Num=S|Cas=4|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=4|Gra=2|Neg=A
A	A	Gen=M|Num=S|Cas=4|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=4|Gra=2|Neg=N
A	A	Gen=M|Num=S|Cas=4|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=4|Gra=3|Neg=A
A	A	Gen=M|Num=S|Cas=4|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=4|Gra=3|Neg=N
A	A	Gen=M|Num=S|Cas=4|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=5|Gra=1|Neg=A
A	A	Gen=M|Num=S|Cas=5|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=5|Gra=1|Neg=N
A	A	Gen=M|Num=S|Cas=5|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=5|Gra=2|Neg=A
A	A	Gen=M|Num=S|Cas=5|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=5|Gra=2|Neg=N
A	A	Gen=M|Num=S|Cas=5|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=5|Gra=3|Neg=A
A	A	Gen=M|Num=S|Cas=5|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=5|Gra=3|Neg=N
A	A	Gen=M|Num=S|Cas=5|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=6|Gra=1|Neg=A
A	A	Gen=M|Num=S|Cas=6|Gra=1|Neg=A|Sem=S
A	A	Gen=M|Num=S|Cas=6|Gra=1|Neg=A|Sem=m
A	A	Gen=M|Num=S|Cas=6|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=6|Gra=1|Neg=A|Var=7
A	A	Gen=M|Num=S|Cas=6|Gra=1|Neg=N
A	A	Gen=M|Num=S|Cas=6|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=6|Gra=1|Neg=N|Var=7
A	A	Gen=M|Num=S|Cas=6|Gra=2|Neg=A
A	A	Gen=M|Num=S|Cas=6|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=6|Gra=2|Neg=N
A	A	Gen=M|Num=S|Cas=6|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=6|Gra=3|Neg=A
A	A	Gen=M|Num=S|Cas=6|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=6|Gra=3|Neg=N
A	A	Gen=M|Num=S|Cas=6|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=7|Gra=1|Neg=A
A	A	Gen=M|Num=S|Cas=7|Gra=1|Neg=A|Sem=S
A	A	Gen=M|Num=S|Cas=7|Gra=1|Neg=A|Sem=m
A	A	Gen=M|Num=S|Cas=7|Gra=1|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=7|Gra=1|Neg=N
A	A	Gen=M|Num=S|Cas=7|Gra=1|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=7|Gra=2|Neg=A
A	A	Gen=M|Num=S|Cas=7|Gra=2|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=7|Gra=2|Neg=N
A	A	Gen=M|Num=S|Cas=7|Gra=2|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=7|Gra=3|Neg=A
A	A	Gen=M|Num=S|Cas=7|Gra=3|Neg=A|Var=6
A	A	Gen=M|Num=S|Cas=7|Gra=3|Neg=N
A	A	Gen=M|Num=S|Cas=7|Gra=3|Neg=N|Var=6
A	A	Gen=M|Num=S|Cas=X|Gra=1|Neg=A
A	A	Gen=N|Num=D|Cas=7|Gra=1|Neg=A
A	A	Gen=N|Num=D|Cas=7|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=D|Cas=7|Gra=1|Neg=N
A	A	Gen=N|Num=D|Cas=7|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=D|Cas=7|Gra=2|Neg=A
A	A	Gen=N|Num=D|Cas=7|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=D|Cas=7|Gra=2|Neg=N
A	A	Gen=N|Num=D|Cas=7|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=D|Cas=7|Gra=3|Neg=A
A	A	Gen=N|Num=D|Cas=7|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=D|Cas=7|Gra=3|Neg=N
A	A	Gen=N|Num=D|Cas=7|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=1|Gra=1|Neg=A
A	A	Gen=N|Num=P|Cas=1|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=1|Gra=1|Neg=N
A	A	Gen=N|Num=P|Cas=1|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=1|Gra=2|Neg=A
A	A	Gen=N|Num=P|Cas=1|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=1|Gra=2|Neg=N
A	A	Gen=N|Num=P|Cas=1|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=1|Gra=3|Neg=A
A	A	Gen=N|Num=P|Cas=1|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=1|Gra=3|Neg=N
A	A	Gen=N|Num=P|Cas=1|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=2|Gra=1|Neg=A
A	A	Gen=N|Num=P|Cas=2|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=2|Gra=1|Neg=N
A	A	Gen=N|Num=P|Cas=2|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=2|Gra=2|Neg=A
A	A	Gen=N|Num=P|Cas=2|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=2|Gra=2|Neg=N
A	A	Gen=N|Num=P|Cas=2|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=2|Gra=3|Neg=A
A	A	Gen=N|Num=P|Cas=2|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=2|Gra=3|Neg=N
A	A	Gen=N|Num=P|Cas=2|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=3|Gra=1|Neg=A
A	A	Gen=N|Num=P|Cas=3|Gra=1|Neg=A|Sem=m
A	A	Gen=N|Num=P|Cas=3|Gra=1|Neg=A|Sem=o
A	A	Gen=N|Num=P|Cas=3|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=3|Gra=1|Neg=N
A	A	Gen=N|Num=P|Cas=3|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=3|Gra=2|Neg=A
A	A	Gen=N|Num=P|Cas=3|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=3|Gra=2|Neg=N
A	A	Gen=N|Num=P|Cas=3|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=3|Gra=3|Neg=A
A	A	Gen=N|Num=P|Cas=3|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=3|Gra=3|Neg=N
A	A	Gen=N|Num=P|Cas=3|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=4|Gra=1|Neg=A
A	A	Gen=N|Num=P|Cas=4|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=4|Gra=1|Neg=N
A	A	Gen=N|Num=P|Cas=4|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=4|Gra=2|Neg=A
A	A	Gen=N|Num=P|Cas=4|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=4|Gra=2|Neg=N
A	A	Gen=N|Num=P|Cas=4|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=4|Gra=3|Neg=A
A	A	Gen=N|Num=P|Cas=4|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=4|Gra=3|Neg=N
A	A	Gen=N|Num=P|Cas=4|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=5|Gra=1|Neg=A
A	A	Gen=N|Num=P|Cas=5|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=5|Gra=1|Neg=N
A	A	Gen=N|Num=P|Cas=5|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=5|Gra=2|Neg=A
A	A	Gen=N|Num=P|Cas=5|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=5|Gra=2|Neg=N
A	A	Gen=N|Num=P|Cas=5|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=5|Gra=3|Neg=A
A	A	Gen=N|Num=P|Cas=5|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=5|Gra=3|Neg=N
A	A	Gen=N|Num=P|Cas=5|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=6|Gra=1|Neg=A
A	A	Gen=N|Num=P|Cas=6|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=6|Gra=1|Neg=N
A	A	Gen=N|Num=P|Cas=6|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=6|Gra=2|Neg=A
A	A	Gen=N|Num=P|Cas=6|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=6|Gra=2|Neg=N
A	A	Gen=N|Num=P|Cas=6|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=6|Gra=3|Neg=A
A	A	Gen=N|Num=P|Cas=6|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=6|Gra=3|Neg=N
A	A	Gen=N|Num=P|Cas=6|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=7|Gra=1|Neg=A
A	A	Gen=N|Num=P|Cas=7|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=7|Gra=1|Neg=A|Var=7
A	A	Gen=N|Num=P|Cas=7|Gra=1|Neg=N
A	A	Gen=N|Num=P|Cas=7|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=7|Gra=1|Neg=N|Var=7
A	A	Gen=N|Num=P|Cas=7|Gra=2|Neg=A
A	A	Gen=N|Num=P|Cas=7|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=7|Gra=2|Neg=A|Var=7
A	A	Gen=N|Num=P|Cas=7|Gra=2|Neg=N
A	A	Gen=N|Num=P|Cas=7|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=7|Gra=2|Neg=N|Var=7
A	A	Gen=N|Num=P|Cas=7|Gra=3|Neg=A
A	A	Gen=N|Num=P|Cas=7|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=P|Cas=7|Gra=3|Neg=A|Var=7
A	A	Gen=N|Num=P|Cas=7|Gra=3|Neg=N
A	A	Gen=N|Num=P|Cas=7|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=P|Cas=7|Gra=3|Neg=N|Var=7
A	A	Gen=N|Num=S|Cas=1|Gra=1|Neg=A
A	A	Gen=N|Num=S|Cas=1|Gra=1|Neg=A|Sem=G
A	A	Gen=N|Num=S|Cas=1|Gra=1|Neg=A|Sem=K
A	A	Gen=N|Num=S|Cas=1|Gra=1|Neg=A|Sem=R
A	A	Gen=N|Num=S|Cas=1|Gra=1|Neg=A|Sem=S
A	A	Gen=N|Num=S|Cas=1|Gra=1|Neg=A|Sem=m
A	A	Gen=N|Num=S|Cas=1|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=1|Gra=1|Neg=N
A	A	Gen=N|Num=S|Cas=1|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=1|Gra=2|Neg=A
A	A	Gen=N|Num=S|Cas=1|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=1|Gra=2|Neg=N
A	A	Gen=N|Num=S|Cas=1|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=1|Gra=3|Neg=A
A	A	Gen=N|Num=S|Cas=1|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=1|Gra=3|Neg=N
A	A	Gen=N|Num=S|Cas=1|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=2|Gra=1|Neg=A
A	A	Gen=N|Num=S|Cas=2|Gra=1|Neg=A|Sem=G
A	A	Gen=N|Num=S|Cas=2|Gra=1|Neg=A|Sem=S
A	A	Gen=N|Num=S|Cas=2|Gra=1|Neg=A|Sem=m
A	A	Gen=N|Num=S|Cas=2|Gra=1|Neg=A|Sem=o
A	A	Gen=N|Num=S|Cas=2|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=2|Gra=1|Neg=N
A	A	Gen=N|Num=S|Cas=2|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=2|Gra=2|Neg=A
A	A	Gen=N|Num=S|Cas=2|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=2|Gra=2|Neg=N
A	A	Gen=N|Num=S|Cas=2|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=2|Gra=3|Neg=A
A	A	Gen=N|Num=S|Cas=2|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=2|Gra=3|Neg=N
A	A	Gen=N|Num=S|Cas=2|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=3|Gra=1|Neg=A
A	A	Gen=N|Num=S|Cas=3|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=3|Gra=1|Neg=N
A	A	Gen=N|Num=S|Cas=3|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=3|Gra=2|Neg=A
A	A	Gen=N|Num=S|Cas=3|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=3|Gra=2|Neg=N
A	A	Gen=N|Num=S|Cas=3|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=3|Gra=3|Neg=A
A	A	Gen=N|Num=S|Cas=3|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=3|Gra=3|Neg=N
A	A	Gen=N|Num=S|Cas=3|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=4|Gra=1|Neg=A
A	A	Gen=N|Num=S|Cas=4|Gra=1|Neg=A|Sem=G
A	A	Gen=N|Num=S|Cas=4|Gra=1|Neg=A|Sem=m
A	A	Gen=N|Num=S|Cas=4|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=4|Gra=1|Neg=N
A	A	Gen=N|Num=S|Cas=4|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=4|Gra=2|Neg=A
A	A	Gen=N|Num=S|Cas=4|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=4|Gra=2|Neg=N
A	A	Gen=N|Num=S|Cas=4|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=4|Gra=3|Neg=A
A	A	Gen=N|Num=S|Cas=4|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=4|Gra=3|Neg=N
A	A	Gen=N|Num=S|Cas=4|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=5|Gra=1|Neg=A
A	A	Gen=N|Num=S|Cas=5|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=5|Gra=1|Neg=N
A	A	Gen=N|Num=S|Cas=5|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=5|Gra=2|Neg=A
A	A	Gen=N|Num=S|Cas=5|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=5|Gra=2|Neg=N
A	A	Gen=N|Num=S|Cas=5|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=5|Gra=3|Neg=A
A	A	Gen=N|Num=S|Cas=5|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=5|Gra=3|Neg=N
A	A	Gen=N|Num=S|Cas=5|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=6|Gra=1|Neg=A
A	A	Gen=N|Num=S|Cas=6|Gra=1|Neg=A|Sem=G
A	A	Gen=N|Num=S|Cas=6|Gra=1|Neg=A|Sem=S
A	A	Gen=N|Num=S|Cas=6|Gra=1|Neg=A|Sem=m
A	A	Gen=N|Num=S|Cas=6|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=6|Gra=1|Neg=A|Var=7
A	A	Gen=N|Num=S|Cas=6|Gra=1|Neg=N
A	A	Gen=N|Num=S|Cas=6|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=6|Gra=1|Neg=N|Var=7
A	A	Gen=N|Num=S|Cas=6|Gra=2|Neg=A
A	A	Gen=N|Num=S|Cas=6|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=6|Gra=2|Neg=N
A	A	Gen=N|Num=S|Cas=6|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=6|Gra=3|Neg=A
A	A	Gen=N|Num=S|Cas=6|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=6|Gra=3|Neg=N
A	A	Gen=N|Num=S|Cas=6|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=7|Gra=1|Neg=A
A	A	Gen=N|Num=S|Cas=7|Gra=1|Neg=A|Sem=G
A	A	Gen=N|Num=S|Cas=7|Gra=1|Neg=A|Sem=S
A	A	Gen=N|Num=S|Cas=7|Gra=1|Neg=A|Sem=m
A	A	Gen=N|Num=S|Cas=7|Gra=1|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=7|Gra=1|Neg=A|Var=8
A	A	Gen=N|Num=S|Cas=7|Gra=1|Neg=N
A	A	Gen=N|Num=S|Cas=7|Gra=1|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=7|Gra=2|Neg=A
A	A	Gen=N|Num=S|Cas=7|Gra=2|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=7|Gra=2|Neg=N
A	A	Gen=N|Num=S|Cas=7|Gra=2|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=7|Gra=3|Neg=A
A	A	Gen=N|Num=S|Cas=7|Gra=3|Neg=A|Var=6
A	A	Gen=N|Num=S|Cas=7|Gra=3|Neg=N
A	A	Gen=N|Num=S|Cas=7|Gra=3|Neg=N|Var=6
A	A	Gen=N|Num=S|Cas=X|Gra=1|Neg=A
A	A	Gen=N|Num=S|Cas=X|Gra=1|Neg=A|Sem=G
A	A	Gen=N|Num=S|Cas=X|Gra=1|Neg=A|Sem=m
A	A	Gen=X|Num=P|Cas=X|Gra=1|Neg=A
A	A	Gen=X|Num=P|Cas=X|Gra=1|Neg=N
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=E
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=G
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=K
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=R
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=S
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=Y
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=m
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Sem=w
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=1
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=8
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=8|Sem=G
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=8|Sem=K
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=8|Sem=R
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=8|Sem=m
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=A|Var=9
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=N
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=N|Var=8
A	A	Gen=X|Num=X|Cas=X|Gra=1|Neg=N|Var=9
A	A	Gen=X|Num=X|Cas=X|Gra=2|Neg=A|Var=8
A	A	Gen=X|Num=X|Cas=X|Gra=3|Neg=A|Var=8
A	C	Gen=F|Num=S|Cas=4|Neg=A
A	C	Gen=F|Num=S|Cas=4|Neg=N
A	C	Gen=M|Num=P|Neg=A
A	C	Gen=M|Num=P|Neg=N
A	C	Gen=N|Num=S|Neg=A
A	C	Gen=N|Num=S|Neg=N
A	C	Gen=Q|Num=W|Neg=A
A	C	Gen=Q|Num=W|Neg=N
A	C	Gen=T|Num=P|Neg=A
A	C	Gen=T|Num=P|Neg=N
A	C	Gen=Y|Num=S|Neg=A
A	C	Gen=Y|Num=S|Neg=N
A	G	Gen=F|Num=D|Cas=7|Neg=A
A	G	Gen=F|Num=D|Cas=7|Neg=A|Var=6
A	G	Gen=F|Num=D|Cas=7|Neg=N
A	G	Gen=F|Num=D|Cas=7|Neg=N|Var=6
A	G	Gen=F|Num=P|Cas=1|Neg=A
A	G	Gen=F|Num=P|Cas=1|Neg=A|Var=6
A	G	Gen=F|Num=P|Cas=1|Neg=N
A	G	Gen=F|Num=P|Cas=1|Neg=N|Var=6
A	G	Gen=F|Num=P|Cas=2|Neg=A
A	G	Gen=F|Num=P|Cas=2|Neg=A|Var=6
A	G	Gen=F|Num=P|Cas=2|Neg=N
A	G	Gen=F|Num=P|Cas=2|Neg=N|Var=6
A	G	Gen=F|Num=P|Cas=3|Neg=A
A	G	Gen=F|Num=P|Cas=3|Neg=A|Var=6
A	G	Gen=F|Num=P|Cas=3|Neg=N
A	G	Gen=F|Num=P|Cas=3|Neg=N|Var=6
A	G	Gen=F|Num=P|Cas=4|Neg=A
A	G	Gen=F|Num=P|Cas=4|Neg=A|Var=6
A	G	Gen=F|Num=P|Cas=4|Neg=N
A	G	Gen=F|Num=P|Cas=4|Neg=N|Var=6
A	G	Gen=F|Num=P|Cas=5|Neg=A
A	G	Gen=F|Num=P|Cas=5|Neg=A|Var=6
A	G	Gen=F|Num=P|Cas=5|Neg=N
A	G	Gen=F|Num=P|Cas=5|Neg=N|Var=6
A	G	Gen=F|Num=P|Cas=6|Neg=A
A	G	Gen=F|Num=P|Cas=6|Neg=A|Var=6
A	G	Gen=F|Num=P|Cas=6|Neg=N
A	G	Gen=F|Num=P|Cas=6|Neg=N|Var=6
A	G	Gen=F|Num=P|Cas=7|Neg=A
A	G	Gen=F|Num=P|Cas=7|Neg=A|Var=6
A	G	Gen=F|Num=P|Cas=7|Neg=N
A	G	Gen=F|Num=P|Cas=7|Neg=N|Var=6
A	G	Gen=F|Num=S|Cas=1|Neg=A
A	G	Gen=F|Num=S|Cas=1|Neg=A|Var=6
A	G	Gen=F|Num=S|Cas=1|Neg=N
A	G	Gen=F|Num=S|Cas=1|Neg=N|Var=6
A	G	Gen=F|Num=S|Cas=2|Neg=A
A	G	Gen=F|Num=S|Cas=2|Neg=A|Var=6
A	G	Gen=F|Num=S|Cas=2|Neg=N
A	G	Gen=F|Num=S|Cas=2|Neg=N|Var=6
A	G	Gen=F|Num=S|Cas=3|Neg=A
A	G	Gen=F|Num=S|Cas=3|Neg=A|Var=6
A	G	Gen=F|Num=S|Cas=3|Neg=N
A	G	Gen=F|Num=S|Cas=3|Neg=N|Var=6
A	G	Gen=F|Num=S|Cas=4|Neg=A
A	G	Gen=F|Num=S|Cas=4|Neg=A|Var=6
A	G	Gen=F|Num=S|Cas=4|Neg=N
A	G	Gen=F|Num=S|Cas=4|Neg=N|Var=6
A	G	Gen=F|Num=S|Cas=5|Neg=A
A	G	Gen=F|Num=S|Cas=5|Neg=A|Var=6
A	G	Gen=F|Num=S|Cas=5|Neg=N
A	G	Gen=F|Num=S|Cas=5|Neg=N|Var=6
A	G	Gen=F|Num=S|Cas=6|Neg=A
A	G	Gen=F|Num=S|Cas=6|Neg=A|Var=6
A	G	Gen=F|Num=S|Cas=6|Neg=N
A	G	Gen=F|Num=S|Cas=6|Neg=N|Var=6
A	G	Gen=F|Num=S|Cas=7|Neg=A
A	G	Gen=F|Num=S|Cas=7|Neg=A|Var=6
A	G	Gen=F|Num=S|Cas=7|Neg=N
A	G	Gen=F|Num=S|Cas=7|Neg=N|Var=6
A	G	Gen=I|Num=P|Cas=1|Neg=A
A	G	Gen=I|Num=P|Cas=1|Neg=A|Var=6
A	G	Gen=I|Num=P|Cas=1|Neg=N
A	G	Gen=I|Num=P|Cas=1|Neg=N|Var=6
A	G	Gen=I|Num=P|Cas=2|Neg=A
A	G	Gen=I|Num=P|Cas=2|Neg=A|Var=6
A	G	Gen=I|Num=P|Cas=2|Neg=N
A	G	Gen=I|Num=P|Cas=2|Neg=N|Var=6
A	G	Gen=I|Num=P|Cas=3|Neg=A
A	G	Gen=I|Num=P|Cas=3|Neg=A|Var=6
A	G	Gen=I|Num=P|Cas=3|Neg=N
A	G	Gen=I|Num=P|Cas=3|Neg=N|Var=6
A	G	Gen=I|Num=P|Cas=4|Neg=A
A	G	Gen=I|Num=P|Cas=4|Neg=A|Var=6
A	G	Gen=I|Num=P|Cas=4|Neg=N
A	G	Gen=I|Num=P|Cas=4|Neg=N|Var=6
A	G	Gen=I|Num=P|Cas=5|Neg=A
A	G	Gen=I|Num=P|Cas=5|Neg=A|Var=6
A	G	Gen=I|Num=P|Cas=5|Neg=N
A	G	Gen=I|Num=P|Cas=5|Neg=N|Var=6
A	G	Gen=I|Num=P|Cas=6|Neg=A
A	G	Gen=I|Num=P|Cas=6|Neg=A|Var=6
A	G	Gen=I|Num=P|Cas=6|Neg=N
A	G	Gen=I|Num=P|Cas=6|Neg=N|Var=6
A	G	Gen=I|Num=P|Cas=7|Neg=A
A	G	Gen=I|Num=P|Cas=7|Neg=A|Var=6
A	G	Gen=I|Num=P|Cas=7|Neg=N
A	G	Gen=I|Num=P|Cas=7|Neg=N|Var=6
A	G	Gen=I|Num=S|Cas=1|Neg=A
A	G	Gen=I|Num=S|Cas=1|Neg=A|Var=6
A	G	Gen=I|Num=S|Cas=1|Neg=N
A	G	Gen=I|Num=S|Cas=1|Neg=N|Var=6
A	G	Gen=I|Num=S|Cas=2|Neg=A
A	G	Gen=I|Num=S|Cas=2|Neg=A|Var=6
A	G	Gen=I|Num=S|Cas=2|Neg=N
A	G	Gen=I|Num=S|Cas=2|Neg=N|Var=6
A	G	Gen=I|Num=S|Cas=3|Neg=A
A	G	Gen=I|Num=S|Cas=3|Neg=A|Var=6
A	G	Gen=I|Num=S|Cas=3|Neg=N
A	G	Gen=I|Num=S|Cas=3|Neg=N|Var=6
A	G	Gen=I|Num=S|Cas=4|Neg=A
A	G	Gen=I|Num=S|Cas=4|Neg=A|Var=6
A	G	Gen=I|Num=S|Cas=4|Neg=N
A	G	Gen=I|Num=S|Cas=4|Neg=N|Var=6
A	G	Gen=I|Num=S|Cas=5|Neg=A
A	G	Gen=I|Num=S|Cas=5|Neg=A|Var=6
A	G	Gen=I|Num=S|Cas=5|Neg=N
A	G	Gen=I|Num=S|Cas=5|Neg=N|Var=6
A	G	Gen=I|Num=S|Cas=6|Neg=A
A	G	Gen=I|Num=S|Cas=6|Neg=A|Var=6
A	G	Gen=I|Num=S|Cas=6|Neg=A|Var=7
A	G	Gen=I|Num=S|Cas=6|Neg=N
A	G	Gen=I|Num=S|Cas=6|Neg=N|Var=6
A	G	Gen=I|Num=S|Cas=6|Neg=N|Var=7
A	G	Gen=I|Num=S|Cas=7|Neg=A
A	G	Gen=I|Num=S|Cas=7|Neg=A|Var=6
A	G	Gen=I|Num=S|Cas=7|Neg=N
A	G	Gen=I|Num=S|Cas=7|Neg=N|Var=6
A	G	Gen=M|Num=P|Cas=1|Neg=A
A	G	Gen=M|Num=P|Cas=1|Neg=A|Var=6
A	G	Gen=M|Num=P|Cas=1|Neg=N
A	G	Gen=M|Num=P|Cas=1|Neg=N|Var=6
A	G	Gen=M|Num=P|Cas=2|Neg=A
A	G	Gen=M|Num=P|Cas=2|Neg=A|Var=6
A	G	Gen=M|Num=P|Cas=2|Neg=N
A	G	Gen=M|Num=P|Cas=2|Neg=N|Var=6
A	G	Gen=M|Num=P|Cas=3|Neg=A
A	G	Gen=M|Num=P|Cas=3|Neg=A|Var=6
A	G	Gen=M|Num=P|Cas=3|Neg=N
A	G	Gen=M|Num=P|Cas=3|Neg=N|Var=6
A	G	Gen=M|Num=P|Cas=4|Neg=A
A	G	Gen=M|Num=P|Cas=4|Neg=A|Var=6
A	G	Gen=M|Num=P|Cas=4|Neg=N
A	G	Gen=M|Num=P|Cas=4|Neg=N|Var=6
A	G	Gen=M|Num=P|Cas=5|Neg=A
A	G	Gen=M|Num=P|Cas=5|Neg=A|Var=6
A	G	Gen=M|Num=P|Cas=5|Neg=N
A	G	Gen=M|Num=P|Cas=5|Neg=N|Var=6
A	G	Gen=M|Num=P|Cas=6|Neg=A
A	G	Gen=M|Num=P|Cas=6|Neg=A|Var=6
A	G	Gen=M|Num=P|Cas=6|Neg=N
A	G	Gen=M|Num=P|Cas=6|Neg=N|Var=6
A	G	Gen=M|Num=P|Cas=7|Neg=A
A	G	Gen=M|Num=P|Cas=7|Neg=A|Var=6
A	G	Gen=M|Num=P|Cas=7|Neg=N
A	G	Gen=M|Num=P|Cas=7|Neg=N|Var=6
A	G	Gen=M|Num=S|Cas=1|Neg=A
A	G	Gen=M|Num=S|Cas=1|Neg=A|Var=6
A	G	Gen=M|Num=S|Cas=1|Neg=N
A	G	Gen=M|Num=S|Cas=1|Neg=N|Var=6
A	G	Gen=M|Num=S|Cas=2|Neg=A
A	G	Gen=M|Num=S|Cas=2|Neg=A|Var=6
A	G	Gen=M|Num=S|Cas=2|Neg=N
A	G	Gen=M|Num=S|Cas=2|Neg=N|Var=6
A	G	Gen=M|Num=S|Cas=3|Neg=A
A	G	Gen=M|Num=S|Cas=3|Neg=A|Var=6
A	G	Gen=M|Num=S|Cas=3|Neg=N
A	G	Gen=M|Num=S|Cas=3|Neg=N|Var=6
A	G	Gen=M|Num=S|Cas=4|Neg=A
A	G	Gen=M|Num=S|Cas=4|Neg=A|Var=6
A	G	Gen=M|Num=S|Cas=4|Neg=N
A	G	Gen=M|Num=S|Cas=4|Neg=N|Var=6
A	G	Gen=M|Num=S|Cas=5|Neg=A
A	G	Gen=M|Num=S|Cas=5|Neg=A|Var=6
A	G	Gen=M|Num=S|Cas=5|Neg=N
A	G	Gen=M|Num=S|Cas=5|Neg=N|Var=6
A	G	Gen=M|Num=S|Cas=6|Neg=A
A	G	Gen=M|Num=S|Cas=6|Neg=A|Var=6
A	G	Gen=M|Num=S|Cas=6|Neg=A|Var=7
A	G	Gen=M|Num=S|Cas=6|Neg=N
A	G	Gen=M|Num=S|Cas=6|Neg=N|Var=6
A	G	Gen=M|Num=S|Cas=6|Neg=N|Var=7
A	G	Gen=M|Num=S|Cas=7|Neg=A
A	G	Gen=M|Num=S|Cas=7|Neg=A|Var=6
A	G	Gen=M|Num=S|Cas=7|Neg=N
A	G	Gen=M|Num=S|Cas=7|Neg=N|Var=6
A	G	Gen=N|Num=D|Cas=7|Neg=A
A	G	Gen=N|Num=D|Cas=7|Neg=A|Var=6
A	G	Gen=N|Num=D|Cas=7|Neg=N
A	G	Gen=N|Num=D|Cas=7|Neg=N|Var=6
A	G	Gen=N|Num=P|Cas=1|Neg=A
A	G	Gen=N|Num=P|Cas=1|Neg=A|Var=6
A	G	Gen=N|Num=P|Cas=1|Neg=N
A	G	Gen=N|Num=P|Cas=1|Neg=N|Var=6
A	G	Gen=N|Num=P|Cas=2|Neg=A
A	G	Gen=N|Num=P|Cas=2|Neg=A|Var=6
A	G	Gen=N|Num=P|Cas=2|Neg=N
A	G	Gen=N|Num=P|Cas=2|Neg=N|Var=6
A	G	Gen=N|Num=P|Cas=3|Neg=A
A	G	Gen=N|Num=P|Cas=3|Neg=A|Var=6
A	G	Gen=N|Num=P|Cas=3|Neg=N
A	G	Gen=N|Num=P|Cas=3|Neg=N|Var=6
A	G	Gen=N|Num=P|Cas=4|Neg=A
A	G	Gen=N|Num=P|Cas=4|Neg=A|Var=6
A	G	Gen=N|Num=P|Cas=4|Neg=N
A	G	Gen=N|Num=P|Cas=4|Neg=N|Var=6
A	G	Gen=N|Num=P|Cas=5|Neg=A
A	G	Gen=N|Num=P|Cas=5|Neg=A|Var=6
A	G	Gen=N|Num=P|Cas=5|Neg=N
A	G	Gen=N|Num=P|Cas=5|Neg=N|Var=6
A	G	Gen=N|Num=P|Cas=6|Neg=A
A	G	Gen=N|Num=P|Cas=6|Neg=A|Var=6
A	G	Gen=N|Num=P|Cas=6|Neg=N
A	G	Gen=N|Num=P|Cas=6|Neg=N|Var=6
A	G	Gen=N|Num=P|Cas=7|Neg=A
A	G	Gen=N|Num=P|Cas=7|Neg=A|Var=6
A	G	Gen=N|Num=P|Cas=7|Neg=N
A	G	Gen=N|Num=P|Cas=7|Neg=N|Var=6
A	G	Gen=N|Num=S|Cas=1|Neg=A
A	G	Gen=N|Num=S|Cas=1|Neg=A|Var=6
A	G	Gen=N|Num=S|Cas=1|Neg=N
A	G	Gen=N|Num=S|Cas=1|Neg=N|Var=6
A	G	Gen=N|Num=S|Cas=2|Neg=A
A	G	Gen=N|Num=S|Cas=2|Neg=A|Var=6
A	G	Gen=N|Num=S|Cas=2|Neg=N
A	G	Gen=N|Num=S|Cas=2|Neg=N|Var=6
A	G	Gen=N|Num=S|Cas=3|Neg=A
A	G	Gen=N|Num=S|Cas=3|Neg=A|Var=6
A	G	Gen=N|Num=S|Cas=3|Neg=N
A	G	Gen=N|Num=S|Cas=3|Neg=N|Var=6
A	G	Gen=N|Num=S|Cas=4|Neg=A
A	G	Gen=N|Num=S|Cas=4|Neg=A|Var=6
A	G	Gen=N|Num=S|Cas=4|Neg=N
A	G	Gen=N|Num=S|Cas=4|Neg=N|Var=6
A	G	Gen=N|Num=S|Cas=5|Neg=A
A	G	Gen=N|Num=S|Cas=5|Neg=A|Var=6
A	G	Gen=N|Num=S|Cas=5|Neg=N
A	G	Gen=N|Num=S|Cas=5|Neg=N|Var=6
A	G	Gen=N|Num=S|Cas=6|Neg=A
A	G	Gen=N|Num=S|Cas=6|Neg=A|Var=6
A	G	Gen=N|Num=S|Cas=6|Neg=A|Var=7
A	G	Gen=N|Num=S|Cas=6|Neg=N
A	G	Gen=N|Num=S|Cas=6|Neg=N|Var=6
A	G	Gen=N|Num=S|Cas=6|Neg=N|Var=7
A	G	Gen=N|Num=S|Cas=7|Neg=A
A	G	Gen=N|Num=S|Cas=7|Neg=A|Var=6
A	G	Gen=N|Num=S|Cas=7|Neg=N
A	G	Gen=N|Num=S|Cas=7|Neg=N|Var=6
A	M	Gen=F|Num=D|Cas=7|Neg=A
A	M	Gen=F|Num=D|Cas=7|Neg=A|Var=6
A	M	Gen=F|Num=D|Cas=7|Neg=N
A	M	Gen=F|Num=D|Cas=7|Neg=N|Var=6
A	M	Gen=F|Num=P|Cas=1|Neg=A
A	M	Gen=F|Num=P|Cas=1|Neg=A|Var=6
A	M	Gen=F|Num=P|Cas=1|Neg=N
A	M	Gen=F|Num=P|Cas=1|Neg=N|Var=6
A	M	Gen=F|Num=P|Cas=2|Neg=A
A	M	Gen=F|Num=P|Cas=2|Neg=A|Var=6
A	M	Gen=F|Num=P|Cas=2|Neg=N
A	M	Gen=F|Num=P|Cas=2|Neg=N|Var=6
A	M	Gen=F|Num=P|Cas=3|Neg=A
A	M	Gen=F|Num=P|Cas=3|Neg=A|Var=6
A	M	Gen=F|Num=P|Cas=3|Neg=N
A	M	Gen=F|Num=P|Cas=3|Neg=N|Var=6
A	M	Gen=F|Num=P|Cas=4|Neg=A
A	M	Gen=F|Num=P|Cas=4|Neg=A|Var=6
A	M	Gen=F|Num=P|Cas=4|Neg=N
A	M	Gen=F|Num=P|Cas=4|Neg=N|Var=6
A	M	Gen=F|Num=P|Cas=5|Neg=A
A	M	Gen=F|Num=P|Cas=5|Neg=A|Var=6
A	M	Gen=F|Num=P|Cas=5|Neg=N
A	M	Gen=F|Num=P|Cas=5|Neg=N|Var=6
A	M	Gen=F|Num=P|Cas=6|Neg=A
A	M	Gen=F|Num=P|Cas=6|Neg=A|Var=6
A	M	Gen=F|Num=P|Cas=6|Neg=N
A	M	Gen=F|Num=P|Cas=6|Neg=N|Var=6
A	M	Gen=F|Num=P|Cas=7|Neg=A
A	M	Gen=F|Num=P|Cas=7|Neg=A|Var=6
A	M	Gen=F|Num=P|Cas=7|Neg=N
A	M	Gen=F|Num=P|Cas=7|Neg=N|Var=6
A	M	Gen=F|Num=S|Cas=1|Neg=A
A	M	Gen=F|Num=S|Cas=1|Neg=A|Var=6
A	M	Gen=F|Num=S|Cas=1|Neg=N
A	M	Gen=F|Num=S|Cas=1|Neg=N|Var=6
A	M	Gen=F|Num=S|Cas=2|Neg=A
A	M	Gen=F|Num=S|Cas=2|Neg=A|Var=6
A	M	Gen=F|Num=S|Cas=2|Neg=N
A	M	Gen=F|Num=S|Cas=2|Neg=N|Var=6
A	M	Gen=F|Num=S|Cas=3|Neg=A
A	M	Gen=F|Num=S|Cas=3|Neg=A|Var=6
A	M	Gen=F|Num=S|Cas=3|Neg=N
A	M	Gen=F|Num=S|Cas=3|Neg=N|Var=6
A	M	Gen=F|Num=S|Cas=4|Neg=A
A	M	Gen=F|Num=S|Cas=4|Neg=A|Var=6
A	M	Gen=F|Num=S|Cas=4|Neg=N
A	M	Gen=F|Num=S|Cas=4|Neg=N|Var=6
A	M	Gen=F|Num=S|Cas=5|Neg=A
A	M	Gen=F|Num=S|Cas=5|Neg=A|Var=6
A	M	Gen=F|Num=S|Cas=5|Neg=N
A	M	Gen=F|Num=S|Cas=5|Neg=N|Var=6
A	M	Gen=F|Num=S|Cas=6|Neg=A
A	M	Gen=F|Num=S|Cas=6|Neg=A|Var=6
A	M	Gen=F|Num=S|Cas=6|Neg=N
A	M	Gen=F|Num=S|Cas=6|Neg=N|Var=6
A	M	Gen=F|Num=S|Cas=7|Neg=A
A	M	Gen=F|Num=S|Cas=7|Neg=A|Var=6
A	M	Gen=F|Num=S|Cas=7|Neg=N
A	M	Gen=F|Num=S|Cas=7|Neg=N|Var=6
A	M	Gen=I|Num=P|Cas=1|Neg=A
A	M	Gen=I|Num=P|Cas=1|Neg=A|Var=6
A	M	Gen=I|Num=P|Cas=1|Neg=N
A	M	Gen=I|Num=P|Cas=1|Neg=N|Var=6
A	M	Gen=I|Num=P|Cas=2|Neg=A
A	M	Gen=I|Num=P|Cas=2|Neg=A|Var=6
A	M	Gen=I|Num=P|Cas=2|Neg=N
A	M	Gen=I|Num=P|Cas=2|Neg=N|Var=6
A	M	Gen=I|Num=P|Cas=3|Neg=A
A	M	Gen=I|Num=P|Cas=3|Neg=A|Var=6
A	M	Gen=I|Num=P|Cas=3|Neg=N
A	M	Gen=I|Num=P|Cas=3|Neg=N|Var=6
A	M	Gen=I|Num=P|Cas=4|Neg=A
A	M	Gen=I|Num=P|Cas=4|Neg=A|Var=6
A	M	Gen=I|Num=P|Cas=4|Neg=N
A	M	Gen=I|Num=P|Cas=4|Neg=N|Var=6
A	M	Gen=I|Num=P|Cas=5|Neg=A
A	M	Gen=I|Num=P|Cas=5|Neg=A|Var=6
A	M	Gen=I|Num=P|Cas=5|Neg=N
A	M	Gen=I|Num=P|Cas=5|Neg=N|Var=6
A	M	Gen=I|Num=P|Cas=6|Neg=A
A	M	Gen=I|Num=P|Cas=6|Neg=A|Var=6
A	M	Gen=I|Num=P|Cas=6|Neg=N
A	M	Gen=I|Num=P|Cas=6|Neg=N|Var=6
A	M	Gen=I|Num=P|Cas=7|Neg=A
A	M	Gen=I|Num=P|Cas=7|Neg=A|Var=6
A	M	Gen=I|Num=P|Cas=7|Neg=N
A	M	Gen=I|Num=P|Cas=7|Neg=N|Var=6
A	M	Gen=I|Num=S|Cas=1|Neg=A
A	M	Gen=I|Num=S|Cas=1|Neg=A|Var=6
A	M	Gen=I|Num=S|Cas=1|Neg=N
A	M	Gen=I|Num=S|Cas=1|Neg=N|Var=6
A	M	Gen=I|Num=S|Cas=2|Neg=A
A	M	Gen=I|Num=S|Cas=2|Neg=A|Var=6
A	M	Gen=I|Num=S|Cas=2|Neg=N
A	M	Gen=I|Num=S|Cas=2|Neg=N|Var=6
A	M	Gen=I|Num=S|Cas=3|Neg=A
A	M	Gen=I|Num=S|Cas=3|Neg=A|Var=6
A	M	Gen=I|Num=S|Cas=3|Neg=N
A	M	Gen=I|Num=S|Cas=3|Neg=N|Var=6
A	M	Gen=I|Num=S|Cas=4|Neg=A
A	M	Gen=I|Num=S|Cas=4|Neg=A|Var=6
A	M	Gen=I|Num=S|Cas=4|Neg=N
A	M	Gen=I|Num=S|Cas=4|Neg=N|Var=6
A	M	Gen=I|Num=S|Cas=5|Neg=A
A	M	Gen=I|Num=S|Cas=5|Neg=A|Var=6
A	M	Gen=I|Num=S|Cas=5|Neg=N
A	M	Gen=I|Num=S|Cas=5|Neg=N|Var=6
A	M	Gen=I|Num=S|Cas=6|Neg=A
A	M	Gen=I|Num=S|Cas=6|Neg=A|Var=6
A	M	Gen=I|Num=S|Cas=6|Neg=A|Var=7
A	M	Gen=I|Num=S|Cas=6|Neg=N
A	M	Gen=I|Num=S|Cas=6|Neg=N|Var=6
A	M	Gen=I|Num=S|Cas=6|Neg=N|Var=7
A	M	Gen=I|Num=S|Cas=7|Neg=A
A	M	Gen=I|Num=S|Cas=7|Neg=A|Var=6
A	M	Gen=I|Num=S|Cas=7|Neg=N
A	M	Gen=I|Num=S|Cas=7|Neg=N|Var=6
A	M	Gen=M|Num=P|Cas=1|Neg=A
A	M	Gen=M|Num=P|Cas=1|Neg=A|Var=6
A	M	Gen=M|Num=P|Cas=1|Neg=N
A	M	Gen=M|Num=P|Cas=1|Neg=N|Var=6
A	M	Gen=M|Num=P|Cas=2|Neg=A
A	M	Gen=M|Num=P|Cas=2|Neg=A|Var=6
A	M	Gen=M|Num=P|Cas=2|Neg=N
A	M	Gen=M|Num=P|Cas=2|Neg=N|Var=6
A	M	Gen=M|Num=P|Cas=3|Neg=A
A	M	Gen=M|Num=P|Cas=3|Neg=A|Var=6
A	M	Gen=M|Num=P|Cas=3|Neg=N
A	M	Gen=M|Num=P|Cas=3|Neg=N|Var=6
A	M	Gen=M|Num=P|Cas=4|Neg=A
A	M	Gen=M|Num=P|Cas=4|Neg=A|Var=6
A	M	Gen=M|Num=P|Cas=4|Neg=N
A	M	Gen=M|Num=P|Cas=4|Neg=N|Var=6
A	M	Gen=M|Num=P|Cas=5|Neg=A
A	M	Gen=M|Num=P|Cas=5|Neg=A|Var=6
A	M	Gen=M|Num=P|Cas=5|Neg=N
A	M	Gen=M|Num=P|Cas=5|Neg=N|Var=6
A	M	Gen=M|Num=P|Cas=6|Neg=A
A	M	Gen=M|Num=P|Cas=6|Neg=A|Var=6
A	M	Gen=M|Num=P|Cas=6|Neg=N
A	M	Gen=M|Num=P|Cas=6|Neg=N|Var=6
A	M	Gen=M|Num=P|Cas=7|Neg=A
A	M	Gen=M|Num=P|Cas=7|Neg=A|Var=6
A	M	Gen=M|Num=P|Cas=7|Neg=N
A	M	Gen=M|Num=P|Cas=7|Neg=N|Var=6
A	M	Gen=M|Num=S|Cas=1|Neg=A
A	M	Gen=M|Num=S|Cas=1|Neg=A|Var=6
A	M	Gen=M|Num=S|Cas=1|Neg=N
A	M	Gen=M|Num=S|Cas=1|Neg=N|Var=6
A	M	Gen=M|Num=S|Cas=2|Neg=A
A	M	Gen=M|Num=S|Cas=2|Neg=A|Var=6
A	M	Gen=M|Num=S|Cas=2|Neg=N
A	M	Gen=M|Num=S|Cas=2|Neg=N|Var=6
A	M	Gen=M|Num=S|Cas=3|Neg=A
A	M	Gen=M|Num=S|Cas=3|Neg=A|Var=6
A	M	Gen=M|Num=S|Cas=3|Neg=N
A	M	Gen=M|Num=S|Cas=3|Neg=N|Var=6
A	M	Gen=M|Num=S|Cas=4|Neg=A
A	M	Gen=M|Num=S|Cas=4|Neg=A|Var=6
A	M	Gen=M|Num=S|Cas=4|Neg=N
A	M	Gen=M|Num=S|Cas=4|Neg=N|Var=6
A	M	Gen=M|Num=S|Cas=5|Neg=A
A	M	Gen=M|Num=S|Cas=5|Neg=A|Var=6
A	M	Gen=M|Num=S|Cas=5|Neg=N
A	M	Gen=M|Num=S|Cas=5|Neg=N|Var=6
A	M	Gen=M|Num=S|Cas=6|Neg=A
A	M	Gen=M|Num=S|Cas=6|Neg=A|Var=6
A	M	Gen=M|Num=S|Cas=6|Neg=A|Var=7
A	M	Gen=M|Num=S|Cas=6|Neg=N
A	M	Gen=M|Num=S|Cas=6|Neg=N|Var=6
A	M	Gen=M|Num=S|Cas=6|Neg=N|Var=7
A	M	Gen=M|Num=S|Cas=7|Neg=A
A	M	Gen=M|Num=S|Cas=7|Neg=A|Var=6
A	M	Gen=M|Num=S|Cas=7|Neg=N
A	M	Gen=M|Num=S|Cas=7|Neg=N|Var=6
A	M	Gen=N|Num=D|Cas=7|Neg=A
A	M	Gen=N|Num=D|Cas=7|Neg=A|Var=6
A	M	Gen=N|Num=D|Cas=7|Neg=N
A	M	Gen=N|Num=D|Cas=7|Neg=N|Var=6
A	M	Gen=N|Num=P|Cas=1|Neg=A
A	M	Gen=N|Num=P|Cas=1|Neg=A|Var=6
A	M	Gen=N|Num=P|Cas=1|Neg=N
A	M	Gen=N|Num=P|Cas=1|Neg=N|Var=6
A	M	Gen=N|Num=P|Cas=2|Neg=A
A	M	Gen=N|Num=P|Cas=2|Neg=A|Var=6
A	M	Gen=N|Num=P|Cas=2|Neg=N
A	M	Gen=N|Num=P|Cas=2|Neg=N|Var=6
A	M	Gen=N|Num=P|Cas=3|Neg=A
A	M	Gen=N|Num=P|Cas=3|Neg=A|Var=6
A	M	Gen=N|Num=P|Cas=3|Neg=N
A	M	Gen=N|Num=P|Cas=3|Neg=N|Var=6
A	M	Gen=N|Num=P|Cas=4|Neg=A
A	M	Gen=N|Num=P|Cas=4|Neg=A|Var=6
A	M	Gen=N|Num=P|Cas=4|Neg=N
A	M	Gen=N|Num=P|Cas=4|Neg=N|Var=6
A	M	Gen=N|Num=P|Cas=5|Neg=A
A	M	Gen=N|Num=P|Cas=5|Neg=A|Var=6
A	M	Gen=N|Num=P|Cas=5|Neg=N
A	M	Gen=N|Num=P|Cas=5|Neg=N|Var=6
A	M	Gen=N|Num=P|Cas=6|Neg=A
A	M	Gen=N|Num=P|Cas=6|Neg=A|Var=6
A	M	Gen=N|Num=P|Cas=6|Neg=N
A	M	Gen=N|Num=P|Cas=6|Neg=N|Var=6
A	M	Gen=N|Num=P|Cas=7|Neg=A
A	M	Gen=N|Num=P|Cas=7|Neg=A|Var=6
A	M	Gen=N|Num=P|Cas=7|Neg=N
A	M	Gen=N|Num=P|Cas=7|Neg=N|Var=6
A	M	Gen=N|Num=S|Cas=1|Neg=A
A	M	Gen=N|Num=S|Cas=1|Neg=A|Var=6
A	M	Gen=N|Num=S|Cas=1|Neg=N
A	M	Gen=N|Num=S|Cas=1|Neg=N|Var=6
A	M	Gen=N|Num=S|Cas=2|Neg=A
A	M	Gen=N|Num=S|Cas=2|Neg=A|Var=6
A	M	Gen=N|Num=S|Cas=2|Neg=N
A	M	Gen=N|Num=S|Cas=2|Neg=N|Var=6
A	M	Gen=N|Num=S|Cas=3|Neg=A
A	M	Gen=N|Num=S|Cas=3|Neg=A|Var=6
A	M	Gen=N|Num=S|Cas=3|Neg=N
A	M	Gen=N|Num=S|Cas=3|Neg=N|Var=6
A	M	Gen=N|Num=S|Cas=4|Neg=A
A	M	Gen=N|Num=S|Cas=4|Neg=A|Var=6
A	M	Gen=N|Num=S|Cas=4|Neg=N
A	M	Gen=N|Num=S|Cas=4|Neg=N|Var=6
A	M	Gen=N|Num=S|Cas=5|Neg=A
A	M	Gen=N|Num=S|Cas=5|Neg=A|Var=6
A	M	Gen=N|Num=S|Cas=5|Neg=N
A	M	Gen=N|Num=S|Cas=5|Neg=N|Var=6
A	M	Gen=N|Num=S|Cas=6|Neg=A
A	M	Gen=N|Num=S|Cas=6|Neg=A|Var=6
A	M	Gen=N|Num=S|Cas=6|Neg=A|Var=7
A	M	Gen=N|Num=S|Cas=6|Neg=N
A	M	Gen=N|Num=S|Cas=6|Neg=N|Var=6
A	M	Gen=N|Num=S|Cas=6|Neg=N|Var=7
A	M	Gen=N|Num=S|Cas=7|Neg=A
A	M	Gen=N|Num=S|Cas=7|Neg=A|Var=6
A	M	Gen=N|Num=S|Cas=7|Neg=N
A	M	Gen=N|Num=S|Cas=7|Neg=N|Var=6
A	O	Gen=F|Num=P
A	O	Gen=F|Num=P|Var=1
A	O	Gen=F|Num=P|Var=6
A	O	Gen=F|Num=S
A	O	Gen=F|Num=S|Var=1
A	O	Gen=I|Num=P
A	O	Gen=I|Num=P|Var=1
A	O	Gen=I|Num=P|Var=6
A	O	Gen=M|Num=P
A	O	Gen=M|Num=P|Var=1
A	O	Gen=M|Num=P|Var=6
A	O	Gen=N|Num=P
A	O	Gen=N|Num=P|Var=1
A	O	Gen=N|Num=P|Var=6
A	O	Gen=N|Num=S
A	O	Gen=N|Num=S|Var=1
A	O	Gen=N|Num=S|Var=6
A	O	Gen=Y|Num=S
A	O	Gen=Y|Num=S|Var=6
A	U	Gen=F|Num=D|Cas=7|PGe=F
A	U	Gen=F|Num=D|Cas=7|PGe=F|Var=6
A	U	Gen=F|Num=D|Cas=7|PGe=M
A	U	Gen=F|Num=D|Cas=7|PGe=M|Var=6
A	U	Gen=F|Num=P|Cas=1|PGe=F
A	U	Gen=F|Num=P|Cas=1|PGe=F|Sem=Y
A	U	Gen=F|Num=P|Cas=1|PGe=F|Sem=m
A	U	Gen=F|Num=P|Cas=1|PGe=F|Var=6
A	U	Gen=F|Num=P|Cas=1|PGe=M
A	U	Gen=F|Num=P|Cas=1|PGe=M|Sem=S
A	U	Gen=F|Num=P|Cas=1|PGe=M|Sem=Y
A	U	Gen=F|Num=P|Cas=1|PGe=M|Sem=m
A	U	Gen=F|Num=P|Cas=1|PGe=M|Var=5
A	U	Gen=F|Num=P|Cas=1|PGe=M|Var=6
A	U	Gen=F|Num=P|Cas=2|PGe=F
A	U	Gen=F|Num=P|Cas=2|PGe=F|Sem=Y
A	U	Gen=F|Num=P|Cas=2|PGe=F|Sem=m
A	U	Gen=F|Num=P|Cas=2|PGe=F|Var=6
A	U	Gen=F|Num=P|Cas=2|PGe=M
A	U	Gen=F|Num=P|Cas=2|PGe=M|Sem=S
A	U	Gen=F|Num=P|Cas=2|PGe=M|Sem=Y
A	U	Gen=F|Num=P|Cas=2|PGe=M|Sem=m
A	U	Gen=F|Num=P|Cas=2|PGe=M|Var=6
A	U	Gen=F|Num=P|Cas=3|PGe=F
A	U	Gen=F|Num=P|Cas=3|PGe=F|Var=6
A	U	Gen=F|Num=P|Cas=3|PGe=M
A	U	Gen=F|Num=P|Cas=3|PGe=M|Sem=S
A	U	Gen=F|Num=P|Cas=3|PGe=M|Sem=Y
A	U	Gen=F|Num=P|Cas=3|PGe=M|Sem=m
A	U	Gen=F|Num=P|Cas=3|PGe=M|Var=6
A	U	Gen=F|Num=P|Cas=4|PGe=F
A	U	Gen=F|Num=P|Cas=4|PGe=F|Sem=Y
A	U	Gen=F|Num=P|Cas=4|PGe=F|Sem=m
A	U	Gen=F|Num=P|Cas=4|PGe=F|Var=6
A	U	Gen=F|Num=P|Cas=4|PGe=M
A	U	Gen=F|Num=P|Cas=4|PGe=M|Sem=E
A	U	Gen=F|Num=P|Cas=4|PGe=M|Sem=S
A	U	Gen=F|Num=P|Cas=4|PGe=M|Sem=m
A	U	Gen=F|Num=P|Cas=4|PGe=M|Var=6
A	U	Gen=F|Num=P|Cas=5|PGe=F
A	U	Gen=F|Num=P|Cas=5|PGe=F|Var=6
A	U	Gen=F|Num=P|Cas=5|PGe=M
A	U	Gen=F|Num=P|Cas=5|PGe=M|Var=6
A	U	Gen=F|Num=P|Cas=6|PGe=F
A	U	Gen=F|Num=P|Cas=6|PGe=F|Var=6
A	U	Gen=F|Num=P|Cas=6|PGe=M
A	U	Gen=F|Num=P|Cas=6|PGe=M|Sem=S
A	U	Gen=F|Num=P|Cas=6|PGe=M|Sem=Y
A	U	Gen=F|Num=P|Cas=6|PGe=M|Sem=m
A	U	Gen=F|Num=P|Cas=6|PGe=M|Var=6
A	U	Gen=F|Num=P|Cas=7|PGe=F
A	U	Gen=F|Num=P|Cas=7|PGe=F|Var=6
A	U	Gen=F|Num=P|Cas=7|PGe=F|Var=7
A	U	Gen=F|Num=P|Cas=7|PGe=M
A	U	Gen=F|Num=P|Cas=7|PGe=M|Sem=S
A	U	Gen=F|Num=P|Cas=7|PGe=M|Sem=m
A	U	Gen=F|Num=P|Cas=7|PGe=M|Var=6
A	U	Gen=F|Num=P|Cas=7|PGe=M|Var=7
A	U	Gen=F|Num=S|Cas=1|PGe=F
A	U	Gen=F|Num=S|Cas=1|PGe=F|Sem=Y
A	U	Gen=F|Num=S|Cas=1|PGe=F|Sem=m
A	U	Gen=F|Num=S|Cas=1|PGe=F|Var=6
A	U	Gen=F|Num=S|Cas=1|PGe=M
A	U	Gen=F|Num=S|Cas=1|PGe=M|Sem=G
A	U	Gen=F|Num=S|Cas=1|PGe=M|Sem=S
A	U	Gen=F|Num=S|Cas=1|PGe=M|Sem=Y
A	U	Gen=F|Num=S|Cas=1|PGe=M|Sem=m
A	U	Gen=F|Num=S|Cas=1|PGe=M|Var=6
A	U	Gen=F|Num=S|Cas=2|PGe=F
A	U	Gen=F|Num=S|Cas=2|PGe=F|Sem=Y
A	U	Gen=F|Num=S|Cas=2|PGe=F|Sem=m
A	U	Gen=F|Num=S|Cas=2|PGe=F|Var=6
A	U	Gen=F|Num=S|Cas=2|PGe=M
A	U	Gen=F|Num=S|Cas=2|PGe=M|Sem=S
A	U	Gen=F|Num=S|Cas=2|PGe=M|Sem=Y
A	U	Gen=F|Num=S|Cas=2|PGe=M|Sem=m
A	U	Gen=F|Num=S|Cas=2|PGe=M|Var=6
A	U	Gen=F|Num=S|Cas=3|PGe=F
A	U	Gen=F|Num=S|Cas=3|PGe=F|Var=6
A	U	Gen=F|Num=S|Cas=3|PGe=M
A	U	Gen=F|Num=S|Cas=3|PGe=M|Sem=S
A	U	Gen=F|Num=S|Cas=3|PGe=M|Sem=Y
A	U	Gen=F|Num=S|Cas=3|PGe=M|Sem=m
A	U	Gen=F|Num=S|Cas=3|PGe=M|Var=6
A	U	Gen=F|Num=S|Cas=4|PGe=F
A	U	Gen=F|Num=S|Cas=4|PGe=F|Sem=S
A	U	Gen=F|Num=S|Cas=4|PGe=F|Sem=m
A	U	Gen=F|Num=S|Cas=4|PGe=F|Var=6
A	U	Gen=F|Num=S|Cas=4|PGe=M
A	U	Gen=F|Num=S|Cas=4|PGe=M|Sem=S
A	U	Gen=F|Num=S|Cas=4|PGe=M|Sem=Y
A	U	Gen=F|Num=S|Cas=4|PGe=M|Sem=m
A	U	Gen=F|Num=S|Cas=4|PGe=M|Var=6
A	U	Gen=F|Num=S|Cas=5|PGe=F
A	U	Gen=F|Num=S|Cas=5|PGe=F|Var=6
A	U	Gen=F|Num=S|Cas=5|PGe=M
A	U	Gen=F|Num=S|Cas=5|PGe=M|Var=6
A	U	Gen=F|Num=S|Cas=6|PGe=F
A	U	Gen=F|Num=S|Cas=6|PGe=F|Var=6
A	U	Gen=F|Num=S|Cas=6|PGe=M
A	U	Gen=F|Num=S|Cas=6|PGe=M|Sem=S
A	U	Gen=F|Num=S|Cas=6|PGe=M|Sem=Y
A	U	Gen=F|Num=S|Cas=6|PGe=M|Sem=m
A	U	Gen=F|Num=S|Cas=6|PGe=M|Var=6
A	U	Gen=F|Num=S|Cas=7|PGe=F
A	U	Gen=F|Num=S|Cas=7|PGe=F|Var=6
A	U	Gen=F|Num=S|Cas=7|PGe=M
A	U	Gen=F|Num=S|Cas=7|PGe=M|Sem=S
A	U	Gen=F|Num=S|Cas=7|PGe=M|Sem=Y
A	U	Gen=F|Num=S|Cas=7|PGe=M|Sem=m
A	U	Gen=F|Num=S|Cas=7|PGe=M|Var=6
A	U	Gen=I|Num=P|Cas=1|PGe=F
A	U	Gen=I|Num=P|Cas=1|PGe=F|Sem=Y
A	U	Gen=I|Num=P|Cas=1|PGe=F|Sem=m
A	U	Gen=I|Num=P|Cas=1|PGe=F|Var=6
A	U	Gen=I|Num=P|Cas=1|PGe=M
A	U	Gen=I|Num=P|Cas=1|PGe=M|Sem=E
A	U	Gen=I|Num=P|Cas=1|PGe=M|Sem=G
A	U	Gen=I|Num=P|Cas=1|PGe=M|Sem=S
A	U	Gen=I|Num=P|Cas=1|PGe=M|Sem=Y
A	U	Gen=I|Num=P|Cas=1|PGe=M|Sem=m
A	U	Gen=I|Num=P|Cas=1|PGe=M|Var=6
A	U	Gen=I|Num=P|Cas=2|PGe=F
A	U	Gen=I|Num=P|Cas=2|PGe=F|Var=6
A	U	Gen=I|Num=P|Cas=2|PGe=M
A	U	Gen=I|Num=P|Cas=2|PGe=M|Sem=S
A	U	Gen=I|Num=P|Cas=2|PGe=M|Sem=Y
A	U	Gen=I|Num=P|Cas=2|PGe=M|Sem=m
A	U	Gen=I|Num=P|Cas=2|PGe=M|Var=6
A	U	Gen=I|Num=P|Cas=3|PGe=F
A	U	Gen=I|Num=P|Cas=3|PGe=F|Var=6
A	U	Gen=I|Num=P|Cas=3|PGe=M
A	U	Gen=I|Num=P|Cas=3|PGe=M|Sem=S
A	U	Gen=I|Num=P|Cas=3|PGe=M|Sem=Y
A	U	Gen=I|Num=P|Cas=3|PGe=M|Sem=m
A	U	Gen=I|Num=P|Cas=3|PGe=M|Var=6
A	U	Gen=I|Num=P|Cas=4|PGe=F
A	U	Gen=I|Num=P|Cas=4|PGe=F|Var=6
A	U	Gen=I|Num=P|Cas=4|PGe=M
A	U	Gen=I|Num=P|Cas=4|PGe=M|Sem=S
A	U	Gen=I|Num=P|Cas=4|PGe=M|Sem=Y
A	U	Gen=I|Num=P|Cas=4|PGe=M|Sem=m
A	U	Gen=I|Num=P|Cas=4|PGe=M|Var=6
A	U	Gen=I|Num=P|Cas=5|PGe=F
A	U	Gen=I|Num=P|Cas=5|PGe=F|Var=6
A	U	Gen=I|Num=P|Cas=5|PGe=M
A	U	Gen=I|Num=P|Cas=5|PGe=M|Var=6
A	U	Gen=I|Num=P|Cas=6|PGe=F
A	U	Gen=I|Num=P|Cas=6|PGe=F|Var=6
A	U	Gen=I|Num=P|Cas=6|PGe=M
A	U	Gen=I|Num=P|Cas=6|PGe=M|Sem=G
A	U	Gen=I|Num=P|Cas=6|PGe=M|Sem=S
A	U	Gen=I|Num=P|Cas=6|PGe=M|Sem=Y
A	U	Gen=I|Num=P|Cas=6|PGe=M|Sem=m
A	U	Gen=I|Num=P|Cas=6|PGe=M|Var=6
A	U	Gen=I|Num=P|Cas=7|PGe=F
A	U	Gen=I|Num=P|Cas=7|PGe=F|Var=6
A	U	Gen=I|Num=P|Cas=7|PGe=F|Var=7
A	U	Gen=I|Num=P|Cas=7|PGe=M
A	U	Gen=I|Num=P|Cas=7|PGe=M|Sem=S
A	U	Gen=I|Num=P|Cas=7|PGe=M|Sem=m
A	U	Gen=I|Num=P|Cas=7|PGe=M|Var=6
A	U	Gen=I|Num=P|Cas=7|PGe=M|Var=7
A	U	Gen=I|Num=S|Cas=1|PGe=F
A	U	Gen=I|Num=S|Cas=1|PGe=F|Sem=Y
A	U	Gen=I|Num=S|Cas=1|PGe=F|Sem=m
A	U	Gen=I|Num=S|Cas=1|PGe=F|Var=6
A	U	Gen=I|Num=S|Cas=1|PGe=M
A	U	Gen=I|Num=S|Cas=1|PGe=M|Sem=E
A	U	Gen=I|Num=S|Cas=1|PGe=M|Sem=G
A	U	Gen=I|Num=S|Cas=1|PGe=M|Sem=S
A	U	Gen=I|Num=S|Cas=1|PGe=M|Sem=Y
A	U	Gen=I|Num=S|Cas=1|PGe=M|Sem=m
A	U	Gen=I|Num=S|Cas=1|PGe=M|Var=6
A	U	Gen=I|Num=S|Cas=2|PGe=F
A	U	Gen=I|Num=S|Cas=2|PGe=F|Sem=Y
A	U	Gen=I|Num=S|Cas=2|PGe=F|Sem=m
A	U	Gen=I|Num=S|Cas=2|PGe=F|Var=6
A	U	Gen=I|Num=S|Cas=2|PGe=M
A	U	Gen=I|Num=S|Cas=2|PGe=M|Sem=G
A	U	Gen=I|Num=S|Cas=2|PGe=M|Sem=S
A	U	Gen=I|Num=S|Cas=2|PGe=M|Sem=Y
A	U	Gen=I|Num=S|Cas=2|PGe=M|Sem=m
A	U	Gen=I|Num=S|Cas=2|PGe=M|Var=6
A	U	Gen=I|Num=S|Cas=3|PGe=F
A	U	Gen=I|Num=S|Cas=3|PGe=F|Var=6
A	U	Gen=I|Num=S|Cas=3|PGe=M
A	U	Gen=I|Num=S|Cas=3|PGe=M|Sem=S
A	U	Gen=I|Num=S|Cas=3|PGe=M|Sem=m
A	U	Gen=I|Num=S|Cas=3|PGe=M|Var=6
A	U	Gen=I|Num=S|Cas=4|PGe=F
A	U	Gen=I|Num=S|Cas=4|PGe=F|Var=6
A	U	Gen=I|Num=S|Cas=4|PGe=M
A	U	Gen=I|Num=S|Cas=4|PGe=M|Sem=E
A	U	Gen=I|Num=S|Cas=4|PGe=M|Sem=G
A	U	Gen=I|Num=S|Cas=4|PGe=M|Sem=S
A	U	Gen=I|Num=S|Cas=4|PGe=M|Sem=Y
A	U	Gen=I|Num=S|Cas=4|PGe=M|Sem=m
A	U	Gen=I|Num=S|Cas=4|PGe=M|Var=6
A	U	Gen=I|Num=S|Cas=5|PGe=F
A	U	Gen=I|Num=S|Cas=5|PGe=F|Var=6
A	U	Gen=I|Num=S|Cas=5|PGe=M
A	U	Gen=I|Num=S|Cas=5|PGe=M|Var=6
A	U	Gen=I|Num=S|Cas=6|PGe=F
A	U	Gen=I|Num=S|Cas=6|PGe=F|Var=1
A	U	Gen=I|Num=S|Cas=6|PGe=F|Var=6
A	U	Gen=I|Num=S|Cas=6|PGe=M
A	U	Gen=I|Num=S|Cas=6|PGe=M|Sem=G
A	U	Gen=I|Num=S|Cas=6|PGe=M|Sem=S
A	U	Gen=I|Num=S|Cas=6|PGe=M|Sem=Y
A	U	Gen=I|Num=S|Cas=6|PGe=M|Sem=m
A	U	Gen=I|Num=S|Cas=6|PGe=M|Var=1
A	U	Gen=I|Num=S|Cas=6|PGe=M|Var=6
A	U	Gen=I|Num=S|Cas=7|PGe=F
A	U	Gen=I|Num=S|Cas=7|PGe=F|Var=6
A	U	Gen=I|Num=S|Cas=7|PGe=M
A	U	Gen=I|Num=S|Cas=7|PGe=M|Sem=S
A	U	Gen=I|Num=S|Cas=7|PGe=M|Sem=Y
A	U	Gen=I|Num=S|Cas=7|PGe=M|Sem=m
A	U	Gen=I|Num=S|Cas=7|PGe=M|Var=6
A	U	Gen=M|Num=P|Cas=1|PGe=F
A	U	Gen=M|Num=P|Cas=1|PGe=F|Sem=Y
A	U	Gen=M|Num=P|Cas=1|PGe=F|Sem=m
A	U	Gen=M|Num=P|Cas=1|PGe=F|Var=6
A	U	Gen=M|Num=P|Cas=1|PGe=M
A	U	Gen=M|Num=P|Cas=1|PGe=M|Sem=S
A	U	Gen=M|Num=P|Cas=1|PGe=M|Sem=m
A	U	Gen=M|Num=P|Cas=1|PGe=M|Var=6
A	U	Gen=M|Num=P|Cas=2|PGe=F
A	U	Gen=M|Num=P|Cas=2|PGe=F|Var=6
A	U	Gen=M|Num=P|Cas=2|PGe=M
A	U	Gen=M|Num=P|Cas=2|PGe=M|Sem=G
A	U	Gen=M|Num=P|Cas=2|PGe=M|Sem=S
A	U	Gen=M|Num=P|Cas=2|PGe=M|Sem=Y
A	U	Gen=M|Num=P|Cas=2|PGe=M|Sem=m
A	U	Gen=M|Num=P|Cas=2|PGe=M|Var=6
A	U	Gen=M|Num=P|Cas=3|PGe=F
A	U	Gen=M|Num=P|Cas=3|PGe=F|Var=6
A	U	Gen=M|Num=P|Cas=3|PGe=M
A	U	Gen=M|Num=P|Cas=3|PGe=M|Sem=S
A	U	Gen=M|Num=P|Cas=3|PGe=M|Sem=m
A	U	Gen=M|Num=P|Cas=3|PGe=M|Var=6
A	U	Gen=M|Num=P|Cas=4|PGe=F
A	U	Gen=M|Num=P|Cas=4|PGe=F|Var=6
A	U	Gen=M|Num=P|Cas=4|PGe=M
A	U	Gen=M|Num=P|Cas=4|PGe=M|Sem=S
A	U	Gen=M|Num=P|Cas=4|PGe=M|Sem=m
A	U	Gen=M|Num=P|Cas=4|PGe=M|Var=6
A	U	Gen=M|Num=P|Cas=5|PGe=F
A	U	Gen=M|Num=P|Cas=5|PGe=F|Var=6
A	U	Gen=M|Num=P|Cas=5|PGe=M
A	U	Gen=M|Num=P|Cas=5|PGe=M|Var=6
A	U	Gen=M|Num=P|Cas=6|PGe=F
A	U	Gen=M|Num=P|Cas=6|PGe=F|Var=6
A	U	Gen=M|Num=P|Cas=6|PGe=M
A	U	Gen=M|Num=P|Cas=6|PGe=M|Sem=S
A	U	Gen=M|Num=P|Cas=6|PGe=M|Sem=m
A	U	Gen=M|Num=P|Cas=6|PGe=M|Var=6
A	U	Gen=M|Num=P|Cas=7|PGe=F
A	U	Gen=M|Num=P|Cas=7|PGe=F|Var=6
A	U	Gen=M|Num=P|Cas=7|PGe=F|Var=7
A	U	Gen=M|Num=P|Cas=7|PGe=M
A	U	Gen=M|Num=P|Cas=7|PGe=M|Sem=S
A	U	Gen=M|Num=P|Cas=7|PGe=M|Sem=m
A	U	Gen=M|Num=P|Cas=7|PGe=M|Var=6
A	U	Gen=M|Num=P|Cas=7|PGe=M|Var=7
A	U	Gen=M|Num=S|Cas=1|PGe=F
A	U	Gen=M|Num=S|Cas=1|PGe=F|Var=6
A	U	Gen=M|Num=S|Cas=1|PGe=M
A	U	Gen=M|Num=S|Cas=1|PGe=M|Sem=S
A	U	Gen=M|Num=S|Cas=1|PGe=M|Sem=Y
A	U	Gen=M|Num=S|Cas=1|PGe=M|Sem=m
A	U	Gen=M|Num=S|Cas=1|PGe=M|Var=6
A	U	Gen=M|Num=S|Cas=2|PGe=F
A	U	Gen=M|Num=S|Cas=2|PGe=F|Var=6
A	U	Gen=M|Num=S|Cas=2|PGe=M
A	U	Gen=M|Num=S|Cas=2|PGe=M|Sem=G
A	U	Gen=M|Num=S|Cas=2|PGe=M|Sem=S
A	U	Gen=M|Num=S|Cas=2|PGe=M|Sem=Y
A	U	Gen=M|Num=S|Cas=2|PGe=M|Sem=m
A	U	Gen=M|Num=S|Cas=2|PGe=M|Var=6
A	U	Gen=M|Num=S|Cas=3|PGe=F
A	U	Gen=M|Num=S|Cas=3|PGe=F|Var=6
A	U	Gen=M|Num=S|Cas=3|PGe=M
A	U	Gen=M|Num=S|Cas=3|PGe=M|Sem=S
A	U	Gen=M|Num=S|Cas=3|PGe=M|Sem=m
A	U	Gen=M|Num=S|Cas=3|PGe=M|Var=6
A	U	Gen=M|Num=S|Cas=4|PGe=F
A	U	Gen=M|Num=S|Cas=4|PGe=F|Sem=Y
A	U	Gen=M|Num=S|Cas=4|PGe=F|Sem=m
A	U	Gen=M|Num=S|Cas=4|PGe=F|Var=6
A	U	Gen=M|Num=S|Cas=4|PGe=M
A	U	Gen=M|Num=S|Cas=4|PGe=M|Sem=S
A	U	Gen=M|Num=S|Cas=4|PGe=M|Sem=Y
A	U	Gen=M|Num=S|Cas=4|PGe=M|Sem=m
A	U	Gen=M|Num=S|Cas=4|PGe=M|Var=6
A	U	Gen=M|Num=S|Cas=5|PGe=F
A	U	Gen=M|Num=S|Cas=5|PGe=F|Var=6
A	U	Gen=M|Num=S|Cas=5|PGe=M
A	U	Gen=M|Num=S|Cas=5|PGe=M|Var=6
A	U	Gen=M|Num=S|Cas=6|PGe=F
A	U	Gen=M|Num=S|Cas=6|PGe=F|Var=1
A	U	Gen=M|Num=S|Cas=6|PGe=F|Var=6
A	U	Gen=M|Num=S|Cas=6|PGe=M
A	U	Gen=M|Num=S|Cas=6|PGe=M|Sem=S
A	U	Gen=M|Num=S|Cas=6|PGe=M|Sem=m
A	U	Gen=M|Num=S|Cas=6|PGe=M|Var=1
A	U	Gen=M|Num=S|Cas=6|PGe=M|Var=6
A	U	Gen=M|Num=S|Cas=7|PGe=F
A	U	Gen=M|Num=S|Cas=7|PGe=F|Var=6
A	U	Gen=M|Num=S|Cas=7|PGe=M
A	U	Gen=M|Num=S|Cas=7|PGe=M|Sem=K
A	U	Gen=M|Num=S|Cas=7|PGe=M|Sem=S
A	U	Gen=M|Num=S|Cas=7|PGe=M|Sem=m
A	U	Gen=M|Num=S|Cas=7|PGe=M|Var=6
A	U	Gen=N|Num=D|Cas=7|PGe=F
A	U	Gen=N|Num=D|Cas=7|PGe=F|Var=6
A	U	Gen=N|Num=D|Cas=7|PGe=M
A	U	Gen=N|Num=D|Cas=7|PGe=M|Var=6
A	U	Gen=N|Num=P|Cas=1|PGe=F
A	U	Gen=N|Num=P|Cas=1|PGe=F|Var=6
A	U	Gen=N|Num=P|Cas=1|PGe=M
A	U	Gen=N|Num=P|Cas=1|PGe=M|Sem=S
A	U	Gen=N|Num=P|Cas=1|PGe=M|Sem=m
A	U	Gen=N|Num=P|Cas=1|PGe=M|Var=6
A	U	Gen=N|Num=P|Cas=2|PGe=F
A	U	Gen=N|Num=P|Cas=2|PGe=F|Var=6
A	U	Gen=N|Num=P|Cas=2|PGe=M
A	U	Gen=N|Num=P|Cas=2|PGe=M|Sem=S
A	U	Gen=N|Num=P|Cas=2|PGe=M|Sem=m
A	U	Gen=N|Num=P|Cas=2|PGe=M|Var=6
A	U	Gen=N|Num=P|Cas=3|PGe=F
A	U	Gen=N|Num=P|Cas=3|PGe=F|Var=6
A	U	Gen=N|Num=P|Cas=3|PGe=M
A	U	Gen=N|Num=P|Cas=3|PGe=M|Sem=S
A	U	Gen=N|Num=P|Cas=3|PGe=M|Sem=m
A	U	Gen=N|Num=P|Cas=3|PGe=M|Var=6
A	U	Gen=N|Num=P|Cas=4|PGe=F
A	U	Gen=N|Num=P|Cas=4|PGe=F|Var=6
A	U	Gen=N|Num=P|Cas=4|PGe=M
A	U	Gen=N|Num=P|Cas=4|PGe=M|Sem=S
A	U	Gen=N|Num=P|Cas=4|PGe=M|Sem=Y
A	U	Gen=N|Num=P|Cas=4|PGe=M|Sem=m
A	U	Gen=N|Num=P|Cas=4|PGe=M|Var=6
A	U	Gen=N|Num=P|Cas=5|PGe=F
A	U	Gen=N|Num=P|Cas=5|PGe=F|Var=6
A	U	Gen=N|Num=P|Cas=5|PGe=M
A	U	Gen=N|Num=P|Cas=5|PGe=M|Var=6
A	U	Gen=N|Num=P|Cas=6|PGe=F
A	U	Gen=N|Num=P|Cas=6|PGe=F|Var=6
A	U	Gen=N|Num=P|Cas=6|PGe=M
A	U	Gen=N|Num=P|Cas=6|PGe=M|Sem=S
A	U	Gen=N|Num=P|Cas=6|PGe=M|Sem=m
A	U	Gen=N|Num=P|Cas=6|PGe=M|Var=6
A	U	Gen=N|Num=P|Cas=7|PGe=F
A	U	Gen=N|Num=P|Cas=7|PGe=F|Var=6
A	U	Gen=N|Num=P|Cas=7|PGe=F|Var=7
A	U	Gen=N|Num=P|Cas=7|PGe=M
A	U	Gen=N|Num=P|Cas=7|PGe=M|Sem=S
A	U	Gen=N|Num=P|Cas=7|PGe=M|Sem=m
A	U	Gen=N|Num=P|Cas=7|PGe=M|Var=6
A	U	Gen=N|Num=P|Cas=7|PGe=M|Var=7
A	U	Gen=N|Num=S|Cas=1|PGe=F
A	U	Gen=N|Num=S|Cas=1|PGe=F|Var=6
A	U	Gen=N|Num=S|Cas=1|PGe=M
A	U	Gen=N|Num=S|Cas=1|PGe=M|Sem=E
A	U	Gen=N|Num=S|Cas=1|PGe=M|Sem=G
A	U	Gen=N|Num=S|Cas=1|PGe=M|Sem=S
A	U	Gen=N|Num=S|Cas=1|PGe=M|Sem=Y
A	U	Gen=N|Num=S|Cas=1|PGe=M|Sem=m
A	U	Gen=N|Num=S|Cas=1|PGe=M|Var=6
A	U	Gen=N|Num=S|Cas=2|PGe=F
A	U	Gen=N|Num=S|Cas=2|PGe=F|Sem=Y
A	U	Gen=N|Num=S|Cas=2|PGe=F|Sem=m
A	U	Gen=N|Num=S|Cas=2|PGe=F|Var=6
A	U	Gen=N|Num=S|Cas=2|PGe=M
A	U	Gen=N|Num=S|Cas=2|PGe=M|Sem=S
A	U	Gen=N|Num=S|Cas=2|PGe=M|Sem=Y
A	U	Gen=N|Num=S|Cas=2|PGe=M|Sem=m
A	U	Gen=N|Num=S|Cas=2|PGe=M|Var=6
A	U	Gen=N|Num=S|Cas=3|PGe=F
A	U	Gen=N|Num=S|Cas=3|PGe=F|Var=6
A	U	Gen=N|Num=S|Cas=3|PGe=M
A	U	Gen=N|Num=S|Cas=3|PGe=M|Sem=S
A	U	Gen=N|Num=S|Cas=3|PGe=M|Sem=m
A	U	Gen=N|Num=S|Cas=3|PGe=M|Var=6
A	U	Gen=N|Num=S|Cas=4|PGe=F
A	U	Gen=N|Num=S|Cas=4|PGe=F|Sem=Y
A	U	Gen=N|Num=S|Cas=4|PGe=F|Sem=m
A	U	Gen=N|Num=S|Cas=4|PGe=F|Var=6
A	U	Gen=N|Num=S|Cas=4|PGe=M
A	U	Gen=N|Num=S|Cas=4|PGe=M|Sem=E
A	U	Gen=N|Num=S|Cas=4|PGe=M|Sem=S
A	U	Gen=N|Num=S|Cas=4|PGe=M|Sem=Y
A	U	Gen=N|Num=S|Cas=4|PGe=M|Sem=m
A	U	Gen=N|Num=S|Cas=4|PGe=M|Var=6
A	U	Gen=N|Num=S|Cas=5|PGe=F
A	U	Gen=N|Num=S|Cas=5|PGe=F|Var=6
A	U	Gen=N|Num=S|Cas=5|PGe=M
A	U	Gen=N|Num=S|Cas=5|PGe=M|Var=6
A	U	Gen=N|Num=S|Cas=6|PGe=F
A	U	Gen=N|Num=S|Cas=6|PGe=F|Var=1
A	U	Gen=N|Num=S|Cas=6|PGe=F|Var=6
A	U	Gen=N|Num=S|Cas=6|PGe=M
A	U	Gen=N|Num=S|Cas=6|PGe=M|Sem=G
A	U	Gen=N|Num=S|Cas=6|PGe=M|Sem=S
A	U	Gen=N|Num=S|Cas=6|PGe=M|Sem=Y
A	U	Gen=N|Num=S|Cas=6|PGe=M|Sem=m
A	U	Gen=N|Num=S|Cas=6|PGe=M|Var=1
A	U	Gen=N|Num=S|Cas=6|PGe=M|Var=6
A	U	Gen=N|Num=S|Cas=7|PGe=F
A	U	Gen=N|Num=S|Cas=7|PGe=F|Var=6
A	U	Gen=N|Num=S|Cas=7|PGe=M
A	U	Gen=N|Num=S|Cas=7|PGe=M|Sem=G
A	U	Gen=N|Num=S|Cas=7|PGe=M|Sem=S
A	U	Gen=N|Num=S|Cas=7|PGe=M|Sem=Y
A	U	Gen=N|Num=S|Cas=7|PGe=M|Sem=m
A	U	Gen=N|Num=S|Cas=7|PGe=M|Var=6
A	U	Gen=X|Num=X|Cas=X|PGe=F|Var=8
A	U	Gen=X|Num=X|Cas=X|PGe=M
A	U	Gen=X|Num=X|Cas=X|PGe=M|Sem=S
A	U	Gen=X|Num=X|Cas=X|PGe=M|Sem=m
A	U	Gen=X|Num=X|Cas=X|PGe=M|Var=6
A	U	Gen=X|Num=X|Cas=X|PGe=M|Var=6|Sem=S
A	U	Gen=X|Num=X|Cas=X|PGe=M|Var=6|Sem=m
A	U	Gen=X|Num=X|Cas=X|PGe=M|Var=8
C	3	_
C	=	_
C	?	Cas=1
C	?	Cas=2
C	?	Cas=3
C	?	Cas=4
C	?	Cas=6
C	?	Cas=7
C	a	Cas=1
C	a	Cas=2
C	a	Cas=2|Var=1
C	a	Cas=3
C	a	Cas=4
C	a	Cas=5
C	a	Cas=6
C	a	Cas=7
C	a	Cas=X
C	a	Cas=X|Var=8
C	d	Gen=F|Num=D|Cas=7
C	d	Gen=F|Num=D|Cas=7|Var=6
C	d	Gen=F|Num=P|Cas=1
C	d	Gen=F|Num=P|Cas=1|Var=6
C	d	Gen=F|Num=P|Cas=2
C	d	Gen=F|Num=P|Cas=2|Var=6
C	d	Gen=F|Num=P|Cas=3
C	d	Gen=F|Num=P|Cas=3|Var=6
C	d	Gen=F|Num=P|Cas=4
C	d	Gen=F|Num=P|Cas=4|Var=6
C	d	Gen=F|Num=P|Cas=5
C	d	Gen=F|Num=P|Cas=5|Var=6
C	d	Gen=F|Num=P|Cas=6
C	d	Gen=F|Num=P|Cas=6|Var=6
C	d	Gen=F|Num=P|Cas=7
C	d	Gen=F|Num=P|Cas=7|Var=6
C	d	Gen=F|Num=P|Cas=7|Var=7
C	d	Gen=F|Num=S|Cas=1
C	d	Gen=F|Num=S|Cas=2
C	d	Gen=F|Num=S|Cas=2|Var=6
C	d	Gen=F|Num=S|Cas=3
C	d	Gen=F|Num=S|Cas=3|Var=6
C	d	Gen=F|Num=S|Cas=4
C	d	Gen=F|Num=S|Cas=4|Var=2
C	d	Gen=F|Num=S|Cas=5
C	d	Gen=F|Num=S|Cas=6
C	d	Gen=F|Num=S|Cas=6|Var=6
C	d	Gen=F|Num=S|Cas=7
C	d	Gen=I|Num=P|Cas=1
C	d	Gen=I|Num=P|Cas=1|Var=6
C	d	Gen=I|Num=P|Cas=2
C	d	Gen=I|Num=P|Cas=2|Var=6
C	d	Gen=I|Num=P|Cas=3
C	d	Gen=I|Num=P|Cas=3|Var=6
C	d	Gen=I|Num=P|Cas=4
C	d	Gen=I|Num=P|Cas=4|Var=6
C	d	Gen=I|Num=P|Cas=5
C	d	Gen=I|Num=P|Cas=5|Var=6
C	d	Gen=I|Num=P|Cas=6
C	d	Gen=I|Num=P|Cas=6|Var=6
C	d	Gen=I|Num=P|Cas=7
C	d	Gen=I|Num=P|Cas=7|Var=6
C	d	Gen=I|Num=P|Cas=7|Var=7
C	d	Gen=I|Num=S|Cas=1
C	d	Gen=I|Num=S|Cas=1|Var=6
C	d	Gen=I|Num=S|Cas=2
C	d	Gen=I|Num=S|Cas=2|Var=6
C	d	Gen=I|Num=S|Cas=3
C	d	Gen=I|Num=S|Cas=3|Var=6
C	d	Gen=I|Num=S|Cas=4
C	d	Gen=I|Num=S|Cas=4|Var=6
C	d	Gen=I|Num=S|Cas=5
C	d	Gen=I|Num=S|Cas=5|Var=6
C	d	Gen=I|Num=S|Cas=6
C	d	Gen=I|Num=S|Cas=6|Var=6
C	d	Gen=I|Num=S|Cas=7
C	d	Gen=I|Num=S|Cas=7|Var=6
C	d	Gen=M|Num=P|Cas=1
C	d	Gen=M|Num=P|Cas=1|Var=6
C	d	Gen=M|Num=P|Cas=2
C	d	Gen=M|Num=P|Cas=2|Var=6
C	d	Gen=M|Num=P|Cas=3
C	d	Gen=M|Num=P|Cas=3|Var=6
C	d	Gen=M|Num=P|Cas=4
C	d	Gen=M|Num=P|Cas=4|Var=6
C	d	Gen=M|Num=P|Cas=5
C	d	Gen=M|Num=P|Cas=5|Var=6
C	d	Gen=M|Num=P|Cas=6
C	d	Gen=M|Num=P|Cas=6|Var=6
C	d	Gen=M|Num=P|Cas=7
C	d	Gen=M|Num=P|Cas=7|Var=6
C	d	Gen=M|Num=P|Cas=7|Var=7
C	d	Gen=M|Num=S|Cas=1
C	d	Gen=M|Num=S|Cas=1|Var=6
C	d	Gen=M|Num=S|Cas=2
C	d	Gen=M|Num=S|Cas=2|Var=6
C	d	Gen=M|Num=S|Cas=3
C	d	Gen=M|Num=S|Cas=3|Var=6
C	d	Gen=M|Num=S|Cas=4
C	d	Gen=M|Num=S|Cas=4|Var=6
C	d	Gen=M|Num=S|Cas=5
C	d	Gen=M|Num=S|Cas=5|Var=6
C	d	Gen=M|Num=S|Cas=6
C	d	Gen=M|Num=S|Cas=6|Var=6
C	d	Gen=M|Num=S|Cas=7
C	d	Gen=M|Num=S|Cas=7|Var=6
C	d	Gen=N|Num=D|Cas=7
C	d	Gen=N|Num=P|Cas=1
C	d	Gen=N|Num=P|Cas=1|Var=6
C	d	Gen=N|Num=P|Cas=2
C	d	Gen=N|Num=P|Cas=2|Var=6
C	d	Gen=N|Num=P|Cas=3
C	d	Gen=N|Num=P|Cas=3|Var=6
C	d	Gen=N|Num=P|Cas=4
C	d	Gen=N|Num=P|Cas=4|Var=6
C	d	Gen=N|Num=P|Cas=5
C	d	Gen=N|Num=P|Cas=5|Var=6
C	d	Gen=N|Num=P|Cas=6
C	d	Gen=N|Num=P|Cas=6|Var=6
C	d	Gen=N|Num=P|Cas=7
C	d	Gen=N|Num=P|Cas=7|Var=6
C	d	Gen=N|Num=P|Cas=7|Var=7
C	d	Gen=N|Num=S|Cas=1
C	d	Gen=N|Num=S|Cas=1|Var=1
C	d	Gen=N|Num=S|Cas=1|Var=6
C	d	Gen=N|Num=S|Cas=2
C	d	Gen=N|Num=S|Cas=2|Var=6
C	d	Gen=N|Num=S|Cas=3
C	d	Gen=N|Num=S|Cas=3|Var=6
C	d	Gen=N|Num=S|Cas=4
C	d	Gen=N|Num=S|Cas=4|Var=1
C	d	Gen=N|Num=S|Cas=4|Var=6
C	d	Gen=N|Num=S|Cas=5
C	d	Gen=N|Num=S|Cas=5|Var=6
C	d	Gen=N|Num=S|Cas=6
C	d	Gen=N|Num=S|Cas=6|Var=6
C	d	Gen=N|Num=S|Cas=7
C	d	Gen=N|Num=S|Cas=7|Var=6
C	d	Gen=X|Num=P|Cas=1
C	d	Gen=X|Num=P|Cas=1|Var=1
C	d	Gen=X|Num=P|Cas=2
C	d	Gen=X|Num=P|Cas=3
C	d	Gen=X|Num=P|Cas=4
C	d	Gen=X|Num=P|Cas=4|Var=1
C	d	Gen=X|Num=P|Cas=5
C	d	Gen=X|Num=P|Cas=5|Var=1
C	d	Gen=X|Num=P|Cas=6
C	d	Gen=X|Num=P|Cas=7
C	d	Gen=X|Num=S|Cas=1
C	d	Gen=X|Num=S|Cas=5
C	d	Gen=Y|Num=S|Cas=2
C	d	Gen=Y|Num=S|Cas=3
C	d	Gen=Y|Num=S|Cas=6
C	d	Gen=Y|Num=S|Cas=7
C	h	Gen=F|Num=D|Cas=7
C	h	Gen=F|Num=P|Cas=1
C	h	Gen=F|Num=P|Cas=4
C	h	Gen=F|Num=P|Cas=5
C	h	Gen=I|Num=P|Cas=1
C	h	Gen=I|Num=P|Cas=5
C	h	Gen=M|Num=P|Cas=1
C	h	Gen=M|Num=P|Cas=5
C	h	Gen=N|Num=P|Cas=1
C	h	Gen=N|Num=P|Cas=4
C	h	Gen=N|Num=P|Cas=5
C	h	Gen=X|Num=P|Cas=2
C	h	Gen=X|Num=P|Cas=3
C	h	Gen=X|Num=P|Cas=6
C	h	Gen=X|Num=P|Cas=7
C	h	Gen=Y|Num=P|Cas=4
C	j	Num=S|Cas=1
C	j	Num=S|Cas=2
C	j	Num=S|Cas=2|Var=1
C	j	Num=S|Cas=3
C	j	Num=S|Cas=3|Var=1
C	j	Num=S|Cas=4
C	j	Num=S|Cas=5
C	j	Num=S|Cas=6
C	j	Num=S|Cas=6|Var=1
C	j	Num=S|Cas=7
C	j	Num=S|Cas=7|Var=1
C	k	Num=P|Cas=1
C	k	Num=P|Cas=2
C	k	Num=P|Cas=3
C	k	Num=P|Cas=4
C	k	Num=P|Cas=5
C	k	Num=P|Cas=6
C	k	Num=P|Cas=7
C	l	Gen=F|Num=D|Cas=7
C	l	Gen=F|Num=D|Cas=7|Var=6
C	l	Gen=F|Num=D|Cas=7|Var=9
C	l	Gen=F|Num=S|Cas=1
C	l	Gen=F|Num=S|Cas=2
C	l	Gen=F|Num=S|Cas=3
C	l	Gen=F|Num=S|Cas=4
C	l	Gen=F|Num=S|Cas=5
C	l	Gen=F|Num=S|Cas=6
C	l	Gen=F|Num=S|Cas=7
C	l	Gen=H|Num=P|Cas=1
C	l	Gen=H|Num=P|Cas=4
C	l	Gen=H|Num=P|Cas=5
C	l	Gen=I|Num=S|Cas=4
C	l	Gen=M|Num=S|Cas=4
C	l	Gen=N|Num=S|Cas=1
C	l	Gen=N|Num=S|Cas=4
C	l	Gen=N|Num=S|Cas=5
C	l	Gen=X|Num=P|Cas=1
C	l	Gen=X|Num=P|Cas=1|Var=6
C	l	Gen=X|Num=P|Cas=2
C	l	Gen=X|Num=P|Cas=2|Var=1
C	l	Gen=X|Num=P|Cas=2|Var=6
C	l	Gen=X|Num=P|Cas=3
C	l	Gen=X|Num=P|Cas=3|Var=1
C	l	Gen=X|Num=P|Cas=3|Var=2
C	l	Gen=X|Num=P|Cas=3|Var=6
C	l	Gen=X|Num=P|Cas=3|Var=9
C	l	Gen=X|Num=P|Cas=4
C	l	Gen=X|Num=P|Cas=4|Var=6
C	l	Gen=X|Num=P|Cas=5
C	l	Gen=X|Num=P|Cas=5|Var=6
C	l	Gen=X|Num=P|Cas=6
C	l	Gen=X|Num=P|Cas=6|Var=1
C	l	Gen=X|Num=P|Cas=6|Var=2
C	l	Gen=X|Num=P|Cas=6|Var=6
C	l	Gen=X|Num=P|Cas=7
C	l	Gen=X|Num=P|Cas=7|Var=1
C	l	Gen=X|Num=P|Cas=7|Var=2
C	l	Gen=X|Num=P|Cas=7|Var=6
C	l	Gen=X|Num=P|Cas=7|Var=9
C	l	Gen=X|Num=P|Cas=X
C	l	Gen=X|Num=P|Cas=X|Sem=K
C	l	Gen=X|Num=P|Cas=X|Sem=m
C	l	Gen=X|Num=S|Cas=2
C	l	Gen=X|Num=S|Cas=3
C	l	Gen=X|Num=S|Cas=6
C	l	Gen=X|Num=S|Cas=7
C	l	Gen=X|Num=S|Cas=X
C	l	Gen=Y|Num=P|Cas=1
C	l	Gen=Y|Num=P|Cas=4
C	l	Gen=Y|Num=P|Cas=5
C	l	Gen=Y|Num=S|Cas=1
C	l	Gen=Y|Num=S|Cas=5
C	l	Gen=Z|Num=S|Cas=2
C	l	Gen=Z|Num=S|Cas=3
C	l	Gen=Z|Num=S|Cas=6
C	l	Gen=Z|Num=S|Cas=7
C	n	Num=P|Cas=2
C	n	Num=P|Cas=2|Var=1
C	n	Num=P|Cas=3
C	n	Num=P|Cas=3|Var=1
C	n	Num=P|Cas=6
C	n	Num=P|Cas=6|Var=1
C	n	Num=P|Cas=7
C	n	Num=P|Cas=7|Var=1
C	n	Num=S|Cas=1
C	n	Num=S|Cas=1|Sem=K
C	n	Num=S|Cas=1|Sem=m
C	n	Num=S|Cas=1|Var=1
C	n	Num=S|Cas=4
C	n	Num=S|Cas=4|Var=1
C	n	Num=S|Cas=5
C	n	Num=S|Cas=5|Var=1
C	n	Num=S|Cas=X
C	n	Num=X|Cas=X
C	o	Var=1
C	o	_
C	r	Gen=F|Num=D|Cas=7
C	r	Gen=F|Num=D|Cas=7|Var=6
C	r	Gen=F|Num=P|Cas=1
C	r	Gen=F|Num=P|Cas=1|Var=6
C	r	Gen=F|Num=P|Cas=2
C	r	Gen=F|Num=P|Cas=2|Var=6
C	r	Gen=F|Num=P|Cas=3
C	r	Gen=F|Num=P|Cas=3|Var=6
C	r	Gen=F|Num=P|Cas=4
C	r	Gen=F|Num=P|Cas=4|Var=6
C	r	Gen=F|Num=P|Cas=5
C	r	Gen=F|Num=P|Cas=5|Var=6
C	r	Gen=F|Num=P|Cas=6
C	r	Gen=F|Num=P|Cas=6|Var=6
C	r	Gen=F|Num=P|Cas=7
C	r	Gen=F|Num=P|Cas=7|Var=6
C	r	Gen=F|Num=P|Cas=7|Var=7
C	r	Gen=F|Num=S|Cas=1
C	r	Gen=F|Num=S|Cas=2
C	r	Gen=F|Num=S|Cas=2|Var=6
C	r	Gen=F|Num=S|Cas=3
C	r	Gen=F|Num=S|Cas=3|Var=6
C	r	Gen=F|Num=S|Cas=4
C	r	Gen=F|Num=S|Cas=5
C	r	Gen=F|Num=S|Cas=6
C	r	Gen=F|Num=S|Cas=6|Var=6
C	r	Gen=F|Num=S|Cas=7
C	r	Gen=I|Num=P|Cas=1
C	r	Gen=I|Num=P|Cas=1|Var=6
C	r	Gen=I|Num=P|Cas=2
C	r	Gen=I|Num=P|Cas=2|Var=6
C	r	Gen=I|Num=P|Cas=3
C	r	Gen=I|Num=P|Cas=3|Var=6
C	r	Gen=I|Num=P|Cas=4
C	r	Gen=I|Num=P|Cas=4|Var=6
C	r	Gen=I|Num=P|Cas=5
C	r	Gen=I|Num=P|Cas=5|Var=6
C	r	Gen=I|Num=P|Cas=6
C	r	Gen=I|Num=P|Cas=6|Var=6
C	r	Gen=I|Num=P|Cas=7
C	r	Gen=I|Num=P|Cas=7|Var=6
C	r	Gen=I|Num=P|Cas=7|Var=7
C	r	Gen=I|Num=S|Cas=1
C	r	Gen=I|Num=S|Cas=1|Var=6
C	r	Gen=I|Num=S|Cas=2
C	r	Gen=I|Num=S|Cas=2|Var=6
C	r	Gen=I|Num=S|Cas=3
C	r	Gen=I|Num=S|Cas=3|Var=6
C	r	Gen=I|Num=S|Cas=4
C	r	Gen=I|Num=S|Cas=4|Var=6
C	r	Gen=I|Num=S|Cas=5
C	r	Gen=I|Num=S|Cas=5|Var=6
C	r	Gen=I|Num=S|Cas=6
C	r	Gen=I|Num=S|Cas=6|Var=6
C	r	Gen=I|Num=S|Cas=6|Var=7
C	r	Gen=I|Num=S|Cas=7
C	r	Gen=I|Num=S|Cas=7|Var=6
C	r	Gen=M|Num=P|Cas=1
C	r	Gen=M|Num=P|Cas=1|Var=6
C	r	Gen=M|Num=P|Cas=2
C	r	Gen=M|Num=P|Cas=2|Var=6
C	r	Gen=M|Num=P|Cas=3
C	r	Gen=M|Num=P|Cas=3|Var=6
C	r	Gen=M|Num=P|Cas=4
C	r	Gen=M|Num=P|Cas=4|Var=6
C	r	Gen=M|Num=P|Cas=5
C	r	Gen=M|Num=P|Cas=5|Var=6
C	r	Gen=M|Num=P|Cas=6
C	r	Gen=M|Num=P|Cas=6|Var=6
C	r	Gen=M|Num=P|Cas=7
C	r	Gen=M|Num=P|Cas=7|Var=6
C	r	Gen=M|Num=P|Cas=7|Var=7
C	r	Gen=M|Num=S|Cas=1
C	r	Gen=M|Num=S|Cas=1|Var=6
C	r	Gen=M|Num=S|Cas=2
C	r	Gen=M|Num=S|Cas=2|Var=6
C	r	Gen=M|Num=S|Cas=3
C	r	Gen=M|Num=S|Cas=3|Var=6
C	r	Gen=M|Num=S|Cas=4
C	r	Gen=M|Num=S|Cas=4|Var=6
C	r	Gen=M|Num=S|Cas=5
C	r	Gen=M|Num=S|Cas=5|Var=6
C	r	Gen=M|Num=S|Cas=6
C	r	Gen=M|Num=S|Cas=6|Var=6
C	r	Gen=M|Num=S|Cas=6|Var=7
C	r	Gen=M|Num=S|Cas=7
C	r	Gen=M|Num=S|Cas=7|Var=6
C	r	Gen=N|Num=D|Cas=7
C	r	Gen=N|Num=P|Cas=1
C	r	Gen=N|Num=P|Cas=1|Var=6
C	r	Gen=N|Num=P|Cas=2
C	r	Gen=N|Num=P|Cas=2|Var=6
C	r	Gen=N|Num=P|Cas=3
C	r	Gen=N|Num=P|Cas=3|Var=6
C	r	Gen=N|Num=P|Cas=4
C	r	Gen=N|Num=P|Cas=4|Var=6
C	r	Gen=N|Num=P|Cas=5
C	r	Gen=N|Num=P|Cas=5|Var=6
C	r	Gen=N|Num=P|Cas=6
C	r	Gen=N|Num=P|Cas=6|Var=6
C	r	Gen=N|Num=P|Cas=7
C	r	Gen=N|Num=P|Cas=7|Var=6
C	r	Gen=N|Num=P|Cas=7|Var=7
C	r	Gen=N|Num=S|Cas=1
C	r	Gen=N|Num=S|Cas=1|Var=6
C	r	Gen=N|Num=S|Cas=2
C	r	Gen=N|Num=S|Cas=2|Var=6
C	r	Gen=N|Num=S|Cas=3
C	r	Gen=N|Num=S|Cas=3|Var=6
C	r	Gen=N|Num=S|Cas=4
C	r	Gen=N|Num=S|Cas=4|Var=6
C	r	Gen=N|Num=S|Cas=5
C	r	Gen=N|Num=S|Cas=5|Var=6
C	r	Gen=N|Num=S|Cas=6
C	r	Gen=N|Num=S|Cas=6|Var=6
C	r	Gen=N|Num=S|Cas=6|Var=7
C	r	Gen=N|Num=S|Cas=7
C	r	Gen=N|Num=S|Cas=7|Var=6
C	r	Gen=X|Num=X|Cas=X
C	r	Gen=X|Num=X|Cas=X|Sem=K
C	r	Gen=X|Num=X|Cas=X|Sem=m
C	u	_
C	v	Var=1
C	v	Var=6
C	v	_
C	w	Gen=F|Num=D|Cas=7
C	w	Gen=F|Num=P|Cas=1
C	w	Gen=F|Num=P|Cas=4
C	w	Gen=F|Num=P|Cas=5
C	w	Gen=F|Num=S|Cas=1
C	w	Gen=F|Num=S|Cas=2
C	w	Gen=F|Num=S|Cas=3
C	w	Gen=F|Num=S|Cas=4
C	w	Gen=F|Num=S|Cas=5
C	w	Gen=F|Num=S|Cas=6
C	w	Gen=F|Num=S|Cas=7
C	w	Gen=I|Num=P|Cas=1
C	w	Gen=I|Num=P|Cas=5
C	w	Gen=I|Num=S|Cas=4
C	w	Gen=M|Num=P|Cas=1
C	w	Gen=M|Num=P|Cas=5
C	w	Gen=M|Num=S|Cas=4
C	w	Gen=N|Num=P|Cas=1
C	w	Gen=N|Num=P|Cas=4
C	w	Gen=N|Num=P|Cas=5
C	w	Gen=N|Num=S|Cas=1
C	w	Gen=N|Num=S|Cas=4
C	w	Gen=N|Num=S|Cas=5
C	w	Gen=X|Num=P|Cas=2
C	w	Gen=X|Num=P|Cas=3
C	w	Gen=X|Num=P|Cas=6
C	w	Gen=X|Num=P|Cas=7
C	w	Gen=Y|Num=P|Cas=4
C	w	Gen=Y|Num=S|Cas=1
C	w	Gen=Y|Num=S|Cas=5
C	w	Gen=Z|Num=S|Cas=2
C	w	Gen=Z|Num=S|Cas=3
C	w	Gen=Z|Num=S|Cas=6
C	w	Gen=Z|Num=S|Cas=7
C	y	Gen=F|Num=P|Cas=1
C	y	Gen=F|Num=P|Cas=2
C	y	Gen=F|Num=P|Cas=3
C	y	Gen=F|Num=P|Cas=4
C	y	Gen=F|Num=P|Cas=5
C	y	Gen=F|Num=P|Cas=6
C	y	Gen=F|Num=P|Cas=7
C	y	Gen=F|Num=P|Cas=7|Var=6
C	y	Gen=F|Num=S|Cas=1
C	y	Gen=F|Num=S|Cas=2
C	y	Gen=F|Num=S|Cas=3
C	y	Gen=F|Num=S|Cas=4
C	y	Gen=F|Num=S|Cas=5
C	y	Gen=F|Num=S|Cas=6
C	y	Gen=F|Num=S|Cas=7
C	z	Gen=F|Num=D|Cas=7
C	z	Gen=F|Num=P|Cas=1
C	z	Gen=F|Num=P|Cas=4
C	z	Gen=F|Num=S|Cas=1
C	z	Gen=F|Num=S|Cas=2
C	z	Gen=F|Num=S|Cas=3
C	z	Gen=F|Num=S|Cas=4
C	z	Gen=F|Num=S|Cas=6
C	z	Gen=F|Num=S|Cas=7
C	z	Gen=I|Num=P|Cas=1
C	z	Gen=I|Num=S|Cas=4
C	z	Gen=M|Num=P|Cas=1
C	z	Gen=M|Num=S|Cas=4
C	z	Gen=N|Num=P|Cas=1
C	z	Gen=N|Num=P|Cas=4
C	z	Gen=N|Num=S|Cas=1
C	z	Gen=N|Num=S|Cas=4
C	z	Gen=X|Num=P|Cas=2
C	z	Gen=X|Num=P|Cas=3
C	z	Gen=X|Num=P|Cas=6
C	z	Gen=X|Num=P|Cas=7
C	z	Gen=Y|Num=P|Cas=4
C	z	Gen=Y|Num=S|Cas=1
C	z	Gen=Z|Num=S|Cas=2
C	z	Gen=Z|Num=S|Cas=3
C	z	Gen=Z|Num=S|Cas=6
C	z	Gen=Z|Num=S|Cas=7
C	}	Var=1
C	}	Var=2
C	}	_
D	!	_
D	b	Neg=A
D	b	Neg=N
D	b	Sem=R
D	b	Sem=m
D	b	Var=1
D	b	Var=2
D	b	Var=4
D	b	Var=6
D	b	Var=7
D	b	Var=8
D	b	Var=9
D	b	_
D	g	Gra=1|Neg=A
D	g	Gra=1|Neg=A|Sem=S
D	g	Gra=1|Neg=A|Sem=m
D	g	Gra=1|Neg=A|Var=1
D	g	Gra=1|Neg=A|Var=3
D	g	Gra=1|Neg=A|Var=8
D	g	Gra=1|Neg=N
D	g	Gra=1|Neg=N|Var=3
D	g	Gra=1|Neg=N|Var=8
D	g	Gra=2|Neg=A
D	g	Gra=2|Neg=A|Sem=K
D	g	Gra=2|Neg=A|Sem=m
D	g	Gra=2|Neg=A|Var=1
D	g	Gra=2|Neg=A|Var=2
D	g	Gra=2|Neg=A|Var=3
D	g	Gra=2|Neg=A|Var=6
D	g	Gra=2|Neg=N
D	g	Gra=2|Neg=N|Var=1
D	g	Gra=2|Neg=N|Var=2
D	g	Gra=2|Neg=N|Var=3
D	g	Gra=2|Neg=N|Var=6
D	g	Gra=3|Neg=A
D	g	Gra=3|Neg=A|Var=1
D	g	Gra=3|Neg=A|Var=2
D	g	Gra=3|Neg=A|Var=3
D	g	Gra=3|Neg=A|Var=6
D	g	Gra=3|Neg=N
D	g	Gra=3|Neg=N|Var=1
D	g	Gra=3|Neg=N|Var=2
D	g	Gra=3|Neg=N|Var=3
D	g	Gra=3|Neg=N|Var=6
I	I	Sem=K
I	I	Sem=m
I	I	_
J	*	_
J	,	Num=P|Per=1
J	,	Num=P|Per=2
J	,	Num=S|Per=1
J	,	Num=S|Per=2
J	,	Num=X|Per=3
J	,	Var=1
J	,	Var=8
J	,	_
J	^	Sem=K
J	^	Var=1
J	^	Var=2
J	^	Var=8
J	^	_
N	;	_
N	N	Gen=F|Num=D|Cas=7|Neg=A
N	N	Gen=F|Num=D|Cas=7|Neg=A|Sem=m
N	N	Gen=F|Num=D|Cas=7|Neg=N
N	N	Gen=F|Num=P|Cas=1|Neg=A
N	N	Gen=F|Num=P|Cas=1|Neg=A|Sem=E
N	N	Gen=F|Num=P|Cas=1|Neg=A|Sem=G
N	N	Gen=F|Num=P|Cas=1|Neg=A|Sem=K
N	N	Gen=F|Num=P|Cas=1|Neg=A|Sem=R
N	N	Gen=F|Num=P|Cas=1|Neg=A|Sem=Y
N	N	Gen=F|Num=P|Cas=1|Neg=A|Sem=m
N	N	Gen=F|Num=P|Cas=1|Neg=A|Var=1
N	N	Gen=F|Num=P|Cas=1|Neg=A|Var=1|Sem=G
N	N	Gen=F|Num=P|Cas=1|Neg=A|Var=1|Sem=m
N	N	Gen=F|Num=P|Cas=1|Neg=A|Var=4
N	N	Gen=F|Num=P|Cas=1|Neg=A|Var=6
N	N	Gen=F|Num=P|Cas=1|Neg=A|Var=8
N	N	Gen=F|Num=P|Cas=1|Neg=N
N	N	Gen=F|Num=P|Cas=1|Neg=N|Var=1
N	N	Gen=F|Num=P|Cas=1|Neg=N|Var=4
N	N	Gen=F|Num=P|Cas=1|Neg=N|Var=6
N	N	Gen=F|Num=P|Cas=1|Neg=N|Var=8
N	N	Gen=F|Num=P|Cas=2|Neg=A
N	N	Gen=F|Num=P|Cas=2|Neg=A|Sem=E
N	N	Gen=F|Num=P|Cas=2|Neg=A|Sem=G
N	N	Gen=F|Num=P|Cas=2|Neg=A|Sem=K
N	N	Gen=F|Num=P|Cas=2|Neg=A|Sem=R
N	N	Gen=F|Num=P|Cas=2|Neg=A|Sem=S
N	N	Gen=F|Num=P|Cas=2|Neg=A|Sem=Y
N	N	Gen=F|Num=P|Cas=2|Neg=A|Sem=m
N	N	Gen=F|Num=P|Cas=2|Neg=A|Var=1
N	N	Gen=F|Num=P|Cas=2|Neg=A|Var=6
N	N	Gen=F|Num=P|Cas=2|Neg=A|Var=8
N	N	Gen=F|Num=P|Cas=2|Neg=N
N	N	Gen=F|Num=P|Cas=2|Neg=N|Var=1
N	N	Gen=F|Num=P|Cas=2|Neg=N|Var=6
N	N	Gen=F|Num=P|Cas=2|Neg=N|Var=8
N	N	Gen=F|Num=P|Cas=3|Neg=A
N	N	Gen=F|Num=P|Cas=3|Neg=A|Sem=E
N	N	Gen=F|Num=P|Cas=3|Neg=A|Sem=G
N	N	Gen=F|Num=P|Cas=3|Neg=A|Sem=m
N	N	Gen=F|Num=P|Cas=3|Neg=A|Var=1
N	N	Gen=F|Num=P|Cas=3|Neg=A|Var=6
N	N	Gen=F|Num=P|Cas=3|Neg=A|Var=8
N	N	Gen=F|Num=P|Cas=3|Neg=N
N	N	Gen=F|Num=P|Cas=3|Neg=N|Var=1
N	N	Gen=F|Num=P|Cas=3|Neg=N|Var=6
N	N	Gen=F|Num=P|Cas=3|Neg=N|Var=8
N	N	Gen=F|Num=P|Cas=4|Neg=A
N	N	Gen=F|Num=P|Cas=4|Neg=A|Sem=E
N	N	Gen=F|Num=P|Cas=4|Neg=A|Sem=G
N	N	Gen=F|Num=P|Cas=4|Neg=A|Sem=K
N	N	Gen=F|Num=P|Cas=4|Neg=A|Sem=R
N	N	Gen=F|Num=P|Cas=4|Neg=A|Sem=m
N	N	Gen=F|Num=P|Cas=4|Neg=A|Var=1
N	N	Gen=F|Num=P|Cas=4|Neg=A|Var=4
N	N	Gen=F|Num=P|Cas=4|Neg=A|Var=6
N	N	Gen=F|Num=P|Cas=4|Neg=A|Var=8
N	N	Gen=F|Num=P|Cas=4|Neg=N
N	N	Gen=F|Num=P|Cas=4|Neg=N|Var=1
N	N	Gen=F|Num=P|Cas=4|Neg=N|Var=4
N	N	Gen=F|Num=P|Cas=4|Neg=N|Var=6
N	N	Gen=F|Num=P|Cas=4|Neg=N|Var=8
N	N	Gen=F|Num=P|Cas=5|Neg=A
N	N	Gen=F|Num=P|Cas=5|Neg=A|Sem=m
N	N	Gen=F|Num=P|Cas=5|Neg=A|Var=1
N	N	Gen=F|Num=P|Cas=5|Neg=A|Var=4
N	N	Gen=F|Num=P|Cas=5|Neg=A|Var=6
N	N	Gen=F|Num=P|Cas=5|Neg=A|Var=8
N	N	Gen=F|Num=P|Cas=5|Neg=N
N	N	Gen=F|Num=P|Cas=5|Neg=N|Var=1
N	N	Gen=F|Num=P|Cas=5|Neg=N|Var=4
N	N	Gen=F|Num=P|Cas=5|Neg=N|Var=6
N	N	Gen=F|Num=P|Cas=5|Neg=N|Var=8
N	N	Gen=F|Num=P|Cas=6|Neg=A
N	N	Gen=F|Num=P|Cas=6|Neg=A|Sem=G
N	N	Gen=F|Num=P|Cas=6|Neg=A|Sem=K
N	N	Gen=F|Num=P|Cas=6|Neg=A|Sem=R
N	N	Gen=F|Num=P|Cas=6|Neg=A|Sem=m
N	N	Gen=F|Num=P|Cas=6|Neg=A|Var=1
N	N	Gen=F|Num=P|Cas=6|Neg=A|Var=6
N	N	Gen=F|Num=P|Cas=6|Neg=A|Var=7
N	N	Gen=F|Num=P|Cas=6|Neg=A|Var=8
N	N	Gen=F|Num=P|Cas=6|Neg=N
N	N	Gen=F|Num=P|Cas=6|Neg=N|Var=1
N	N	Gen=F|Num=P|Cas=6|Neg=N|Var=6
N	N	Gen=F|Num=P|Cas=6|Neg=N|Var=7
N	N	Gen=F|Num=P|Cas=6|Neg=N|Var=8
N	N	Gen=F|Num=P|Cas=7|Neg=A
N	N	Gen=F|Num=P|Cas=7|Neg=A|Sem=G
N	N	Gen=F|Num=P|Cas=7|Neg=A|Sem=m
N	N	Gen=F|Num=P|Cas=7|Neg=A|Var=1
N	N	Gen=F|Num=P|Cas=7|Neg=A|Var=2
N	N	Gen=F|Num=P|Cas=7|Neg=A|Var=6
N	N	Gen=F|Num=P|Cas=7|Neg=A|Var=7
N	N	Gen=F|Num=P|Cas=7|Neg=A|Var=8
N	N	Gen=F|Num=P|Cas=7|Neg=N
N	N	Gen=F|Num=P|Cas=7|Neg=N|Var=1
N	N	Gen=F|Num=P|Cas=7|Neg=N|Var=2
N	N	Gen=F|Num=P|Cas=7|Neg=N|Var=6
N	N	Gen=F|Num=P|Cas=7|Neg=N|Var=7
N	N	Gen=F|Num=P|Cas=7|Neg=N|Var=8
N	N	Gen=F|Num=P|Cas=X|Neg=A
N	N	Gen=F|Num=P|Cas=X|Neg=A|Sem=G
N	N	Gen=F|Num=P|Cas=X|Neg=A|Sem=K
N	N	Gen=F|Num=P|Cas=X|Neg=A|Sem=R
N	N	Gen=F|Num=P|Cas=X|Neg=A|Sem=m
N	N	Gen=F|Num=P|Cas=X|Neg=A|Var=8
N	N	Gen=F|Num=P|Cas=X|Neg=A|Var=8|Sem=K
N	N	Gen=F|Num=P|Cas=X|Neg=A|Var=8|Sem=R
N	N	Gen=F|Num=P|Cas=X|Neg=A|Var=8|Sem=m
N	N	Gen=F|Num=P|Cas=X|Neg=A|Var=8|Sem=w
N	N	Gen=F|Num=P|Cas=X|Neg=N
N	N	Gen=F|Num=P|Cas=X|Neg=N|Var=8
N	N	Gen=F|Num=S|Cas=1|Neg=A
N	N	Gen=F|Num=S|Cas=1|Neg=A|Sem=E
N	N	Gen=F|Num=S|Cas=1|Neg=A|Sem=G
N	N	Gen=F|Num=S|Cas=1|Neg=A|Sem=K
N	N	Gen=F|Num=S|Cas=1|Neg=A|Sem=L
N	N	Gen=F|Num=S|Cas=1|Neg=A|Sem=R
N	N	Gen=F|Num=S|Cas=1|Neg=A|Sem=S
N	N	Gen=F|Num=S|Cas=1|Neg=A|Sem=U
N	N	Gen=F|Num=S|Cas=1|Neg=A|Sem=Y
N	N	Gen=F|Num=S|Cas=1|Neg=A|Sem=j
N	N	Gen=F|Num=S|Cas=1|Neg=A|Sem=m
N	N	Gen=F|Num=S|Cas=1|Neg=A|Var=1
N	N	Gen=F|Num=S|Cas=1|Neg=A|Var=1|Sem=G
N	N	Gen=F|Num=S|Cas=1|Neg=A|Var=1|Sem=K
N	N	Gen=F|Num=S|Cas=1|Neg=A|Var=1|Sem=R
N	N	Gen=F|Num=S|Cas=1|Neg=A|Var=1|Sem=Y
N	N	Gen=F|Num=S|Cas=1|Neg=A|Var=1|Sem=m
N	N	Gen=F|Num=S|Cas=1|Neg=A|Var=8
N	N	Gen=F|Num=S|Cas=1|Neg=N
N	N	Gen=F|Num=S|Cas=1|Neg=N|Var=1
N	N	Gen=F|Num=S|Cas=1|Neg=N|Var=8
N	N	Gen=F|Num=S|Cas=2|Neg=A
N	N	Gen=F|Num=S|Cas=2|Neg=A|Sem=E
N	N	Gen=F|Num=S|Cas=2|Neg=A|Sem=G
N	N	Gen=F|Num=S|Cas=2|Neg=A|Sem=K
N	N	Gen=F|Num=S|Cas=2|Neg=A|Sem=L
N	N	Gen=F|Num=S|Cas=2|Neg=A|Sem=R
N	N	Gen=F|Num=S|Cas=2|Neg=A|Sem=S
N	N	Gen=F|Num=S|Cas=2|Neg=A|Sem=U
N	N	Gen=F|Num=S|Cas=2|Neg=A|Sem=Y
N	N	Gen=F|Num=S|Cas=2|Neg=A|Sem=j
N	N	Gen=F|Num=S|Cas=2|Neg=A|Sem=m
N	N	Gen=F|Num=S|Cas=2|Neg=A|Var=1
N	N	Gen=F|Num=S|Cas=2|Neg=A|Var=1|Sem=G
N	N	Gen=F|Num=S|Cas=2|Neg=A|Var=1|Sem=m
N	N	Gen=F|Num=S|Cas=2|Neg=A|Var=2
N	N	Gen=F|Num=S|Cas=2|Neg=A|Var=6
N	N	Gen=F|Num=S|Cas=2|Neg=A|Var=8
N	N	Gen=F|Num=S|Cas=2|Neg=A|Var=8|Sem=G
N	N	Gen=F|Num=S|Cas=2|Neg=A|Var=8|Sem=m
N	N	Gen=F|Num=S|Cas=2|Neg=N
N	N	Gen=F|Num=S|Cas=2|Neg=N|Var=1
N	N	Gen=F|Num=S|Cas=2|Neg=N|Var=2
N	N	Gen=F|Num=S|Cas=2|Neg=N|Var=6
N	N	Gen=F|Num=S|Cas=2|Neg=N|Var=8
N	N	Gen=F|Num=S|Cas=3|Neg=A
N	N	Gen=F|Num=S|Cas=3|Neg=A|Sem=E
N	N	Gen=F|Num=S|Cas=3|Neg=A|Sem=G
N	N	Gen=F|Num=S|Cas=3|Neg=A|Sem=K
N	N	Gen=F|Num=S|Cas=3|Neg=A|Sem=R
N	N	Gen=F|Num=S|Cas=3|Neg=A|Sem=S
N	N	Gen=F|Num=S|Cas=3|Neg=A|Sem=Y
N	N	Gen=F|Num=S|Cas=3|Neg=A|Sem=m
N	N	Gen=F|Num=S|Cas=3|Neg=A|Var=1
N	N	Gen=F|Num=S|Cas=3|Neg=A|Var=1|Sem=G
N	N	Gen=F|Num=S|Cas=3|Neg=A|Var=1|Sem=m
N	N	Gen=F|Num=S|Cas=3|Neg=A|Var=2
N	N	Gen=F|Num=S|Cas=3|Neg=A|Var=6
N	N	Gen=F|Num=S|Cas=3|Neg=A|Var=8
N	N	Gen=F|Num=S|Cas=3|Neg=N
N	N	Gen=F|Num=S|Cas=3|Neg=N|Var=1
N	N	Gen=F|Num=S|Cas=3|Neg=N|Var=2
N	N	Gen=F|Num=S|Cas=3|Neg=N|Var=6
N	N	Gen=F|Num=S|Cas=3|Neg=N|Var=8
N	N	Gen=F|Num=S|Cas=4|Neg=A
N	N	Gen=F|Num=S|Cas=4|Neg=A|Sem=E
N	N	Gen=F|Num=S|Cas=4|Neg=A|Sem=G
N	N	Gen=F|Num=S|Cas=4|Neg=A|Sem=K
N	N	Gen=F|Num=S|Cas=4|Neg=A|Sem=R
N	N	Gen=F|Num=S|Cas=4|Neg=A|Sem=S
N	N	Gen=F|Num=S|Cas=4|Neg=A|Sem=Y
N	N	Gen=F|Num=S|Cas=4|Neg=A|Sem=j
N	N	Gen=F|Num=S|Cas=4|Neg=A|Sem=m
N	N	Gen=F|Num=S|Cas=4|Neg=A|Var=1
N	N	Gen=F|Num=S|Cas=4|Neg=A|Var=8
N	N	Gen=F|Num=S|Cas=4|Neg=N
N	N	Gen=F|Num=S|Cas=4|Neg=N|Var=1
N	N	Gen=F|Num=S|Cas=4|Neg=N|Var=8
N	N	Gen=F|Num=S|Cas=5|Neg=A
N	N	Gen=F|Num=S|Cas=5|Neg=A|Sem=G
N	N	Gen=F|Num=S|Cas=5|Neg=A|Sem=R
N	N	Gen=F|Num=S|Cas=5|Neg=A|Sem=Y
N	N	Gen=F|Num=S|Cas=5|Neg=A|Sem=m
N	N	Gen=F|Num=S|Cas=5|Neg=A|Var=1
N	N	Gen=F|Num=S|Cas=5|Neg=A|Var=8
N	N	Gen=F|Num=S|Cas=5|Neg=N
N	N	Gen=F|Num=S|Cas=5|Neg=N|Var=1
N	N	Gen=F|Num=S|Cas=5|Neg=N|Var=8
N	N	Gen=F|Num=S|Cas=6|Neg=A
N	N	Gen=F|Num=S|Cas=6|Neg=A|Sem=G
N	N	Gen=F|Num=S|Cas=6|Neg=A|Sem=K
N	N	Gen=F|Num=S|Cas=6|Neg=A|Sem=R
N	N	Gen=F|Num=S|Cas=6|Neg=A|Sem=S
N	N	Gen=F|Num=S|Cas=6|Neg=A|Sem=Y
N	N	Gen=F|Num=S|Cas=6|Neg=A|Sem=j
N	N	Gen=F|Num=S|Cas=6|Neg=A|Sem=m
N	N	Gen=F|Num=S|Cas=6|Neg=A|Var=1
N	N	Gen=F|Num=S|Cas=6|Neg=A|Var=1|Sem=G
N	N	Gen=F|Num=S|Cas=6|Neg=A|Var=1|Sem=K
N	N	Gen=F|Num=S|Cas=6|Neg=A|Var=1|Sem=m
N	N	Gen=F|Num=S|Cas=6|Neg=A|Var=6
N	N	Gen=F|Num=S|Cas=6|Neg=A|Var=8
N	N	Gen=F|Num=S|Cas=6|Neg=N
N	N	Gen=F|Num=S|Cas=6|Neg=N|Var=1
N	N	Gen=F|Num=S|Cas=6|Neg=N|Var=6
N	N	Gen=F|Num=S|Cas=6|Neg=N|Var=8
N	N	Gen=F|Num=S|Cas=7|Neg=A
N	N	Gen=F|Num=S|Cas=7|Neg=A|Sem=E
N	N	Gen=F|Num=S|Cas=7|Neg=A|Sem=G
N	N	Gen=F|Num=S|Cas=7|Neg=A|Sem=K
N	N	Gen=F|Num=S|Cas=7|Neg=A|Sem=L
N	N	Gen=F|Num=S|Cas=7|Neg=A|Sem=R
N	N	Gen=F|Num=S|Cas=7|Neg=A|Sem=S
N	N	Gen=F|Num=S|Cas=7|Neg=A|Sem=Y
N	N	Gen=F|Num=S|Cas=7|Neg=A|Sem=m
N	N	Gen=F|Num=S|Cas=7|Neg=A|Var=1
N	N	Gen=F|Num=S|Cas=7|Neg=A|Var=1|Sem=G
N	N	Gen=F|Num=S|Cas=7|Neg=A|Var=1|Sem=Y
N	N	Gen=F|Num=S|Cas=7|Neg=A|Var=1|Sem=m
N	N	Gen=F|Num=S|Cas=7|Neg=A|Var=3
N	N	Gen=F|Num=S|Cas=7|Neg=A|Var=6
N	N	Gen=F|Num=S|Cas=7|Neg=A|Var=8
N	N	Gen=F|Num=S|Cas=7|Neg=N
N	N	Gen=F|Num=S|Cas=7|Neg=N|Var=1
N	N	Gen=F|Num=S|Cas=7|Neg=N|Var=3
N	N	Gen=F|Num=S|Cas=7|Neg=N|Var=6
N	N	Gen=F|Num=S|Cas=7|Neg=N|Var=8
N	N	Gen=F|Num=S|Cas=X|Neg=A
N	N	Gen=F|Num=S|Cas=X|Neg=A|Sem=G
N	N	Gen=F|Num=S|Cas=X|Neg=A|Sem=K
N	N	Gen=F|Num=S|Cas=X|Neg=A|Sem=R
N	N	Gen=F|Num=S|Cas=X|Neg=A|Sem=S
N	N	Gen=F|Num=S|Cas=X|Neg=A|Sem=Y
N	N	Gen=F|Num=S|Cas=X|Neg=A|Sem=m
N	N	Gen=F|Num=S|Cas=X|Neg=A|Var=8
N	N	Gen=F|Num=S|Cas=X|Neg=A|Var=8|Sem=R
N	N	Gen=F|Num=S|Cas=X|Neg=A|Var=8|Sem=m
N	N	Gen=F|Num=S|Cas=X|Neg=N
N	N	Gen=F|Num=S|Cas=X|Neg=N|Var=8
N	N	Gen=F|Num=X|Cas=X|Neg=A
N	N	Gen=F|Num=X|Cas=X|Neg=A|Sem=G
N	N	Gen=F|Num=X|Cas=X|Neg=A|Sem=K
N	N	Gen=F|Num=X|Cas=X|Neg=A|Sem=R
N	N	Gen=F|Num=X|Cas=X|Neg=A|Sem=S
N	N	Gen=F|Num=X|Cas=X|Neg=A|Sem=Y
N	N	Gen=F|Num=X|Cas=X|Neg=A|Sem=m
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=1
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=2
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=8
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=G
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=K
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=R
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=S
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=Y
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=b
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=j
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=m
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=p
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=8|Sem=w
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=9
N	N	Gen=F|Num=X|Cas=X|Neg=A|Var=9|Sem=m
N	N	Gen=F|Num=X|Cas=X|Neg=N
N	N	Gen=F|Num=X|Cas=X|Neg=N|Var=1
N	N	Gen=F|Num=X|Cas=X|Neg=N|Var=2
N	N	Gen=F|Num=X|Cas=X|Neg=N|Var=8
N	N	Gen=F|Num=X|Cas=X|Neg=N|Var=9
N	N	Gen=I|Num=P|Cas=1|Neg=A
N	N	Gen=I|Num=P|Cas=1|Neg=A|Sem=G
N	N	Gen=I|Num=P|Cas=1|Neg=A|Sem=K
N	N	Gen=I|Num=P|Cas=1|Neg=A|Sem=R
N	N	Gen=I|Num=P|Cas=1|Neg=A|Sem=b
N	N	Gen=I|Num=P|Cas=1|Neg=A|Sem=m
N	N	Gen=I|Num=P|Cas=1|Neg=A|Var=1
N	N	Gen=I|Num=P|Cas=1|Neg=A|Var=2
N	N	Gen=I|Num=P|Cas=1|Neg=A|Var=3
N	N	Gen=I|Num=P|Cas=1|Neg=A|Var=6
N	N	Gen=I|Num=P|Cas=1|Neg=A|Var=8
N	N	Gen=I|Num=P|Cas=1|Neg=A|Var=9
N	N	Gen=I|Num=P|Cas=1|Neg=N
N	N	Gen=I|Num=P|Cas=1|Neg=N|Var=1
N	N	Gen=I|Num=P|Cas=1|Neg=N|Var=2
N	N	Gen=I|Num=P|Cas=1|Neg=N|Var=3
N	N	Gen=I|Num=P|Cas=1|Neg=N|Var=6
N	N	Gen=I|Num=P|Cas=1|Neg=N|Var=8
N	N	Gen=I|Num=P|Cas=1|Neg=N|Var=9
N	N	Gen=I|Num=P|Cas=2|Neg=A
N	N	Gen=I|Num=P|Cas=2|Neg=A|Sem=G
N	N	Gen=I|Num=P|Cas=2|Neg=A|Sem=H
N	N	Gen=I|Num=P|Cas=2|Neg=A|Sem=K
N	N	Gen=I|Num=P|Cas=2|Neg=A|Sem=R
N	N	Gen=I|Num=P|Cas=2|Neg=A|Sem=b
N	N	Gen=I|Num=P|Cas=2|Neg=A|Sem=m
N	N	Gen=I|Num=P|Cas=2|Neg=A|Var=1
N	N	Gen=I|Num=P|Cas=2|Neg=A|Var=1|Sem=G
N	N	Gen=I|Num=P|Cas=2|Neg=A|Var=1|Sem=m
N	N	Gen=I|Num=P|Cas=2|Neg=A|Var=2
N	N	Gen=I|Num=P|Cas=2|Neg=A|Var=3
N	N	Gen=I|Num=P|Cas=2|Neg=A|Var=6
N	N	Gen=I|Num=P|Cas=2|Neg=A|Var=8
N	N	Gen=I|Num=P|Cas=2|Neg=A|Var=9
N	N	Gen=I|Num=P|Cas=2|Neg=N
N	N	Gen=I|Num=P|Cas=2|Neg=N|Var=1
N	N	Gen=I|Num=P|Cas=2|Neg=N|Var=2
N	N	Gen=I|Num=P|Cas=2|Neg=N|Var=3
N	N	Gen=I|Num=P|Cas=2|Neg=N|Var=6
N	N	Gen=I|Num=P|Cas=2|Neg=N|Var=8
N	N	Gen=I|Num=P|Cas=2|Neg=N|Var=9
N	N	Gen=I|Num=P|Cas=3|Neg=A
N	N	Gen=I|Num=P|Cas=3|Neg=A|Sem=G
N	N	Gen=I|Num=P|Cas=3|Neg=A|Sem=R
N	N	Gen=I|Num=P|Cas=3|Neg=A|Sem=m
N	N	Gen=I|Num=P|Cas=3|Neg=A|Var=1
N	N	Gen=I|Num=P|Cas=3|Neg=A|Var=2
N	N	Gen=I|Num=P|Cas=3|Neg=A|Var=6
N	N	Gen=I|Num=P|Cas=3|Neg=A|Var=7
N	N	Gen=I|Num=P|Cas=3|Neg=A|Var=8
N	N	Gen=I|Num=P|Cas=3|Neg=A|Var=9
N	N	Gen=I|Num=P|Cas=3|Neg=N
N	N	Gen=I|Num=P|Cas=3|Neg=N|Var=1
N	N	Gen=I|Num=P|Cas=3|Neg=N|Var=2
N	N	Gen=I|Num=P|Cas=3|Neg=N|Var=6
N	N	Gen=I|Num=P|Cas=3|Neg=N|Var=7
N	N	Gen=I|Num=P|Cas=3|Neg=N|Var=8
N	N	Gen=I|Num=P|Cas=3|Neg=N|Var=9
N	N	Gen=I|Num=P|Cas=4|Neg=A
N	N	Gen=I|Num=P|Cas=4|Neg=A|Sem=G
N	N	Gen=I|Num=P|Cas=4|Neg=A|Sem=L
N	N	Gen=I|Num=P|Cas=4|Neg=A|Sem=R
N	N	Gen=I|Num=P|Cas=4|Neg=A|Sem=b
N	N	Gen=I|Num=P|Cas=4|Neg=A|Sem=m
N	N	Gen=I|Num=P|Cas=4|Neg=A|Var=1
N	N	Gen=I|Num=P|Cas=4|Neg=A|Var=2
N	N	Gen=I|Num=P|Cas=4|Neg=A|Var=3
N	N	Gen=I|Num=P|Cas=4|Neg=A|Var=6
N	N	Gen=I|Num=P|Cas=4|Neg=A|Var=8
N	N	Gen=I|Num=P|Cas=4|Neg=A|Var=9
N	N	Gen=I|Num=P|Cas=4|Neg=N
N	N	Gen=I|Num=P|Cas=4|Neg=N|Var=1
N	N	Gen=I|Num=P|Cas=4|Neg=N|Var=2
N	N	Gen=I|Num=P|Cas=4|Neg=N|Var=3
N	N	Gen=I|Num=P|Cas=4|Neg=N|Var=6
N	N	Gen=I|Num=P|Cas=4|Neg=N|Var=8
N	N	Gen=I|Num=P|Cas=4|Neg=N|Var=9
N	N	Gen=I|Num=P|Cas=5|Neg=A
N	N	Gen=I|Num=P|Cas=5|Neg=A|Var=1
N	N	Gen=I|Num=P|Cas=5|Neg=A|Var=2
N	N	Gen=I|Num=P|Cas=5|Neg=A|Var=3
N	N	Gen=I|Num=P|Cas=5|Neg=A|Var=6
N	N	Gen=I|Num=P|Cas=5|Neg=A|Var=8
N	N	Gen=I|Num=P|Cas=5|Neg=A|Var=9
N	N	Gen=I|Num=P|Cas=5|Neg=N
N	N	Gen=I|Num=P|Cas=5|Neg=N|Var=1
N	N	Gen=I|Num=P|Cas=5|Neg=N|Var=2
N	N	Gen=I|Num=P|Cas=5|Neg=N|Var=3
N	N	Gen=I|Num=P|Cas=5|Neg=N|Var=6
N	N	Gen=I|Num=P|Cas=5|Neg=N|Var=8
N	N	Gen=I|Num=P|Cas=5|Neg=N|Var=9
N	N	Gen=I|Num=P|Cas=6|Neg=A
N	N	Gen=I|Num=P|Cas=6|Neg=A|Sem=G
N	N	Gen=I|Num=P|Cas=6|Neg=A|Sem=K
N	N	Gen=I|Num=P|Cas=6|Neg=A|Sem=R
N	N	Gen=I|Num=P|Cas=6|Neg=A|Sem=b
N	N	Gen=I|Num=P|Cas=6|Neg=A|Sem=m
N	N	Gen=I|Num=P|Cas=6|Neg=A|Var=1
N	N	Gen=I|Num=P|Cas=6|Neg=A|Var=1|Sem=G
N	N	Gen=I|Num=P|Cas=6|Neg=A|Var=1|Sem=m
N	N	Gen=I|Num=P|Cas=6|Neg=A|Var=2
N	N	Gen=I|Num=P|Cas=6|Neg=A|Var=3
N	N	Gen=I|Num=P|Cas=6|Neg=A|Var=6
N	N	Gen=I|Num=P|Cas=6|Neg=A|Var=6|Sem=G
N	N	Gen=I|Num=P|Cas=6|Neg=A|Var=6|Sem=m
N	N	Gen=I|Num=P|Cas=6|Neg=A|Var=8
N	N	Gen=I|Num=P|Cas=6|Neg=A|Var=9
N	N	Gen=I|Num=P|Cas=6|Neg=N
N	N	Gen=I|Num=P|Cas=6|Neg=N|Var=1
N	N	Gen=I|Num=P|Cas=6|Neg=N|Var=2
N	N	Gen=I|Num=P|Cas=6|Neg=N|Var=3
N	N	Gen=I|Num=P|Cas=6|Neg=N|Var=6
N	N	Gen=I|Num=P|Cas=6|Neg=N|Var=8
N	N	Gen=I|Num=P|Cas=6|Neg=N|Var=9
N	N	Gen=I|Num=P|Cas=7|Neg=A
N	N	Gen=I|Num=P|Cas=7|Neg=A|Sem=G
N	N	Gen=I|Num=P|Cas=7|Neg=A|Sem=R
N	N	Gen=I|Num=P|Cas=7|Neg=A|Sem=b
N	N	Gen=I|Num=P|Cas=7|Neg=A|Sem=m
N	N	Gen=I|Num=P|Cas=7|Neg=A|Var=1
N	N	Gen=I|Num=P|Cas=7|Neg=A|Var=2
N	N	Gen=I|Num=P|Cas=7|Neg=A|Var=6
N	N	Gen=I|Num=P|Cas=7|Neg=A|Var=7
N	N	Gen=I|Num=P|Cas=7|Neg=A|Var=8
N	N	Gen=I|Num=P|Cas=7|Neg=A|Var=9
N	N	Gen=I|Num=P|Cas=7|Neg=N
N	N	Gen=I|Num=P|Cas=7|Neg=N|Var=1
N	N	Gen=I|Num=P|Cas=7|Neg=N|Var=2
N	N	Gen=I|Num=P|Cas=7|Neg=N|Var=6
N	N	Gen=I|Num=P|Cas=7|Neg=N|Var=7
N	N	Gen=I|Num=P|Cas=7|Neg=N|Var=8
N	N	Gen=I|Num=P|Cas=7|Neg=N|Var=9
N	N	Gen=I|Num=P|Cas=X|Neg=A
N	N	Gen=I|Num=P|Cas=X|Neg=A|Sem=G
N	N	Gen=I|Num=P|Cas=X|Neg=A|Sem=K
N	N	Gen=I|Num=P|Cas=X|Neg=A|Sem=R
N	N	Gen=I|Num=P|Cas=X|Neg=A|Sem=m
N	N	Gen=I|Num=P|Cas=X|Neg=A|Var=1
N	N	Gen=I|Num=P|Cas=X|Neg=A|Var=8
N	N	Gen=I|Num=P|Cas=X|Neg=A|Var=8|Sem=G
N	N	Gen=I|Num=P|Cas=X|Neg=A|Var=8|Sem=K
N	N	Gen=I|Num=P|Cas=X|Neg=A|Var=8|Sem=R
N	N	Gen=I|Num=P|Cas=X|Neg=A|Var=8|Sem=m
N	N	Gen=I|Num=P|Cas=X|Neg=N
N	N	Gen=I|Num=P|Cas=X|Neg=N|Var=1
N	N	Gen=I|Num=P|Cas=X|Neg=N|Var=8
N	N	Gen=I|Num=S|Cas=1|Neg=A
N	N	Gen=I|Num=S|Cas=1|Neg=A|Sem=E
N	N	Gen=I|Num=S|Cas=1|Neg=A|Sem=G
N	N	Gen=I|Num=S|Cas=1|Neg=A|Sem=H
N	N	Gen=I|Num=S|Cas=1|Neg=A|Sem=K
N	N	Gen=I|Num=S|Cas=1|Neg=A|Sem=L
N	N	Gen=I|Num=S|Cas=1|Neg=A|Sem=R
N	N	Gen=I|Num=S|Cas=1|Neg=A|Sem=S
N	N	Gen=I|Num=S|Cas=1|Neg=A|Sem=U
N	N	Gen=I|Num=S|Cas=1|Neg=A|Sem=Y
N	N	Gen=I|Num=S|Cas=1|Neg=A|Sem=b
N	N	Gen=I|Num=S|Cas=1|Neg=A|Sem=m
N	N	Gen=I|Num=S|Cas=1|Neg=A|Var=1
N	N	Gen=I|Num=S|Cas=1|Neg=A|Var=1|Sem=G
N	N	Gen=I|Num=S|Cas=1|Neg=A|Var=1|Sem=m
N	N	Gen=I|Num=S|Cas=1|Neg=A|Var=2
N	N	Gen=I|Num=S|Cas=1|Neg=A|Var=6
N	N	Gen=I|Num=S|Cas=1|Neg=A|Var=8
N	N	Gen=I|Num=S|Cas=1|Neg=N
N	N	Gen=I|Num=S|Cas=1|Neg=N|Var=1
N	N	Gen=I|Num=S|Cas=1|Neg=N|Var=2
N	N	Gen=I|Num=S|Cas=1|Neg=N|Var=6
N	N	Gen=I|Num=S|Cas=1|Neg=N|Var=8
N	N	Gen=I|Num=S|Cas=2|Neg=A
N	N	Gen=I|Num=S|Cas=2|Neg=A|Sem=G
N	N	Gen=I|Num=S|Cas=2|Neg=A|Sem=H
N	N	Gen=I|Num=S|Cas=2|Neg=A|Sem=K
N	N	Gen=I|Num=S|Cas=2|Neg=A|Sem=L
N	N	Gen=I|Num=S|Cas=2|Neg=A|Sem=R
N	N	Gen=I|Num=S|Cas=2|Neg=A|Sem=S
N	N	Gen=I|Num=S|Cas=2|Neg=A|Sem=Y
N	N	Gen=I|Num=S|Cas=2|Neg=A|Sem=b
N	N	Gen=I|Num=S|Cas=2|Neg=A|Sem=m
N	N	Gen=I|Num=S|Cas=2|Neg=A|Sem=u
N	N	Gen=I|Num=S|Cas=2|Neg=A|Sem=w
N	N	Gen=I|Num=S|Cas=2|Neg=A|Var=1
N	N	Gen=I|Num=S|Cas=2|Neg=A|Var=1|Sem=G
N	N	Gen=I|Num=S|Cas=2|Neg=A|Var=1|Sem=K
N	N	Gen=I|Num=S|Cas=2|Neg=A|Var=1|Sem=R
N	N	Gen=I|Num=S|Cas=2|Neg=A|Var=1|Sem=m
N	N	Gen=I|Num=S|Cas=2|Neg=A|Var=2
N	N	Gen=I|Num=S|Cas=2|Neg=A|Var=2|Sem=G
N	N	Gen=I|Num=S|Cas=2|Neg=A|Var=2|Sem=m
N	N	Gen=I|Num=S|Cas=2|Neg=A|Var=6
N	N	Gen=I|Num=S|Cas=2|Neg=A|Var=8
N	N	Gen=I|Num=S|Cas=2|Neg=A|Var=9
N	N	Gen=I|Num=S|Cas=2|Neg=N
N	N	Gen=I|Num=S|Cas=2|Neg=N|Var=1
N	N	Gen=I|Num=S|Cas=2|Neg=N|Var=2
N	N	Gen=I|Num=S|Cas=2|Neg=N|Var=6
N	N	Gen=I|Num=S|Cas=2|Neg=N|Var=8
N	N	Gen=I|Num=S|Cas=2|Neg=N|Var=9
N	N	Gen=I|Num=S|Cas=3|Neg=A
N	N	Gen=I|Num=S|Cas=3|Neg=A|Sem=E
N	N	Gen=I|Num=S|Cas=3|Neg=A|Sem=G
N	N	Gen=I|Num=S|Cas=3|Neg=A|Sem=K
N	N	Gen=I|Num=S|Cas=3|Neg=A|Sem=R
N	N	Gen=I|Num=S|Cas=3|Neg=A|Sem=Y
N	N	Gen=I|Num=S|Cas=3|Neg=A|Sem=b
N	N	Gen=I|Num=S|Cas=3|Neg=A|Sem=m
N	N	Gen=I|Num=S|Cas=3|Neg=A|Var=1
N	N	Gen=I|Num=S|Cas=3|Neg=A|Var=2
N	N	Gen=I|Num=S|Cas=3|Neg=A|Var=6
N	N	Gen=I|Num=S|Cas=3|Neg=A|Var=8
N	N	Gen=I|Num=S|Cas=3|Neg=A|Var=9
N	N	Gen=I|Num=S|Cas=3|Neg=N
N	N	Gen=I|Num=S|Cas=3|Neg=N|Var=1
N	N	Gen=I|Num=S|Cas=3|Neg=N|Var=2
N	N	Gen=I|Num=S|Cas=3|Neg=N|Var=6
N	N	Gen=I|Num=S|Cas=3|Neg=N|Var=8
N	N	Gen=I|Num=S|Cas=3|Neg=N|Var=9
N	N	Gen=I|Num=S|Cas=4|Neg=A
N	N	Gen=I|Num=S|Cas=4|Neg=A|Sem=G
N	N	Gen=I|Num=S|Cas=4|Neg=A|Sem=K
N	N	Gen=I|Num=S|Cas=4|Neg=A|Sem=L
N	N	Gen=I|Num=S|Cas=4|Neg=A|Sem=R
N	N	Gen=I|Num=S|Cas=4|Neg=A|Sem=U
N	N	Gen=I|Num=S|Cas=4|Neg=A|Sem=Y
N	N	Gen=I|Num=S|Cas=4|Neg=A|Sem=b
N	N	Gen=I|Num=S|Cas=4|Neg=A|Sem=m
N	N	Gen=I|Num=S|Cas=4|Neg=A|Sem=w
N	N	Gen=I|Num=S|Cas=4|Neg=A|Var=1
N	N	Gen=I|Num=S|Cas=4|Neg=A|Var=1|Sem=R
N	N	Gen=I|Num=S|Cas=4|Neg=A|Var=1|Sem=m
N	N	Gen=I|Num=S|Cas=4|Neg=A|Var=2
N	N	Gen=I|Num=S|Cas=4|Neg=A|Var=6
N	N	Gen=I|Num=S|Cas=4|Neg=A|Var=6|Sem=R
N	N	Gen=I|Num=S|Cas=4|Neg=A|Var=6|Sem=m
N	N	Gen=I|Num=S|Cas=4|Neg=A|Var=8
N	N	Gen=I|Num=S|Cas=4|Neg=N
N	N	Gen=I|Num=S|Cas=4|Neg=N|Var=1
N	N	Gen=I|Num=S|Cas=4|Neg=N|Var=2
N	N	Gen=I|Num=S|Cas=4|Neg=N|Var=6
N	N	Gen=I|Num=S|Cas=4|Neg=N|Var=8
N	N	Gen=I|Num=S|Cas=5|Neg=A
N	N	Gen=I|Num=S|Cas=5|Neg=A|Var=1
N	N	Gen=I|Num=S|Cas=5|Neg=A|Var=2
N	N	Gen=I|Num=S|Cas=5|Neg=A|Var=6
N	N	Gen=I|Num=S|Cas=5|Neg=A|Var=8
N	N	Gen=I|Num=S|Cas=5|Neg=A|Var=9
N	N	Gen=I|Num=S|Cas=5|Neg=N
N	N	Gen=I|Num=S|Cas=5|Neg=N|Var=1
N	N	Gen=I|Num=S|Cas=5|Neg=N|Var=2
N	N	Gen=I|Num=S|Cas=5|Neg=N|Var=6
N	N	Gen=I|Num=S|Cas=5|Neg=N|Var=8
N	N	Gen=I|Num=S|Cas=5|Neg=N|Var=9
N	N	Gen=I|Num=S|Cas=6|Neg=A
N	N	Gen=I|Num=S|Cas=6|Neg=A|Sem=G
N	N	Gen=I|Num=S|Cas=6|Neg=A|Sem=K
N	N	Gen=I|Num=S|Cas=6|Neg=A|Sem=R
N	N	Gen=I|Num=S|Cas=6|Neg=A|Sem=S
N	N	Gen=I|Num=S|Cas=6|Neg=A|Sem=Y
N	N	Gen=I|Num=S|Cas=6|Neg=A|Sem=m
N	N	Gen=I|Num=S|Cas=6|Neg=A|Sem=w
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=1
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=1|Sem=E
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=1|Sem=G
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=1|Sem=K
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=1|Sem=R
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=1|Sem=S
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=1|Sem=m
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=2
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=2|Sem=G
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=2|Sem=m
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=6
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=7
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=8
N	N	Gen=I|Num=S|Cas=6|Neg=A|Var=9
N	N	Gen=I|Num=S|Cas=6|Neg=N
N	N	Gen=I|Num=S|Cas=6|Neg=N|Var=1
N	N	Gen=I|Num=S|Cas=6|Neg=N|Var=2
N	N	Gen=I|Num=S|Cas=6|Neg=N|Var=6
N	N	Gen=I|Num=S|Cas=6|Neg=N|Var=7
N	N	Gen=I|Num=S|Cas=6|Neg=N|Var=8
N	N	Gen=I|Num=S|Cas=6|Neg=N|Var=9
N	N	Gen=I|Num=S|Cas=7|Neg=A
N	N	Gen=I|Num=S|Cas=7|Neg=A|Sem=G
N	N	Gen=I|Num=S|Cas=7|Neg=A|Sem=K
N	N	Gen=I|Num=S|Cas=7|Neg=A|Sem=R
N	N	Gen=I|Num=S|Cas=7|Neg=A|Sem=Y
N	N	Gen=I|Num=S|Cas=7|Neg=A|Sem=b
N	N	Gen=I|Num=S|Cas=7|Neg=A|Sem=m
N	N	Gen=I|Num=S|Cas=7|Neg=A|Var=1
N	N	Gen=I|Num=S|Cas=7|Neg=A|Var=2
N	N	Gen=I|Num=S|Cas=7|Neg=A|Var=6
N	N	Gen=I|Num=S|Cas=7|Neg=A|Var=8
N	N	Gen=I|Num=S|Cas=7|Neg=A|Var=9
N	N	Gen=I|Num=S|Cas=7|Neg=N
N	N	Gen=I|Num=S|Cas=7|Neg=N|Var=1
N	N	Gen=I|Num=S|Cas=7|Neg=N|Var=2
N	N	Gen=I|Num=S|Cas=7|Neg=N|Var=6
N	N	Gen=I|Num=S|Cas=7|Neg=N|Var=8
N	N	Gen=I|Num=S|Cas=7|Neg=N|Var=9
N	N	Gen=I|Num=S|Cas=X|Neg=A
N	N	Gen=I|Num=S|Cas=X|Neg=A|Sem=G
N	N	Gen=I|Num=S|Cas=X|Neg=A|Sem=K
N	N	Gen=I|Num=S|Cas=X|Neg=A|Sem=R
N	N	Gen=I|Num=S|Cas=X|Neg=A|Sem=m
N	N	Gen=I|Num=S|Cas=X|Neg=A|Var=8
N	N	Gen=I|Num=S|Cas=X|Neg=A|Var=8|Sem=K
N	N	Gen=I|Num=S|Cas=X|Neg=A|Var=8|Sem=m
N	N	Gen=I|Num=S|Cas=X|Neg=N
N	N	Gen=I|Num=S|Cas=X|Neg=N|Var=8
N	N	Gen=I|Num=X|Cas=X|Neg=A
N	N	Gen=I|Num=X|Cas=X|Neg=A|Sem=E
N	N	Gen=I|Num=X|Cas=X|Neg=A|Sem=G
N	N	Gen=I|Num=X|Cas=X|Neg=A|Sem=H
N	N	Gen=I|Num=X|Cas=X|Neg=A|Sem=K
N	N	Gen=I|Num=X|Cas=X|Neg=A|Sem=R
N	N	Gen=I|Num=X|Cas=X|Neg=A|Sem=S
N	N	Gen=I|Num=X|Cas=X|Neg=A|Sem=Y
N	N	Gen=I|Num=X|Cas=X|Neg=A|Sem=m
N	N	Gen=I|Num=X|Cas=X|Neg=A|Var=1
N	N	Gen=I|Num=X|Cas=X|Neg=A|Var=8
N	N	Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=G
N	N	Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=H
N	N	Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=K
N	N	Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=L
N	N	Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=R
N	N	Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=U
N	N	Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=b
N	N	Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=c
N	N	Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=g
N	N	Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=m
N	N	Gen=I|Num=X|Cas=X|Neg=A|Var=8|Sem=p
N	N	Gen=I|Num=X|Cas=X|Neg=N
N	N	Gen=I|Num=X|Cas=X|Neg=N|Var=1
N	N	Gen=I|Num=X|Cas=X|Neg=N|Var=8
N	N	Gen=M|Num=P|Cas=1|Neg=A
N	N	Gen=M|Num=P|Cas=1|Neg=A|Sem=E
N	N	Gen=M|Num=P|Cas=1|Neg=A|Sem=K
N	N	Gen=M|Num=P|Cas=1|Neg=A|Sem=S
N	N	Gen=M|Num=P|Cas=1|Neg=A|Sem=Y
N	N	Gen=M|Num=P|Cas=1|Neg=A|Sem=m
N	N	Gen=M|Num=P|Cas=1|Neg=A|Var=1
N	N	Gen=M|Num=P|Cas=1|Neg=A|Var=1|Sem=E
N	N	Gen=M|Num=P|Cas=1|Neg=A|Var=1|Sem=S
N	N	Gen=M|Num=P|Cas=1|Neg=A|Var=1|Sem=m
N	N	Gen=M|Num=P|Cas=1|Neg=A|Var=2
N	N	Gen=M|Num=P|Cas=1|Neg=A|Var=6
N	N	Gen=M|Num=P|Cas=1|Neg=A|Var=6|Sem=S
N	N	Gen=M|Num=P|Cas=1|Neg=A|Var=6|Sem=Y
N	N	Gen=M|Num=P|Cas=1|Neg=A|Var=6|Sem=m
N	N	Gen=M|Num=P|Cas=1|Neg=A|Var=7
N	N	Gen=M|Num=P|Cas=1|Neg=A|Var=8
N	N	Gen=M|Num=P|Cas=1|Neg=N
N	N	Gen=M|Num=P|Cas=1|Neg=N|Var=1
N	N	Gen=M|Num=P|Cas=1|Neg=N|Var=2
N	N	Gen=M|Num=P|Cas=1|Neg=N|Var=6
N	N	Gen=M|Num=P|Cas=1|Neg=N|Var=7
N	N	Gen=M|Num=P|Cas=1|Neg=N|Var=8
N	N	Gen=M|Num=P|Cas=2|Neg=A
N	N	Gen=M|Num=P|Cas=2|Neg=A|Sem=E
N	N	Gen=M|Num=P|Cas=2|Neg=A|Sem=G
N	N	Gen=M|Num=P|Cas=2|Neg=A|Sem=K
N	N	Gen=M|Num=P|Cas=2|Neg=A|Sem=S
N	N	Gen=M|Num=P|Cas=2|Neg=A|Sem=Y
N	N	Gen=M|Num=P|Cas=2|Neg=A|Sem=m
N	N	Gen=M|Num=P|Cas=2|Neg=A|Var=1
N	N	Gen=M|Num=P|Cas=2|Neg=A|Var=2
N	N	Gen=M|Num=P|Cas=2|Neg=A|Var=6
N	N	Gen=M|Num=P|Cas=2|Neg=A|Var=8
N	N	Gen=M|Num=P|Cas=2|Neg=N
N	N	Gen=M|Num=P|Cas=2|Neg=N|Var=1
N	N	Gen=M|Num=P|Cas=2|Neg=N|Var=2
N	N	Gen=M|Num=P|Cas=2|Neg=N|Var=6
N	N	Gen=M|Num=P|Cas=2|Neg=N|Var=8
N	N	Gen=M|Num=P|Cas=3|Neg=A
N	N	Gen=M|Num=P|Cas=3|Neg=A|Sem=E
N	N	Gen=M|Num=P|Cas=3|Neg=A|Sem=S
N	N	Gen=M|Num=P|Cas=3|Neg=A|Sem=m
N	N	Gen=M|Num=P|Cas=3|Neg=A|Var=1
N	N	Gen=M|Num=P|Cas=3|Neg=A|Var=2
N	N	Gen=M|Num=P|Cas=3|Neg=A|Var=6
N	N	Gen=M|Num=P|Cas=3|Neg=A|Var=7
N	N	Gen=M|Num=P|Cas=3|Neg=A|Var=8
N	N	Gen=M|Num=P|Cas=3|Neg=N
N	N	Gen=M|Num=P|Cas=3|Neg=N|Var=1
N	N	Gen=M|Num=P|Cas=3|Neg=N|Var=2
N	N	Gen=M|Num=P|Cas=3|Neg=N|Var=6
N	N	Gen=M|Num=P|Cas=3|Neg=N|Var=7
N	N	Gen=M|Num=P|Cas=3|Neg=N|Var=8
N	N	Gen=M|Num=P|Cas=4|Neg=A
N	N	Gen=M|Num=P|Cas=4|Neg=A|Sem=E
N	N	Gen=M|Num=P|Cas=4|Neg=A|Sem=S
N	N	Gen=M|Num=P|Cas=4|Neg=A|Sem=Y
N	N	Gen=M|Num=P|Cas=4|Neg=A|Sem=m
N	N	Gen=M|Num=P|Cas=4|Neg=A|Var=1
N	N	Gen=M|Num=P|Cas=4|Neg=A|Var=2
N	N	Gen=M|Num=P|Cas=4|Neg=A|Var=6
N	N	Gen=M|Num=P|Cas=4|Neg=A|Var=7
N	N	Gen=M|Num=P|Cas=4|Neg=A|Var=8
N	N	Gen=M|Num=P|Cas=4|Neg=N
N	N	Gen=M|Num=P|Cas=4|Neg=N|Var=1
N	N	Gen=M|Num=P|Cas=4|Neg=N|Var=2
N	N	Gen=M|Num=P|Cas=4|Neg=N|Var=6
N	N	Gen=M|Num=P|Cas=4|Neg=N|Var=7
N	N	Gen=M|Num=P|Cas=4|Neg=N|Var=8
N	N	Gen=M|Num=P|Cas=5|Neg=A
N	N	Gen=M|Num=P|Cas=5|Neg=A|Sem=E
N	N	Gen=M|Num=P|Cas=5|Neg=A|Sem=m
N	N	Gen=M|Num=P|Cas=5|Neg=A|Var=1
N	N	Gen=M|Num=P|Cas=5|Neg=A|Var=2
N	N	Gen=M|Num=P|Cas=5|Neg=A|Var=5
N	N	Gen=M|Num=P|Cas=5|Neg=A|Var=6
N	N	Gen=M|Num=P|Cas=5|Neg=A|Var=7
N	N	Gen=M|Num=P|Cas=5|Neg=A|Var=8
N	N	Gen=M|Num=P|Cas=5|Neg=N
N	N	Gen=M|Num=P|Cas=5|Neg=N|Var=1
N	N	Gen=M|Num=P|Cas=5|Neg=N|Var=2
N	N	Gen=M|Num=P|Cas=5|Neg=N|Var=5
N	N	Gen=M|Num=P|Cas=5|Neg=N|Var=6
N	N	Gen=M|Num=P|Cas=5|Neg=N|Var=7
N	N	Gen=M|Num=P|Cas=5|Neg=N|Var=8
N	N	Gen=M|Num=P|Cas=6|Neg=A
N	N	Gen=M|Num=P|Cas=6|Neg=A|Sem=E
N	N	Gen=M|Num=P|Cas=6|Neg=A|Sem=S
N	N	Gen=M|Num=P|Cas=6|Neg=A|Sem=Y
N	N	Gen=M|Num=P|Cas=6|Neg=A|Sem=m
N	N	Gen=M|Num=P|Cas=6|Neg=A|Var=1
N	N	Gen=M|Num=P|Cas=6|Neg=A|Var=2
N	N	Gen=M|Num=P|Cas=6|Neg=A|Var=6
N	N	Gen=M|Num=P|Cas=6|Neg=A|Var=7
N	N	Gen=M|Num=P|Cas=6|Neg=A|Var=8
N	N	Gen=M|Num=P|Cas=6|Neg=N
N	N	Gen=M|Num=P|Cas=6|Neg=N|Var=1
N	N	Gen=M|Num=P|Cas=6|Neg=N|Var=2
N	N	Gen=M|Num=P|Cas=6|Neg=N|Var=6
N	N	Gen=M|Num=P|Cas=6|Neg=N|Var=7
N	N	Gen=M|Num=P|Cas=6|Neg=N|Var=8
N	N	Gen=M|Num=P|Cas=7|Neg=A
N	N	Gen=M|Num=P|Cas=7|Neg=A|Sem=E
N	N	Gen=M|Num=P|Cas=7|Neg=A|Sem=S
N	N	Gen=M|Num=P|Cas=7|Neg=A|Sem=Y
N	N	Gen=M|Num=P|Cas=7|Neg=A|Sem=m
N	N	Gen=M|Num=P|Cas=7|Neg=A|Var=1
N	N	Gen=M|Num=P|Cas=7|Neg=A|Var=2
N	N	Gen=M|Num=P|Cas=7|Neg=A|Var=6
N	N	Gen=M|Num=P|Cas=7|Neg=A|Var=7
N	N	Gen=M|Num=P|Cas=7|Neg=A|Var=8
N	N	Gen=M|Num=P|Cas=7|Neg=N
N	N	Gen=M|Num=P|Cas=7|Neg=N|Var=1
N	N	Gen=M|Num=P|Cas=7|Neg=N|Var=2
N	N	Gen=M|Num=P|Cas=7|Neg=N|Var=6
N	N	Gen=M|Num=P|Cas=7|Neg=N|Var=7
N	N	Gen=M|Num=P|Cas=7|Neg=N|Var=8
N	N	Gen=M|Num=P|Cas=X|Neg=A
N	N	Gen=M|Num=P|Cas=X|Neg=A|Sem=K
N	N	Gen=M|Num=P|Cas=X|Neg=A|Sem=R
N	N	Gen=M|Num=P|Cas=X|Neg=A|Sem=Y
N	N	Gen=M|Num=P|Cas=X|Neg=A|Sem=m
N	N	Gen=M|Num=P|Cas=X|Neg=A|Var=8
N	N	Gen=M|Num=P|Cas=X|Neg=A|Var=8|Sem=K
N	N	Gen=M|Num=P|Cas=X|Neg=A|Var=8|Sem=m
N	N	Gen=M|Num=P|Cas=X|Neg=N
N	N	Gen=M|Num=P|Cas=X|Neg=N|Var=8
N	N	Gen=M|Num=S|Cas=1|Neg=A
N	N	Gen=M|Num=S|Cas=1|Neg=A|Sem=E
N	N	Gen=M|Num=S|Cas=1|Neg=A|Sem=G
N	N	Gen=M|Num=S|Cas=1|Neg=A|Sem=K
N	N	Gen=M|Num=S|Cas=1|Neg=A|Sem=R
N	N	Gen=M|Num=S|Cas=1|Neg=A|Sem=S
N	N	Gen=M|Num=S|Cas=1|Neg=A|Sem=Y
N	N	Gen=M|Num=S|Cas=1|Neg=A|Sem=m
N	N	Gen=M|Num=S|Cas=1|Neg=A|Sem=y
N	N	Gen=M|Num=S|Cas=1|Neg=A|Var=1
N	N	Gen=M|Num=S|Cas=1|Neg=A|Var=1|Sem=S
N	N	Gen=M|Num=S|Cas=1|Neg=A|Var=1|Sem=Y
N	N	Gen=M|Num=S|Cas=1|Neg=A|Var=1|Sem=m
N	N	Gen=M|Num=S|Cas=1|Neg=A|Var=2
N	N	Gen=M|Num=S|Cas=1|Neg=A|Var=6
N	N	Gen=M|Num=S|Cas=1|Neg=A|Var=8
N	N	Gen=M|Num=S|Cas=1|Neg=N
N	N	Gen=M|Num=S|Cas=1|Neg=N|Var=1
N	N	Gen=M|Num=S|Cas=1|Neg=N|Var=2
N	N	Gen=M|Num=S|Cas=1|Neg=N|Var=6
N	N	Gen=M|Num=S|Cas=1|Neg=N|Var=8
N	N	Gen=M|Num=S|Cas=2|Neg=A
N	N	Gen=M|Num=S|Cas=2|Neg=A|Sem=E
N	N	Gen=M|Num=S|Cas=2|Neg=A|Sem=G
N	N	Gen=M|Num=S|Cas=2|Neg=A|Sem=K
N	N	Gen=M|Num=S|Cas=2|Neg=A|Sem=S
N	N	Gen=M|Num=S|Cas=2|Neg=A|Sem=Y
N	N	Gen=M|Num=S|Cas=2|Neg=A|Sem=m
N	N	Gen=M|Num=S|Cas=2|Neg=A|Var=1
N	N	Gen=M|Num=S|Cas=2|Neg=A|Var=1|Sem=S
N	N	Gen=M|Num=S|Cas=2|Neg=A|Var=1|Sem=Y
N	N	Gen=M|Num=S|Cas=2|Neg=A|Var=1|Sem=m
N	N	Gen=M|Num=S|Cas=2|Neg=A|Var=2
N	N	Gen=M|Num=S|Cas=2|Neg=A|Var=2|Sem=S
N	N	Gen=M|Num=S|Cas=2|Neg=A|Var=2|Sem=Y
N	N	Gen=M|Num=S|Cas=2|Neg=A|Var=2|Sem=m
N	N	Gen=M|Num=S|Cas=2|Neg=A|Var=3
N	N	Gen=M|Num=S|Cas=2|Neg=A|Var=6
N	N	Gen=M|Num=S|Cas=2|Neg=A|Var=7
N	N	Gen=M|Num=S|Cas=2|Neg=A|Var=8
N	N	Gen=M|Num=S|Cas=2|Neg=N
N	N	Gen=M|Num=S|Cas=2|Neg=N|Var=1
N	N	Gen=M|Num=S|Cas=2|Neg=N|Var=2
N	N	Gen=M|Num=S|Cas=2|Neg=N|Var=3
N	N	Gen=M|Num=S|Cas=2|Neg=N|Var=6
N	N	Gen=M|Num=S|Cas=2|Neg=N|Var=7
N	N	Gen=M|Num=S|Cas=2|Neg=N|Var=8
N	N	Gen=M|Num=S|Cas=3|Neg=A
N	N	Gen=M|Num=S|Cas=3|Neg=A|Sem=E
N	N	Gen=M|Num=S|Cas=3|Neg=A|Sem=G
N	N	Gen=M|Num=S|Cas=3|Neg=A|Sem=S
N	N	Gen=M|Num=S|Cas=3|Neg=A|Sem=Y
N	N	Gen=M|Num=S|Cas=3|Neg=A|Sem=m
N	N	Gen=M|Num=S|Cas=3|Neg=A|Var=1
N	N	Gen=M|Num=S|Cas=3|Neg=A|Var=1|Sem=E
N	N	Gen=M|Num=S|Cas=3|Neg=A|Var=1|Sem=S
N	N	Gen=M|Num=S|Cas=3|Neg=A|Var=1|Sem=Y
N	N	Gen=M|Num=S|Cas=3|Neg=A|Var=1|Sem=m
N	N	Gen=M|Num=S|Cas=3|Neg=A|Var=2
N	N	Gen=M|Num=S|Cas=3|Neg=A|Var=3
N	N	Gen=M|Num=S|Cas=3|Neg=A|Var=6
N	N	Gen=M|Num=S|Cas=3|Neg=A|Var=7
N	N	Gen=M|Num=S|Cas=3|Neg=A|Var=8
N	N	Gen=M|Num=S|Cas=3|Neg=N
N	N	Gen=M|Num=S|Cas=3|Neg=N|Var=1
N	N	Gen=M|Num=S|Cas=3|Neg=N|Var=2
N	N	Gen=M|Num=S|Cas=3|Neg=N|Var=3
N	N	Gen=M|Num=S|Cas=3|Neg=N|Var=6
N	N	Gen=M|Num=S|Cas=3|Neg=N|Var=7
N	N	Gen=M|Num=S|Cas=3|Neg=N|Var=8
N	N	Gen=M|Num=S|Cas=4|Neg=A
N	N	Gen=M|Num=S|Cas=4|Neg=A|Sem=E
N	N	Gen=M|Num=S|Cas=4|Neg=A|Sem=G
N	N	Gen=M|Num=S|Cas=4|Neg=A|Sem=K
N	N	Gen=M|Num=S|Cas=4|Neg=A|Sem=R
N	N	Gen=M|Num=S|Cas=4|Neg=A|Sem=S
N	N	Gen=M|Num=S|Cas=4|Neg=A|Sem=Y
N	N	Gen=M|Num=S|Cas=4|Neg=A|Sem=m
N	N	Gen=M|Num=S|Cas=4|Neg=A|Var=1
N	N	Gen=M|Num=S|Cas=4|Neg=A|Var=1|Sem=Y
N	N	Gen=M|Num=S|Cas=4|Neg=A|Var=1|Sem=m
N	N	Gen=M|Num=S|Cas=4|Neg=A|Var=2
N	N	Gen=M|Num=S|Cas=4|Neg=A|Var=6
N	N	Gen=M|Num=S|Cas=4|Neg=A|Var=7
N	N	Gen=M|Num=S|Cas=4|Neg=A|Var=8
N	N	Gen=M|Num=S|Cas=4|Neg=N
N	N	Gen=M|Num=S|Cas=4|Neg=N|Var=1
N	N	Gen=M|Num=S|Cas=4|Neg=N|Var=2
N	N	Gen=M|Num=S|Cas=4|Neg=N|Var=6
N	N	Gen=M|Num=S|Cas=4|Neg=N|Var=7
N	N	Gen=M|Num=S|Cas=4|Neg=N|Var=8
N	N	Gen=M|Num=S|Cas=5|Neg=A
N	N	Gen=M|Num=S|Cas=5|Neg=A|Sem=S
N	N	Gen=M|Num=S|Cas=5|Neg=A|Sem=Y
N	N	Gen=M|Num=S|Cas=5|Neg=A|Sem=m
N	N	Gen=M|Num=S|Cas=5|Neg=A|Var=1
N	N	Gen=M|Num=S|Cas=5|Neg=A|Var=2
N	N	Gen=M|Num=S|Cas=5|Neg=A|Var=4
N	N	Gen=M|Num=S|Cas=5|Neg=A|Var=5
N	N	Gen=M|Num=S|Cas=5|Neg=A|Var=6
N	N	Gen=M|Num=S|Cas=5|Neg=A|Var=7
N	N	Gen=M|Num=S|Cas=5|Neg=A|Var=8
N	N	Gen=M|Num=S|Cas=5|Neg=N
N	N	Gen=M|Num=S|Cas=5|Neg=N|Var=1
N	N	Gen=M|Num=S|Cas=5|Neg=N|Var=2
N	N	Gen=M|Num=S|Cas=5|Neg=N|Var=4
N	N	Gen=M|Num=S|Cas=5|Neg=N|Var=5
N	N	Gen=M|Num=S|Cas=5|Neg=N|Var=6
N	N	Gen=M|Num=S|Cas=5|Neg=N|Var=7
N	N	Gen=M|Num=S|Cas=5|Neg=N|Var=8
N	N	Gen=M|Num=S|Cas=6|Neg=A
N	N	Gen=M|Num=S|Cas=6|Neg=A|Sem=E
N	N	Gen=M|Num=S|Cas=6|Neg=A|Sem=S
N	N	Gen=M|Num=S|Cas=6|Neg=A|Sem=Y
N	N	Gen=M|Num=S|Cas=6|Neg=A|Sem=m
N	N	Gen=M|Num=S|Cas=6|Neg=A|Var=1
N	N	Gen=M|Num=S|Cas=6|Neg=A|Var=1|Sem=G
N	N	Gen=M|Num=S|Cas=6|Neg=A|Var=1|Sem=S
N	N	Gen=M|Num=S|Cas=6|Neg=A|Var=1|Sem=Y
N	N	Gen=M|Num=S|Cas=6|Neg=A|Var=1|Sem=m
N	N	Gen=M|Num=S|Cas=6|Neg=A|Var=2
N	N	Gen=M|Num=S|Cas=6|Neg=A|Var=3
N	N	Gen=M|Num=S|Cas=6|Neg=A|Var=6
N	N	Gen=M|Num=S|Cas=6|Neg=A|Var=7
N	N	Gen=M|Num=S|Cas=6|Neg=A|Var=8
N	N	Gen=M|Num=S|Cas=6|Neg=N
N	N	Gen=M|Num=S|Cas=6|Neg=N|Var=1
N	N	Gen=M|Num=S|Cas=6|Neg=N|Var=2
N	N	Gen=M|Num=S|Cas=6|Neg=N|Var=3
N	N	Gen=M|Num=S|Cas=6|Neg=N|Var=6
N	N	Gen=M|Num=S|Cas=6|Neg=N|Var=7
N	N	Gen=M|Num=S|Cas=6|Neg=N|Var=8
N	N	Gen=M|Num=S|Cas=7|Neg=A
N	N	Gen=M|Num=S|Cas=7|Neg=A|Sem=E
N	N	Gen=M|Num=S|Cas=7|Neg=A|Sem=G
N	N	Gen=M|Num=S|Cas=7|Neg=A|Sem=K
N	N	Gen=M|Num=S|Cas=7|Neg=A|Sem=R
N	N	Gen=M|Num=S|Cas=7|Neg=A|Sem=S
N	N	Gen=M|Num=S|Cas=7|Neg=A|Sem=Y
N	N	Gen=M|Num=S|Cas=7|Neg=A|Sem=m
N	N	Gen=M|Num=S|Cas=7|Neg=A|Var=1
N	N	Gen=M|Num=S|Cas=7|Neg=A|Var=2
N	N	Gen=M|Num=S|Cas=7|Neg=A|Var=6
N	N	Gen=M|Num=S|Cas=7|Neg=A|Var=8
N	N	Gen=M|Num=S|Cas=7|Neg=N
N	N	Gen=M|Num=S|Cas=7|Neg=N|Var=1
N	N	Gen=M|Num=S|Cas=7|Neg=N|Var=2
N	N	Gen=M|Num=S|Cas=7|Neg=N|Var=6
N	N	Gen=M|Num=S|Cas=7|Neg=N|Var=8
N	N	Gen=M|Num=S|Cas=X|Neg=A
N	N	Gen=M|Num=S|Cas=X|Neg=A|Sem=G
N	N	Gen=M|Num=S|Cas=X|Neg=A|Sem=K
N	N	Gen=M|Num=S|Cas=X|Neg=A|Sem=S
N	N	Gen=M|Num=S|Cas=X|Neg=A|Sem=Y
N	N	Gen=M|Num=S|Cas=X|Neg=A|Sem=m
N	N	Gen=M|Num=S|Cas=X|Neg=A|Var=1
N	N	Gen=M|Num=S|Cas=X|Neg=A|Var=8
N	N	Gen=M|Num=S|Cas=X|Neg=N
N	N	Gen=M|Num=S|Cas=X|Neg=N|Var=1
N	N	Gen=M|Num=S|Cas=X|Neg=N|Var=8
N	N	Gen=M|Num=X|Cas=X|Neg=A
N	N	Gen=M|Num=X|Cas=X|Neg=A|Sem=E
N	N	Gen=M|Num=X|Cas=X|Neg=A|Sem=G
N	N	Gen=M|Num=X|Cas=X|Neg=A|Sem=K
N	N	Gen=M|Num=X|Cas=X|Neg=A|Sem=R
N	N	Gen=M|Num=X|Cas=X|Neg=A|Sem=S
N	N	Gen=M|Num=X|Cas=X|Neg=A|Sem=Y
N	N	Gen=M|Num=X|Cas=X|Neg=A|Sem=m
N	N	Gen=M|Num=X|Cas=X|Neg=A|Var=1
N	N	Gen=M|Num=X|Cas=X|Neg=A|Var=8
N	N	Gen=M|Num=X|Cas=X|Neg=A|Var=8|Sem=G
N	N	Gen=M|Num=X|Cas=X|Neg=A|Var=8|Sem=K
N	N	Gen=M|Num=X|Cas=X|Neg=A|Var=8|Sem=S
N	N	Gen=M|Num=X|Cas=X|Neg=A|Var=8|Sem=Y
N	N	Gen=M|Num=X|Cas=X|Neg=A|Var=8|Sem=m
N	N	Gen=M|Num=X|Cas=X|Neg=N
N	N	Gen=M|Num=X|Cas=X|Neg=N|Var=1
N	N	Gen=M|Num=X|Cas=X|Neg=N|Var=8
N	N	Gen=N|Num=P|Cas=1|Neg=A
N	N	Gen=N|Num=P|Cas=1|Neg=A|Sem=G
N	N	Gen=N|Num=P|Cas=1|Neg=A|Sem=K
N	N	Gen=N|Num=P|Cas=1|Neg=A|Sem=R
N	N	Gen=N|Num=P|Cas=1|Neg=A|Sem=U
N	N	Gen=N|Num=P|Cas=1|Neg=A|Sem=m
N	N	Gen=N|Num=P|Cas=1|Neg=A|Var=1
N	N	Gen=N|Num=P|Cas=1|Neg=A|Var=2
N	N	Gen=N|Num=P|Cas=1|Neg=A|Var=3
N	N	Gen=N|Num=P|Cas=1|Neg=A|Var=6
N	N	Gen=N|Num=P|Cas=1|Neg=A|Var=8
N	N	Gen=N|Num=P|Cas=1|Neg=N
N	N	Gen=N|Num=P|Cas=1|Neg=N|Var=1
N	N	Gen=N|Num=P|Cas=1|Neg=N|Var=2
N	N	Gen=N|Num=P|Cas=1|Neg=N|Var=3
N	N	Gen=N|Num=P|Cas=1|Neg=N|Var=6
N	N	Gen=N|Num=P|Cas=1|Neg=N|Var=8
N	N	Gen=N|Num=P|Cas=2|Neg=A
N	N	Gen=N|Num=P|Cas=2|Neg=A|Sem=G
N	N	Gen=N|Num=P|Cas=2|Neg=A|Sem=K
N	N	Gen=N|Num=P|Cas=2|Neg=A|Sem=R
N	N	Gen=N|Num=P|Cas=2|Neg=A|Sem=U
N	N	Gen=N|Num=P|Cas=2|Neg=A|Sem=m
N	N	Gen=N|Num=P|Cas=2|Neg=A|Var=1
N	N	Gen=N|Num=P|Cas=2|Neg=A|Var=2
N	N	Gen=N|Num=P|Cas=2|Neg=A|Var=3
N	N	Gen=N|Num=P|Cas=2|Neg=A|Var=6
N	N	Gen=N|Num=P|Cas=2|Neg=A|Var=8
N	N	Gen=N|Num=P|Cas=2|Neg=N
N	N	Gen=N|Num=P|Cas=2|Neg=N|Var=1
N	N	Gen=N|Num=P|Cas=2|Neg=N|Var=2
N	N	Gen=N|Num=P|Cas=2|Neg=N|Var=3
N	N	Gen=N|Num=P|Cas=2|Neg=N|Var=6
N	N	Gen=N|Num=P|Cas=2|Neg=N|Var=8
N	N	Gen=N|Num=P|Cas=3|Neg=A
N	N	Gen=N|Num=P|Cas=3|Neg=A|Sem=m
N	N	Gen=N|Num=P|Cas=3|Neg=A|Var=1
N	N	Gen=N|Num=P|Cas=3|Neg=A|Var=2
N	N	Gen=N|Num=P|Cas=3|Neg=A|Var=3
N	N	Gen=N|Num=P|Cas=3|Neg=A|Var=6
N	N	Gen=N|Num=P|Cas=3|Neg=A|Var=7
N	N	Gen=N|Num=P|Cas=3|Neg=A|Var=8
N	N	Gen=N|Num=P|Cas=3|Neg=N
N	N	Gen=N|Num=P|Cas=3|Neg=N|Var=1
N	N	Gen=N|Num=P|Cas=3|Neg=N|Var=2
N	N	Gen=N|Num=P|Cas=3|Neg=N|Var=3
N	N	Gen=N|Num=P|Cas=3|Neg=N|Var=6
N	N	Gen=N|Num=P|Cas=3|Neg=N|Var=7
N	N	Gen=N|Num=P|Cas=3|Neg=N|Var=8
N	N	Gen=N|Num=P|Cas=4|Neg=A
N	N	Gen=N|Num=P|Cas=4|Neg=A|Sem=m
N	N	Gen=N|Num=P|Cas=4|Neg=A|Var=1
N	N	Gen=N|Num=P|Cas=4|Neg=A|Var=2
N	N	Gen=N|Num=P|Cas=4|Neg=A|Var=3
N	N	Gen=N|Num=P|Cas=4|Neg=A|Var=6
N	N	Gen=N|Num=P|Cas=4|Neg=A|Var=8
N	N	Gen=N|Num=P|Cas=4|Neg=N
N	N	Gen=N|Num=P|Cas=4|Neg=N|Var=1
N	N	Gen=N|Num=P|Cas=4|Neg=N|Var=2
N	N	Gen=N|Num=P|Cas=4|Neg=N|Var=3
N	N	Gen=N|Num=P|Cas=4|Neg=N|Var=6
N	N	Gen=N|Num=P|Cas=4|Neg=N|Var=8
N	N	Gen=N|Num=P|Cas=5|Neg=A
N	N	Gen=N|Num=P|Cas=5|Neg=A|Sem=m
N	N	Gen=N|Num=P|Cas=5|Neg=A|Var=1
N	N	Gen=N|Num=P|Cas=5|Neg=A|Var=2
N	N	Gen=N|Num=P|Cas=5|Neg=A|Var=3
N	N	Gen=N|Num=P|Cas=5|Neg=A|Var=6
N	N	Gen=N|Num=P|Cas=5|Neg=A|Var=8
N	N	Gen=N|Num=P|Cas=5|Neg=N
N	N	Gen=N|Num=P|Cas=5|Neg=N|Var=1
N	N	Gen=N|Num=P|Cas=5|Neg=N|Var=2
N	N	Gen=N|Num=P|Cas=5|Neg=N|Var=3
N	N	Gen=N|Num=P|Cas=5|Neg=N|Var=6
N	N	Gen=N|Num=P|Cas=5|Neg=N|Var=8
N	N	Gen=N|Num=P|Cas=6|Neg=A
N	N	Gen=N|Num=P|Cas=6|Neg=A|Sem=m
N	N	Gen=N|Num=P|Cas=6|Neg=A|Var=1
N	N	Gen=N|Num=P|Cas=6|Neg=A|Var=2
N	N	Gen=N|Num=P|Cas=6|Neg=A|Var=3
N	N	Gen=N|Num=P|Cas=6|Neg=A|Var=6
N	N	Gen=N|Num=P|Cas=6|Neg=A|Var=8
N	N	Gen=N|Num=P|Cas=6|Neg=N
N	N	Gen=N|Num=P|Cas=6|Neg=N|Var=1
N	N	Gen=N|Num=P|Cas=6|Neg=N|Var=2
N	N	Gen=N|Num=P|Cas=6|Neg=N|Var=3
N	N	Gen=N|Num=P|Cas=6|Neg=N|Var=6
N	N	Gen=N|Num=P|Cas=6|Neg=N|Var=8
N	N	Gen=N|Num=P|Cas=7|Neg=A
N	N	Gen=N|Num=P|Cas=7|Neg=A|Sem=G
N	N	Gen=N|Num=P|Cas=7|Neg=A|Sem=m
N	N	Gen=N|Num=P|Cas=7|Neg=A|Var=1
N	N	Gen=N|Num=P|Cas=7|Neg=A|Var=2
N	N	Gen=N|Num=P|Cas=7|Neg=A|Var=3
N	N	Gen=N|Num=P|Cas=7|Neg=A|Var=6
N	N	Gen=N|Num=P|Cas=7|Neg=A|Var=7
N	N	Gen=N|Num=P|Cas=7|Neg=A|Var=8
N	N	Gen=N|Num=P|Cas=7|Neg=N
N	N	Gen=N|Num=P|Cas=7|Neg=N|Var=1
N	N	Gen=N|Num=P|Cas=7|Neg=N|Var=2
N	N	Gen=N|Num=P|Cas=7|Neg=N|Var=3
N	N	Gen=N|Num=P|Cas=7|Neg=N|Var=6
N	N	Gen=N|Num=P|Cas=7|Neg=N|Var=7
N	N	Gen=N|Num=P|Cas=7|Neg=N|Var=8
N	N	Gen=N|Num=P|Cas=X|Neg=A
N	N	Gen=N|Num=P|Cas=X|Neg=A|Sem=K
N	N	Gen=N|Num=P|Cas=X|Neg=A|Sem=m
N	N	Gen=N|Num=P|Cas=X|Neg=A|Var=1
N	N	Gen=N|Num=P|Cas=X|Neg=A|Var=8
N	N	Gen=N|Num=P|Cas=X|Neg=A|Var=8|Sem=K
N	N	Gen=N|Num=P|Cas=X|Neg=A|Var=8|Sem=m
N	N	Gen=N|Num=P|Cas=X|Neg=N
N	N	Gen=N|Num=P|Cas=X|Neg=N|Var=1
N	N	Gen=N|Num=P|Cas=X|Neg=N|Var=8
N	N	Gen=N|Num=S|Cas=1|Neg=A
N	N	Gen=N|Num=S|Cas=1|Neg=A|Sem=G
N	N	Gen=N|Num=S|Cas=1|Neg=A|Sem=K
N	N	Gen=N|Num=S|Cas=1|Neg=A|Sem=R
N	N	Gen=N|Num=S|Cas=1|Neg=A|Sem=S
N	N	Gen=N|Num=S|Cas=1|Neg=A|Sem=U
N	N	Gen=N|Num=S|Cas=1|Neg=A|Sem=Y
N	N	Gen=N|Num=S|Cas=1|Neg=A|Sem=m
N	N	Gen=N|Num=S|Cas=1|Neg=A|Var=1
N	N	Gen=N|Num=S|Cas=1|Neg=A|Var=2
N	N	Gen=N|Num=S|Cas=1|Neg=A|Var=6
N	N	Gen=N|Num=S|Cas=1|Neg=A|Var=8
N	N	Gen=N|Num=S|Cas=1|Neg=N
N	N	Gen=N|Num=S|Cas=1|Neg=N|Var=1
N	N	Gen=N|Num=S|Cas=1|Neg=N|Var=2
N	N	Gen=N|Num=S|Cas=1|Neg=N|Var=6
N	N	Gen=N|Num=S|Cas=1|Neg=N|Var=8
N	N	Gen=N|Num=S|Cas=2|Neg=A
N	N	Gen=N|Num=S|Cas=2|Neg=A|Sem=G
N	N	Gen=N|Num=S|Cas=2|Neg=A|Sem=H
N	N	Gen=N|Num=S|Cas=2|Neg=A|Sem=K
N	N	Gen=N|Num=S|Cas=2|Neg=A|Sem=R
N	N	Gen=N|Num=S|Cas=2|Neg=A|Sem=Y
N	N	Gen=N|Num=S|Cas=2|Neg=A|Sem=m
N	N	Gen=N|Num=S|Cas=2|Neg=A|Var=1
N	N	Gen=N|Num=S|Cas=2|Neg=A|Var=2
N	N	Gen=N|Num=S|Cas=2|Neg=A|Var=3
N	N	Gen=N|Num=S|Cas=2|Neg=A|Var=6
N	N	Gen=N|Num=S|Cas=2|Neg=A|Var=8
N	N	Gen=N|Num=S|Cas=2|Neg=N
N	N	Gen=N|Num=S|Cas=2|Neg=N|Var=1
N	N	Gen=N|Num=S|Cas=2|Neg=N|Var=2
N	N	Gen=N|Num=S|Cas=2|Neg=N|Var=3
N	N	Gen=N|Num=S|Cas=2|Neg=N|Var=6
N	N	Gen=N|Num=S|Cas=2|Neg=N|Var=8
N	N	Gen=N|Num=S|Cas=3|Neg=A
N	N	Gen=N|Num=S|Cas=3|Neg=A|Sem=G
N	N	Gen=N|Num=S|Cas=3|Neg=A|Sem=K
N	N	Gen=N|Num=S|Cas=3|Neg=A|Sem=Y
N	N	Gen=N|Num=S|Cas=3|Neg=A|Sem=m
N	N	Gen=N|Num=S|Cas=3|Neg=A|Var=1
N	N	Gen=N|Num=S|Cas=3|Neg=A|Var=2
N	N	Gen=N|Num=S|Cas=3|Neg=A|Var=3
N	N	Gen=N|Num=S|Cas=3|Neg=A|Var=6
N	N	Gen=N|Num=S|Cas=3|Neg=A|Var=8
N	N	Gen=N|Num=S|Cas=3|Neg=N
N	N	Gen=N|Num=S|Cas=3|Neg=N|Var=1
N	N	Gen=N|Num=S|Cas=3|Neg=N|Var=2
N	N	Gen=N|Num=S|Cas=3|Neg=N|Var=3
N	N	Gen=N|Num=S|Cas=3|Neg=N|Var=6
N	N	Gen=N|Num=S|Cas=3|Neg=N|Var=8
N	N	Gen=N|Num=S|Cas=4|Neg=A
N	N	Gen=N|Num=S|Cas=4|Neg=A|Sem=G
N	N	Gen=N|Num=S|Cas=4|Neg=A|Sem=K
N	N	Gen=N|Num=S|Cas=4|Neg=A|Sem=R
N	N	Gen=N|Num=S|Cas=4|Neg=A|Sem=m
N	N	Gen=N|Num=S|Cas=4|Neg=A|Sem=o
N	N	Gen=N|Num=S|Cas=4|Neg=A|Var=1
N	N	Gen=N|Num=S|Cas=4|Neg=A|Var=2
N	N	Gen=N|Num=S|Cas=4|Neg=A|Var=6
N	N	Gen=N|Num=S|Cas=4|Neg=A|Var=8
N	N	Gen=N|Num=S|Cas=4|Neg=N
N	N	Gen=N|Num=S|Cas=4|Neg=N|Var=1
N	N	Gen=N|Num=S|Cas=4|Neg=N|Var=2
N	N	Gen=N|Num=S|Cas=4|Neg=N|Var=6
N	N	Gen=N|Num=S|Cas=4|Neg=N|Var=8
N	N	Gen=N|Num=S|Cas=5|Neg=A
N	N	Gen=N|Num=S|Cas=5|Neg=A|Sem=G
N	N	Gen=N|Num=S|Cas=5|Neg=A|Sem=m
N	N	Gen=N|Num=S|Cas=5|Neg=A|Var=1
N	N	Gen=N|Num=S|Cas=5|Neg=A|Var=2
N	N	Gen=N|Num=S|Cas=5|Neg=A|Var=6
N	N	Gen=N|Num=S|Cas=5|Neg=A|Var=8
N	N	Gen=N|Num=S|Cas=5|Neg=N
N	N	Gen=N|Num=S|Cas=5|Neg=N|Var=1
N	N	Gen=N|Num=S|Cas=5|Neg=N|Var=2
N	N	Gen=N|Num=S|Cas=5|Neg=N|Var=6
N	N	Gen=N|Num=S|Cas=5|Neg=N|Var=8
N	N	Gen=N|Num=S|Cas=6|Neg=A
N	N	Gen=N|Num=S|Cas=6|Neg=A|Sem=G
N	N	Gen=N|Num=S|Cas=6|Neg=A|Sem=K
N	N	Gen=N|Num=S|Cas=6|Neg=A|Sem=R
N	N	Gen=N|Num=S|Cas=6|Neg=A|Sem=S
N	N	Gen=N|Num=S|Cas=6|Neg=A|Sem=U
N	N	Gen=N|Num=S|Cas=6|Neg=A|Sem=Y
N	N	Gen=N|Num=S|Cas=6|Neg=A|Sem=m
N	N	Gen=N|Num=S|Cas=6|Neg=A|Var=1
N	N	Gen=N|Num=S|Cas=6|Neg=A|Var=1|Sem=G
N	N	Gen=N|Num=S|Cas=6|Neg=A|Var=1|Sem=m
N	N	Gen=N|Num=S|Cas=6|Neg=A|Var=2
N	N	Gen=N|Num=S|Cas=6|Neg=A|Var=6
N	N	Gen=N|Num=S|Cas=6|Neg=A|Var=7
N	N	Gen=N|Num=S|Cas=6|Neg=A|Var=8
N	N	Gen=N|Num=S|Cas=6|Neg=N
N	N	Gen=N|Num=S|Cas=6|Neg=N|Var=1
N	N	Gen=N|Num=S|Cas=6|Neg=N|Var=2
N	N	Gen=N|Num=S|Cas=6|Neg=N|Var=6
N	N	Gen=N|Num=S|Cas=6|Neg=N|Var=7
N	N	Gen=N|Num=S|Cas=6|Neg=N|Var=8
N	N	Gen=N|Num=S|Cas=7|Neg=A
N	N	Gen=N|Num=S|Cas=7|Neg=A|Sem=G
N	N	Gen=N|Num=S|Cas=7|Neg=A|Sem=K
N	N	Gen=N|Num=S|Cas=7|Neg=A|Sem=S
N	N	Gen=N|Num=S|Cas=7|Neg=A|Sem=m
N	N	Gen=N|Num=S|Cas=7|Neg=A|Var=1
N	N	Gen=N|Num=S|Cas=7|Neg=A|Var=2
N	N	Gen=N|Num=S|Cas=7|Neg=A|Var=6
N	N	Gen=N|Num=S|Cas=7|Neg=A|Var=8
N	N	Gen=N|Num=S|Cas=7|Neg=A|Var=8|Sem=G
N	N	Gen=N|Num=S|Cas=7|Neg=A|Var=8|Sem=m
N	N	Gen=N|Num=S|Cas=7|Neg=N
N	N	Gen=N|Num=S|Cas=7|Neg=N|Var=1
N	N	Gen=N|Num=S|Cas=7|Neg=N|Var=2
N	N	Gen=N|Num=S|Cas=7|Neg=N|Var=6
N	N	Gen=N|Num=S|Cas=7|Neg=N|Var=8
N	N	Gen=N|Num=S|Cas=X|Neg=A
N	N	Gen=N|Num=S|Cas=X|Neg=A|Sem=E
N	N	Gen=N|Num=S|Cas=X|Neg=A|Sem=G
N	N	Gen=N|Num=S|Cas=X|Neg=A|Sem=K
N	N	Gen=N|Num=S|Cas=X|Neg=A|Sem=m
N	N	Gen=N|Num=S|Cas=X|Neg=A|Var=8
N	N	Gen=N|Num=S|Cas=X|Neg=N
N	N	Gen=N|Num=S|Cas=X|Neg=N|Var=8
N	N	Gen=N|Num=X|Cas=X|Neg=A
N	N	Gen=N|Num=X|Cas=X|Neg=A|Sem=G
N	N	Gen=N|Num=X|Cas=X|Neg=A|Sem=K
N	N	Gen=N|Num=X|Cas=X|Neg=A|Sem=R
N	N	Gen=N|Num=X|Cas=X|Neg=A|Sem=S
N	N	Gen=N|Num=X|Cas=X|Neg=A|Sem=Y
N	N	Gen=N|Num=X|Cas=X|Neg=A|Sem=m
N	N	Gen=N|Num=X|Cas=X|Neg=A|Sem=w
N	N	Gen=N|Num=X|Cas=X|Neg=A|Var=1
N	N	Gen=N|Num=X|Cas=X|Neg=A|Var=2
N	N	Gen=N|Num=X|Cas=X|Neg=A|Var=2|Sem=G
N	N	Gen=N|Num=X|Cas=X|Neg=A|Var=2|Sem=m
N	N	Gen=N|Num=X|Cas=X|Neg=A|Var=8
N	N	Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=G
N	N	Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=K
N	N	Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=R
N	N	Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=g
N	N	Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=m
N	N	Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=w
N	N	Gen=N|Num=X|Cas=X|Neg=A|Var=8|Sem=y
N	N	Gen=N|Num=X|Cas=X|Neg=A|Var=9
N	N	Gen=N|Num=X|Cas=X|Neg=N
N	N	Gen=N|Num=X|Cas=X|Neg=N|Var=1
N	N	Gen=N|Num=X|Cas=X|Neg=N|Var=2
N	N	Gen=N|Num=X|Cas=X|Neg=N|Var=8
N	N	Gen=N|Num=X|Cas=X|Neg=N|Var=9
N	N	Gen=X|Num=P|Cas=X|Neg=A
N	N	Gen=X|Num=P|Cas=X|Neg=A|Sem=K
N	N	Gen=X|Num=P|Cas=X|Neg=A|Sem=m
N	N	Gen=X|Num=P|Cas=X|Neg=N
N	N	Gen=X|Num=S|Cas=X|Neg=A
N	N	Gen=X|Num=S|Cas=X|Neg=A|Sem=K
N	N	Gen=X|Num=S|Cas=X|Neg=A|Sem=R
N	N	Gen=X|Num=S|Cas=X|Neg=A|Sem=m
N	N	Gen=X|Num=S|Cas=X|Neg=N
N	N	Gen=X|Num=X|Cas=X|Neg=A
N	N	Gen=X|Num=X|Cas=X|Neg=A|Sem=G
N	N	Gen=X|Num=X|Cas=X|Neg=A|Sem=K
N	N	Gen=X|Num=X|Cas=X|Neg=A|Sem=R
N	N	Gen=X|Num=X|Cas=X|Neg=A|Sem=S
N	N	Gen=X|Num=X|Cas=X|Neg=A|Sem=Y
N	N	Gen=X|Num=X|Cas=X|Neg=A|Sem=m
N	N	Gen=X|Num=X|Cas=X|Neg=A|Var=8
N	N	Gen=X|Num=X|Cas=X|Neg=A|Var=8|Sem=G
N	N	Gen=X|Num=X|Cas=X|Neg=A|Var=8|Sem=K
N	N	Gen=X|Num=X|Cas=X|Neg=A|Var=8|Sem=R
N	N	Gen=X|Num=X|Cas=X|Neg=A|Var=8|Sem=S
N	N	Gen=X|Num=X|Cas=X|Neg=A|Var=8|Sem=Y
N	N	Gen=X|Num=X|Cas=X|Neg=A|Var=8|Sem=m
N	N	Gen=X|Num=X|Cas=X|Neg=N
N	N	Gen=X|Num=X|Cas=X|Neg=N|Var=8
P	0	_
P	1	Gen=F|Num=D|Cas=7|PGe=F|PNu=S|Per=3
P	1	Gen=F|Num=D|Cas=7|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=F|Num=S|Cas=X|PGe=F|PNu=S|Per=3
P	1	Gen=F|Num=S|Cas=X|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=I|Num=S|Cas=4|PGe=F|PNu=S|Per=3
P	1	Gen=I|Num=S|Cas=4|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=M|Num=S|Cas=4|PGe=F|PNu=S|Per=3
P	1	Gen=M|Num=S|Cas=4|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=N|Num=S|Cas=4|PGe=F|PNu=S|Per=3
P	1	Gen=N|Num=S|Cas=4|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=X|Num=P|Cas=1|PGe=F|PNu=S|Per=3
P	1	Gen=X|Num=P|Cas=1|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=X|Num=P|Cas=2|PGe=F|PNu=S|Per=3
P	1	Gen=X|Num=P|Cas=2|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=X|Num=P|Cas=3|PGe=F|PNu=S|Per=3
P	1	Gen=X|Num=P|Cas=3|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=X|Num=P|Cas=4|PGe=F|PNu=S|Per=3
P	1	Gen=X|Num=P|Cas=4|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=X|Num=P|Cas=6|PGe=F|PNu=S|Per=3
P	1	Gen=X|Num=P|Cas=6|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=X|Num=P|Cas=7|PGe=F|PNu=S|Per=3
P	1	Gen=X|Num=P|Cas=7|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=X|Num=X|Cas=X|PGe=X|PNu=P|Per=3
P	1	Gen=X|Num=X|Cas=X|PGe=X|PNu=P|Per=3|Var=2
P	1	Gen=X|Num=X|Cas=X|PGe=Z|PNu=S|Per=3
P	1	Gen=X|Num=X|Cas=X|PGe=Z|PNu=S|Per=3|Var=2
P	1	Gen=Z|Num=S|Cas=1|PGe=F|PNu=S|Per=3
P	1	Gen=Z|Num=S|Cas=1|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=Z|Num=S|Cas=2|PGe=F|PNu=S|Per=3
P	1	Gen=Z|Num=S|Cas=2|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=Z|Num=S|Cas=3|PGe=F|PNu=S|Per=3
P	1	Gen=Z|Num=S|Cas=3|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=Z|Num=S|Cas=6|PGe=F|PNu=S|Per=3
P	1	Gen=Z|Num=S|Cas=6|PGe=F|PNu=S|Per=3|Var=2
P	1	Gen=Z|Num=S|Cas=7|PGe=F|PNu=S|Per=3
P	1	Gen=Z|Num=S|Cas=7|PGe=F|PNu=S|Per=3|Var=2
P	4	Gen=F|Num=D|Cas=7
P	4	Gen=F|Num=D|Cas=7|Var=6
P	4	Gen=F|Num=P|Cas=1
P	4	Gen=F|Num=P|Cas=1|Var=6
P	4	Gen=F|Num=P|Cas=4
P	4	Gen=F|Num=P|Cas=4|Var=6
P	4	Gen=F|Num=S|Cas=1
P	4	Gen=F|Num=S|Cas=1|Var=3
P	4	Gen=F|Num=S|Cas=1|Var=6
P	4	Gen=F|Num=S|Cas=2
P	4	Gen=F|Num=S|Cas=2|Var=6
P	4	Gen=F|Num=S|Cas=3
P	4	Gen=F|Num=S|Cas=3|Var=6
P	4	Gen=F|Num=S|Cas=4
P	4	Gen=F|Num=S|Cas=4|Var=6
P	4	Gen=F|Num=S|Cas=6
P	4	Gen=F|Num=S|Cas=6|Var=6
P	4	Gen=F|Num=S|Cas=7
P	4	Gen=F|Num=S|Cas=7|Var=6
P	4	Gen=I|Num=P|Cas=1
P	4	Gen=I|Num=P|Cas=1|Var=6
P	4	Gen=I|Num=S|Cas=4
P	4	Gen=I|Num=S|Cas=4|Var=6
P	4	Gen=M|Num=P|Cas=1
P	4	Gen=M|Num=P|Cas=1|Var=6
P	4	Gen=M|Num=S|Cas=4
P	4	Gen=M|Num=S|Cas=4|Var=6
P	4	Gen=N|Num=P|Cas=1
P	4	Gen=N|Num=P|Cas=1|Var=6
P	4	Gen=N|Num=P|Cas=1|Var=7
P	4	Gen=N|Num=P|Cas=4
P	4	Gen=N|Num=P|Cas=4|Var=6
P	4	Gen=N|Num=S|Cas=1
P	4	Gen=N|Num=S|Cas=1|Var=6
P	4	Gen=N|Num=S|Cas=4
P	4	Gen=N|Num=S|Cas=4|Var=6
P	4	Gen=X|Num=D|Cas=7
P	4	Gen=X|Num=P|Cas=2
P	4	Gen=X|Num=P|Cas=2|Var=6
P	4	Gen=X|Num=P|Cas=3
P	4	Gen=X|Num=P|Cas=3|Var=6
P	4	Gen=X|Num=P|Cas=6
P	4	Gen=X|Num=P|Cas=6|Var=6
P	4	Gen=X|Num=P|Cas=7
P	4	Gen=X|Num=P|Cas=7|Var=6
P	4	Gen=X|Num=P|Cas=7|Var=7
P	4	Gen=X|Num=X|Cas=X
P	4	Gen=Y|Num=P|Cas=4
P	4	Gen=Y|Num=P|Cas=4|Var=6
P	4	Gen=Y|Num=S|Cas=1
P	4	Gen=Y|Num=S|Cas=1|Var=3
P	4	Gen=Y|Num=S|Cas=1|Var=6
P	4	Gen=Z|Num=S|Cas=2
P	4	Gen=Z|Num=S|Cas=2|Var=6
P	4	Gen=Z|Num=S|Cas=3
P	4	Gen=Z|Num=S|Cas=3|Var=6
P	4	Gen=Z|Num=S|Cas=6
P	4	Gen=Z|Num=S|Cas=6|Var=6
P	4	Gen=Z|Num=S|Cas=7
P	4	Gen=Z|Num=S|Cas=7|Var=6
P	5	Gen=F|Num=S|Cas=2|Per=3
P	5	Gen=F|Num=S|Cas=3|Per=3
P	5	Gen=F|Num=S|Cas=4|Per=3
P	5	Gen=F|Num=S|Cas=4|Per=3|Var=6
P	5	Gen=F|Num=S|Cas=6|Per=3
P	5	Gen=F|Num=S|Cas=7|Per=3
P	5	Gen=N|Num=S|Cas=4|Per=3
P	5	Gen=X|Num=P|Cas=2|Per=3
P	5	Gen=X|Num=P|Cas=3|Per=3
P	5	Gen=X|Num=P|Cas=4|Per=3
P	5	Gen=X|Num=P|Cas=6|Per=3
P	5	Gen=X|Num=P|Cas=7|Per=3
P	5	Gen=Z|Num=S|Cas=2|Per=3
P	5	Gen=Z|Num=S|Cas=2|Per=3|Var=1
P	5	Gen=Z|Num=S|Cas=3|Per=3
P	5	Gen=Z|Num=S|Cas=4|Per=3
P	5	Gen=Z|Num=S|Cas=4|Per=3|Var=1
P	5	Gen=Z|Num=S|Cas=6|Per=3
P	5	Gen=Z|Num=S|Cas=7|Per=3
P	6	Num=X|Cas=2
P	6	Num=X|Cas=3
P	6	Num=X|Cas=4
P	6	Num=X|Cas=6
P	6	Num=X|Cas=7
P	7	Num=S|Cas=3
P	7	Num=S|Cas=3|Per=2
P	7	Num=S|Cas=4
P	7	Num=S|Cas=4|Per=2
P	7	Num=X|Cas=3
P	7	Num=X|Cas=4
P	8	Gen=F|Num=D|Cas=7
P	8	Gen=F|Num=D|Cas=7|Var=6
P	8	Gen=F|Num=P|Cas=1|Var=1
P	8	Gen=F|Num=P|Cas=4|Var=1
P	8	Gen=F|Num=S|Cas=1|Var=1
P	8	Gen=F|Num=S|Cas=2
P	8	Gen=F|Num=S|Cas=2|Var=1
P	8	Gen=F|Num=S|Cas=2|Var=6
P	8	Gen=F|Num=S|Cas=3
P	8	Gen=F|Num=S|Cas=3|Var=1
P	8	Gen=F|Num=S|Cas=3|Var=6
P	8	Gen=F|Num=S|Cas=4
P	8	Gen=F|Num=S|Cas=4|Var=1
P	8	Gen=F|Num=S|Cas=4|Var=6
P	8	Gen=F|Num=S|Cas=5|Var=1
P	8	Gen=F|Num=S|Cas=6
P	8	Gen=F|Num=S|Cas=6|Var=1
P	8	Gen=F|Num=S|Cas=6|Var=6
P	8	Gen=F|Num=S|Cas=7
P	8	Gen=F|Num=S|Cas=7|Var=1
P	8	Gen=H|Num=P|Cas=1
P	8	Gen=H|Num=P|Cas=5
P	8	Gen=H|Num=P|Cas=5|Var=7
P	8	Gen=H|Num=S|Cas=1
P	8	Gen=H|Num=S|Cas=5
P	8	Gen=I|Num=P|Cas=1
P	8	Gen=I|Num=P|Cas=1|Var=1
P	8	Gen=I|Num=P|Cas=1|Var=7
P	8	Gen=I|Num=P|Cas=5
P	8	Gen=I|Num=P|Cas=5|Var=1
P	8	Gen=I|Num=P|Cas=5|Var=7
P	8	Gen=I|Num=S|Cas=4
P	8	Gen=I|Num=S|Cas=4|Var=6
P	8	Gen=M|Num=P|Cas=1
P	8	Gen=M|Num=P|Cas=1|Var=1
P	8	Gen=M|Num=P|Cas=5
P	8	Gen=M|Num=P|Cas=5|Var=1
P	8	Gen=M|Num=S|Cas=4
P	8	Gen=M|Num=S|Cas=4|Var=6
P	8	Gen=N|Num=P|Cas=1|Var=1
P	8	Gen=N|Num=P|Cas=4|Var=1
P	8	Gen=N|Num=P|Cas=5|Var=1
P	8	Gen=N|Num=S|Cas=1|Var=1
P	8	Gen=N|Num=S|Cas=4
P	8	Gen=N|Num=S|Cas=4|Var=1
P	8	Gen=N|Num=S|Cas=5|Var=1
P	8	Gen=X|Num=P|Cas=2
P	8	Gen=X|Num=P|Cas=2|Var=6
P	8	Gen=X|Num=P|Cas=3
P	8	Gen=X|Num=P|Cas=3|Var=6
P	8	Gen=X|Num=P|Cas=4
P	8	Gen=X|Num=P|Cas=4|Var=7
P	8	Gen=X|Num=P|Cas=6
P	8	Gen=X|Num=P|Cas=6|Var=6
P	8	Gen=X|Num=P|Cas=7
P	8	Gen=Y|Num=P|Cas=4|Var=1
P	8	Gen=Y|Num=S|Cas=1
P	8	Gen=Y|Num=S|Cas=1|Var=6
P	8	Gen=Y|Num=S|Cas=5
P	8	Gen=Y|Num=S|Cas=5|Var=6
P	8	Gen=Z|Num=S|Cas=2
P	8	Gen=Z|Num=S|Cas=2|Var=6
P	8	Gen=Z|Num=S|Cas=3
P	8	Gen=Z|Num=S|Cas=3|Var=6
P	8	Gen=Z|Num=S|Cas=6
P	8	Gen=Z|Num=S|Cas=6|Var=6
P	8	Gen=Z|Num=S|Cas=6|Var=7
P	8	Gen=Z|Num=S|Cas=7
P	8	Gen=Z|Num=S|Cas=7|Var=6
P	9	Gen=F|Num=S|Cas=2
P	9	Gen=F|Num=S|Cas=2|Var=2
P	9	Gen=F|Num=S|Cas=3
P	9	Gen=F|Num=S|Cas=3|Var=2
P	9	Gen=F|Num=S|Cas=4
P	9	Gen=F|Num=S|Cas=4|Var=2
P	9	Gen=F|Num=S|Cas=6
P	9	Gen=F|Num=S|Cas=6|Var=2
P	9	Gen=F|Num=S|Cas=7
P	9	Gen=F|Num=S|Cas=7|Var=2
P	9	Gen=N|Num=S|Cas=4
P	9	Gen=N|Num=S|Cas=4|Var=2
P	9	Gen=X|Num=P|Cas=2
P	9	Gen=X|Num=P|Cas=2|Var=2
P	9	Gen=X|Num=P|Cas=3
P	9	Gen=X|Num=P|Cas=3|Var=2
P	9	Gen=X|Num=P|Cas=4
P	9	Gen=X|Num=P|Cas=4|Var=2
P	9	Gen=X|Num=P|Cas=6
P	9	Gen=X|Num=P|Cas=6|Var=2
P	9	Gen=X|Num=P|Cas=7
P	9	Gen=X|Num=P|Cas=7|Var=2
P	9	Gen=Z|Num=S|Cas=2
P	9	Gen=Z|Num=S|Cas=2|Var=1
P	9	Gen=Z|Num=S|Cas=2|Var=2
P	9	Gen=Z|Num=S|Cas=2|Var=3
P	9	Gen=Z|Num=S|Cas=3
P	9	Gen=Z|Num=S|Cas=3|Var=2
P	9	Gen=Z|Num=S|Cas=4
P	9	Gen=Z|Num=S|Cas=4|Var=1
P	9	Gen=Z|Num=S|Cas=4|Var=2
P	9	Gen=Z|Num=S|Cas=4|Var=3
P	9	Gen=Z|Num=S|Cas=6
P	9	Gen=Z|Num=S|Cas=6|Var=2
P	9	Gen=Z|Num=S|Cas=7
P	9	Gen=Z|Num=S|Cas=7|Var=2
P	D	Gen=F|Num=D|Cas=7
P	D	Gen=F|Num=D|Cas=7|Var=2
P	D	Gen=F|Num=D|Cas=7|Var=5
P	D	Gen=F|Num=D|Cas=7|Var=6
P	D	Gen=F|Num=P|Cas=1
P	D	Gen=F|Num=P|Cas=1|Var=5
P	D	Gen=F|Num=P|Cas=1|Var=6
P	D	Gen=F|Num=P|Cas=2
P	D	Gen=F|Num=P|Cas=3
P	D	Gen=F|Num=P|Cas=4
P	D	Gen=F|Num=P|Cas=4|Var=5
P	D	Gen=F|Num=P|Cas=4|Var=6
P	D	Gen=F|Num=P|Cas=6
P	D	Gen=F|Num=P|Cas=7
P	D	Gen=F|Num=S|Cas=1
P	D	Gen=F|Num=S|Cas=1|Var=5
P	D	Gen=F|Num=S|Cas=1|Var=6
P	D	Gen=F|Num=S|Cas=1|Var=8
P	D	Gen=F|Num=S|Cas=2
P	D	Gen=F|Num=S|Cas=2|Var=5
P	D	Gen=F|Num=S|Cas=2|Var=6
P	D	Gen=F|Num=S|Cas=2|Var=8
P	D	Gen=F|Num=S|Cas=3
P	D	Gen=F|Num=S|Cas=3|Var=5
P	D	Gen=F|Num=S|Cas=3|Var=6
P	D	Gen=F|Num=S|Cas=3|Var=8
P	D	Gen=F|Num=S|Cas=4
P	D	Gen=F|Num=S|Cas=4|Var=5
P	D	Gen=F|Num=S|Cas=4|Var=6
P	D	Gen=F|Num=S|Cas=4|Var=8
P	D	Gen=F|Num=S|Cas=6
P	D	Gen=F|Num=S|Cas=6|Var=5
P	D	Gen=F|Num=S|Cas=6|Var=6
P	D	Gen=F|Num=S|Cas=6|Var=8
P	D	Gen=F|Num=S|Cas=7
P	D	Gen=F|Num=S|Cas=7|Var=5
P	D	Gen=F|Num=S|Cas=7|Var=6
P	D	Gen=F|Num=S|Cas=7|Var=8
P	D	Gen=I|Num=P|Cas=1
P	D	Gen=I|Num=P|Cas=1|Var=5
P	D	Gen=I|Num=P|Cas=1|Var=6
P	D	Gen=I|Num=P|Cas=2
P	D	Gen=I|Num=P|Cas=3
P	D	Gen=I|Num=P|Cas=4
P	D	Gen=I|Num=P|Cas=4|Var=5
P	D	Gen=I|Num=P|Cas=4|Var=6
P	D	Gen=I|Num=P|Cas=6
P	D	Gen=I|Num=P|Cas=7
P	D	Gen=I|Num=S|Cas=1|Var=8
P	D	Gen=I|Num=S|Cas=2
P	D	Gen=I|Num=S|Cas=2|Var=8
P	D	Gen=I|Num=S|Cas=3
P	D	Gen=I|Num=S|Cas=3|Var=8
P	D	Gen=I|Num=S|Cas=4
P	D	Gen=I|Num=S|Cas=4|Var=5
P	D	Gen=I|Num=S|Cas=4|Var=6
P	D	Gen=I|Num=S|Cas=4|Var=8
P	D	Gen=I|Num=S|Cas=6
P	D	Gen=I|Num=S|Cas=6|Var=8
P	D	Gen=I|Num=S|Cas=7
P	D	Gen=I|Num=S|Cas=7|Var=8
P	D	Gen=M|Num=P|Cas=1
P	D	Gen=M|Num=P|Cas=1|Var=1
P	D	Gen=M|Num=P|Cas=1|Var=5
P	D	Gen=M|Num=P|Cas=1|Var=6
P	D	Gen=M|Num=P|Cas=2
P	D	Gen=M|Num=P|Cas=3
P	D	Gen=M|Num=P|Cas=4
P	D	Gen=M|Num=P|Cas=4|Var=5
P	D	Gen=M|Num=P|Cas=4|Var=6
P	D	Gen=M|Num=P|Cas=6
P	D	Gen=M|Num=P|Cas=7
P	D	Gen=M|Num=S|Cas=1
P	D	Gen=M|Num=S|Cas=1|Var=8
P	D	Gen=M|Num=S|Cas=2
P	D	Gen=M|Num=S|Cas=2|Var=8
P	D	Gen=M|Num=S|Cas=3
P	D	Gen=M|Num=S|Cas=3|Var=8
P	D	Gen=M|Num=S|Cas=4
P	D	Gen=M|Num=S|Cas=4|Var=5
P	D	Gen=M|Num=S|Cas=4|Var=6
P	D	Gen=M|Num=S|Cas=4|Var=8
P	D	Gen=M|Num=S|Cas=6|Var=8
P	D	Gen=M|Num=S|Cas=7
P	D	Gen=M|Num=S|Cas=7|Var=8
P	D	Gen=N|Num=P|Cas=1
P	D	Gen=N|Num=P|Cas=1|Var=5
P	D	Gen=N|Num=P|Cas=1|Var=6
P	D	Gen=N|Num=P|Cas=2
P	D	Gen=N|Num=P|Cas=3
P	D	Gen=N|Num=P|Cas=4
P	D	Gen=N|Num=P|Cas=4|Var=5
P	D	Gen=N|Num=P|Cas=4|Var=6
P	D	Gen=N|Num=P|Cas=6
P	D	Gen=N|Num=P|Cas=7
P	D	Gen=N|Num=S|Cas=1
P	D	Gen=N|Num=S|Cas=1|Sem=m
P	D	Gen=N|Num=S|Cas=1|Var=2
P	D	Gen=N|Num=S|Cas=1|Var=5
P	D	Gen=N|Num=S|Cas=1|Var=6
P	D	Gen=N|Num=S|Cas=1|Var=8
P	D	Gen=N|Num=S|Cas=2
P	D	Gen=N|Num=S|Cas=2|Var=8
P	D	Gen=N|Num=S|Cas=3
P	D	Gen=N|Num=S|Cas=3|Var=8
P	D	Gen=N|Num=S|Cas=4
P	D	Gen=N|Num=S|Cas=4|Var=2
P	D	Gen=N|Num=S|Cas=4|Var=5
P	D	Gen=N|Num=S|Cas=4|Var=6
P	D	Gen=N|Num=S|Cas=4|Var=8
P	D	Gen=N|Num=S|Cas=6
P	D	Gen=N|Num=S|Cas=6|Var=8
P	D	Gen=N|Num=S|Cas=7
P	D	Gen=N|Num=S|Cas=7|Var=8
P	D	Gen=X|Num=P|Cas=2
P	D	Gen=X|Num=P|Cas=2|Var=1
P	D	Gen=X|Num=P|Cas=2|Var=2
P	D	Gen=X|Num=P|Cas=2|Var=5
P	D	Gen=X|Num=P|Cas=2|Var=6
P	D	Gen=X|Num=P|Cas=3
P	D	Gen=X|Num=P|Cas=3|Var=1
P	D	Gen=X|Num=P|Cas=3|Var=2
P	D	Gen=X|Num=P|Cas=3|Var=5
P	D	Gen=X|Num=P|Cas=3|Var=6
P	D	Gen=X|Num=P|Cas=6
P	D	Gen=X|Num=P|Cas=6|Var=1
P	D	Gen=X|Num=P|Cas=6|Var=2
P	D	Gen=X|Num=P|Cas=6|Var=5
P	D	Gen=X|Num=P|Cas=6|Var=6
P	D	Gen=X|Num=P|Cas=7
P	D	Gen=X|Num=P|Cas=7|Var=2
P	D	Gen=X|Num=P|Cas=7|Var=5
P	D	Gen=X|Num=P|Cas=7|Var=6
P	D	Gen=X|Num=P|Cas=X
P	D	Gen=X|Num=P|Cas=X|Var=8
P	D	Gen=X|Num=S|Cas=1
P	D	Gen=X|Num=S|Cas=2
P	D	Gen=X|Num=S|Cas=3
P	D	Gen=X|Num=S|Cas=4
P	D	Gen=X|Num=S|Cas=6
P	D	Gen=X|Num=S|Cas=7
P	D	Gen=X|Num=S|Cas=X
P	D	Gen=X|Num=S|Cas=X|Var=8
P	D	Gen=X|Num=X|Cas=X
P	D	Gen=X|Num=X|Cas=X|Var=8
P	D	Gen=Y|Num=S|Cas=1
P	D	Gen=Y|Num=S|Cas=1|Var=5
P	D	Gen=Y|Num=S|Cas=1|Var=6
P	D	Gen=Y|Num=S|Cas=4
P	D	Gen=Z|Num=S|Cas=2
P	D	Gen=Z|Num=S|Cas=2|Var=5
P	D	Gen=Z|Num=S|Cas=2|Var=6
P	D	Gen=Z|Num=S|Cas=3
P	D	Gen=Z|Num=S|Cas=3|Var=5
P	D	Gen=Z|Num=S|Cas=3|Var=6
P	D	Gen=Z|Num=S|Cas=6
P	D	Gen=Z|Num=S|Cas=6|Var=1
P	D	Gen=Z|Num=S|Cas=6|Var=2
P	D	Gen=Z|Num=S|Cas=6|Var=5
P	D	Gen=Z|Num=S|Cas=6|Var=6
P	D	Gen=Z|Num=S|Cas=7
P	D	Gen=Z|Num=S|Cas=7|Var=5
P	D	Gen=Z|Num=S|Cas=7|Var=6
P	E	Cas=1
P	E	Cas=2
P	E	Cas=3
P	E	Cas=4
P	E	Cas=6
P	E	Cas=7
P	H	Gen=Z|Num=S|Cas=2|Per=3
P	H	Gen=Z|Num=S|Cas=3|Per=3
P	H	Gen=Z|Num=S|Cas=4|Per=3
P	H	Num=S|Cas=2|Per=1
P	H	Num=S|Cas=2|Per=2
P	H	Num=S|Cas=3|Per=1
P	H	Num=S|Cas=3|Per=2
P	H	Num=S|Cas=4|Per=1
P	H	Num=S|Cas=4|Per=2
P	J	Gen=F|Num=D|Cas=7
P	J	Gen=F|Num=P|Cas=1
P	J	Gen=F|Num=P|Cas=4
P	J	Gen=F|Num=S|Cas=1
P	J	Gen=F|Num=S|Cas=1|Var=2
P	J	Gen=F|Num=S|Cas=2
P	J	Gen=F|Num=S|Cas=2|Var=2
P	J	Gen=F|Num=S|Cas=3
P	J	Gen=F|Num=S|Cas=3|Var=2
P	J	Gen=F|Num=S|Cas=4
P	J	Gen=F|Num=S|Cas=4|Var=2
P	J	Gen=F|Num=S|Cas=6
P	J	Gen=F|Num=S|Cas=7
P	J	Gen=F|Num=S|Cas=7|Var=2
P	J	Gen=I|Num=P|Cas=1
P	J	Gen=I|Num=S|Cas=4
P	J	Gen=I|Num=S|Cas=4|Var=2
P	J	Gen=M|Num=P|Cas=1
P	J	Gen=M|Num=S|Cas=4
P	J	Gen=M|Num=S|Cas=4|Var=2
P	J	Gen=N|Num=P|Cas=1
P	J	Gen=N|Num=P|Cas=4
P	J	Gen=N|Num=S|Cas=1
P	J	Gen=N|Num=S|Cas=1|Var=2
P	J	Gen=N|Num=S|Cas=4
P	J	Gen=N|Num=S|Cas=4|Var=2
P	J	Gen=X|Num=P|Cas=1
P	J	Gen=X|Num=P|Cas=1|Var=2
P	J	Gen=X|Num=P|Cas=2
P	J	Gen=X|Num=P|Cas=2|Var=2
P	J	Gen=X|Num=P|Cas=3
P	J	Gen=X|Num=P|Cas=3|Var=2
P	J	Gen=X|Num=P|Cas=4
P	J	Gen=X|Num=P|Cas=4|Var=2
P	J	Gen=X|Num=P|Cas=6
P	J	Gen=X|Num=P|Cas=7
P	J	Gen=X|Num=P|Cas=7|Var=2
P	J	Gen=Y|Num=P|Cas=4
P	J	Gen=Y|Num=S|Cas=1
P	J	Gen=Y|Num=S|Cas=1|Var=2
P	J	Gen=Z|Num=S|Cas=2
P	J	Gen=Z|Num=S|Cas=2|Var=1
P	J	Gen=Z|Num=S|Cas=2|Var=2
P	J	Gen=Z|Num=S|Cas=2|Var=3
P	J	Gen=Z|Num=S|Cas=3
P	J	Gen=Z|Num=S|Cas=3|Var=2
P	J	Gen=Z|Num=S|Cas=4|Var=1
P	J	Gen=Z|Num=S|Cas=4|Var=3
P	J	Gen=Z|Num=S|Cas=6
P	J	Gen=Z|Num=S|Cas=7
P	J	Gen=Z|Num=S|Cas=7|Var=2
P	K	Gen=M|Cas=1
P	K	Gen=M|Cas=1|Per=2
P	K	Gen=M|Cas=1|Var=1
P	K	Gen=M|Cas=1|Var=2
P	K	Gen=M|Cas=2
P	K	Gen=M|Cas=2|Per=2
P	K	Gen=M|Cas=2|Var=2
P	K	Gen=M|Cas=3
P	K	Gen=M|Cas=3|Per=2
P	K	Gen=M|Cas=3|Var=2
P	K	Gen=M|Cas=4
P	K	Gen=M|Cas=4|Per=2
P	K	Gen=M|Cas=4|Var=2
P	K	Gen=M|Cas=6
P	K	Gen=M|Cas=6|Per=2
P	K	Gen=M|Cas=6|Var=2
P	K	Gen=M|Cas=7
P	K	Gen=M|Cas=7|Per=2
P	K	Gen=M|Cas=7|Var=2
P	L	Gen=F|Num=D|Cas=7
P	L	Gen=F|Num=P|Cas=1
P	L	Gen=F|Num=P|Cas=1|Var=6
P	L	Gen=F|Num=P|Cas=4
P	L	Gen=F|Num=P|Cas=5
P	L	Gen=F|Num=S|Cas=1
P	L	Gen=F|Num=S|Cas=2
P	L	Gen=F|Num=S|Cas=3
P	L	Gen=F|Num=S|Cas=4
P	L	Gen=F|Num=S|Cas=5
P	L	Gen=F|Num=S|Cas=5|Var=1
P	L	Gen=F|Num=S|Cas=6
P	L	Gen=F|Num=S|Cas=7
P	L	Gen=I|Num=P|Cas=1
P	L	Gen=I|Num=P|Cas=1|Var=6
P	L	Gen=I|Num=P|Cas=5
P	L	Gen=I|Num=S|Cas=4
P	L	Gen=I|Num=S|Cas=4|Var=1
P	L	Gen=M|Num=P|Cas=1
P	L	Gen=M|Num=P|Cas=1|Var=6
P	L	Gen=M|Num=P|Cas=5
P	L	Gen=M|Num=S|Cas=4
P	L	Gen=N|Num=P|Cas=1
P	L	Gen=N|Num=P|Cas=1|Var=6
P	L	Gen=N|Num=P|Cas=4
P	L	Gen=N|Num=P|Cas=5
P	L	Gen=N|Num=S|Cas=1
P	L	Gen=N|Num=S|Cas=1|Var=1
P	L	Gen=N|Num=S|Cas=1|Var=6
P	L	Gen=N|Num=S|Cas=4
P	L	Gen=N|Num=S|Cas=4|Sem=m
P	L	Gen=N|Num=S|Cas=4|Var=1
P	L	Gen=N|Num=S|Cas=4|Var=6
P	L	Gen=N|Num=S|Cas=5
P	L	Gen=N|Num=S|Cas=5|Var=1
P	L	Gen=X|Num=P|Cas=2
P	L	Gen=X|Num=P|Cas=3
P	L	Gen=X|Num=P|Cas=6
P	L	Gen=X|Num=P|Cas=7
P	L	Gen=X|Num=P|Cas=7|Var=6
P	L	Gen=X|Num=X|Cas=X
P	L	Gen=X|Num=X|Cas=X|Sem=K
P	L	Gen=X|Num=X|Cas=X|Sem=m
P	L	Gen=X|Num=X|Cas=X|Var=8
P	L	Gen=Y|Num=P|Cas=4
P	L	Gen=Y|Num=S|Cas=1
P	L	Gen=Y|Num=S|Cas=1|Var=1
P	L	Gen=Y|Num=S|Cas=4
P	L	Gen=Y|Num=S|Cas=5
P	L	Gen=Y|Num=S|Cas=5|Var=1
P	L	Gen=Z|Num=S|Cas=2
P	L	Gen=Z|Num=S|Cas=3
P	L	Gen=Z|Num=S|Cas=6
P	L	Gen=Z|Num=S|Cas=7
P	P	Gen=F|Num=P|Cas=1|Per=3
P	P	Gen=F|Num=P|Cas=2|Per=3
P	P	Gen=F|Num=P|Cas=3|Per=3
P	P	Gen=F|Num=P|Cas=4|Per=3
P	P	Gen=F|Num=P|Cas=6|Per=3
P	P	Gen=F|Num=P|Cas=7|Per=3
P	P	Gen=F|Num=P|Cas=X|Per=3
P	P	Gen=F|Num=S|Cas=1|Per=3
P	P	Gen=F|Num=S|Cas=2|Per=3
P	P	Gen=F|Num=S|Cas=3|Per=3
P	P	Gen=F|Num=S|Cas=4|Per=3
P	P	Gen=F|Num=S|Cas=6|Per=3
P	P	Gen=F|Num=S|Cas=7|Per=3
P	P	Gen=F|Num=S|Cas=X|Per=3
P	P	Gen=I|Num=P|Cas=1|Per=3
P	P	Gen=I|Num=P|Cas=2|Per=3
P	P	Gen=I|Num=P|Cas=3|Per=3
P	P	Gen=I|Num=P|Cas=4|Per=3
P	P	Gen=I|Num=P|Cas=6|Per=3
P	P	Gen=I|Num=P|Cas=7|Per=3
P	P	Gen=I|Num=S|Cas=2|Per=3
P	P	Gen=I|Num=S|Cas=3|Per=3
P	P	Gen=I|Num=S|Cas=4|Per=3
P	P	Gen=I|Num=S|Cas=6|Per=3
P	P	Gen=I|Num=S|Cas=7|Per=3
P	P	Gen=M|Num=P|Cas=1|Per=3
P	P	Gen=M|Num=P|Cas=2|Per=3
P	P	Gen=M|Num=P|Cas=3|Per=3
P	P	Gen=M|Num=P|Cas=4|Per=3
P	P	Gen=M|Num=P|Cas=6|Per=3
P	P	Gen=M|Num=P|Cas=7|Per=3
P	P	Gen=M|Num=P|Cas=X|Per=3
P	P	Gen=M|Num=S|Cas=1|Per=3
P	P	Gen=M|Num=S|Cas=1|Per=3|Sem=R
P	P	Gen=M|Num=S|Cas=1|Per=3|Sem=m
P	P	Gen=M|Num=S|Cas=2|Per=3
P	P	Gen=M|Num=S|Cas=3|Per=3
P	P	Gen=M|Num=S|Cas=4|Per=3
P	P	Gen=M|Num=S|Cas=6|Per=3
P	P	Gen=M|Num=S|Cas=7|Per=3
P	P	Gen=M|Num=S|Cas=X|Per=3
P	P	Gen=N|Num=P|Cas=1|Per=3
P	P	Gen=N|Num=P|Cas=2|Per=3
P	P	Gen=N|Num=P|Cas=3|Per=3
P	P	Gen=N|Num=P|Cas=4|Per=3
P	P	Gen=N|Num=P|Cas=7|Per=3
P	P	Gen=N|Num=P|Cas=X|Per=3
P	P	Gen=N|Num=S|Cas=1|Per=3
P	P	Gen=N|Num=S|Cas=2|Per=3
P	P	Gen=N|Num=S|Cas=4|Per=3
P	P	Gen=N|Num=S|Cas=6|Per=3
P	P	Gen=N|Num=S|Cas=7|Per=3
P	P	Gen=N|Num=S|Cas=X|Per=3
P	P	Gen=N|Num=S|Cas=X|Per=3|Sem=R
P	P	Gen=N|Num=S|Cas=X|Per=3|Sem=m
P	P	Gen=X|Num=P|Cas=1|Per=3
P	P	Gen=X|Num=P|Cas=2|Per=3
P	P	Gen=X|Num=P|Cas=3|Per=3
P	P	Gen=X|Num=P|Cas=4|Per=3
P	P	Gen=X|Num=P|Cas=7|Per=3
P	P	Gen=X|Num=P|Cas=X|Per=3
P	P	Gen=X|Num=S|Cas=3|Per=3
P	P	Gen=Y|Num=S|Cas=1|Per=3
P	P	Gen=Y|Num=S|Cas=2|Per=3
P	P	Gen=Y|Num=S|Cas=4|Per=3
P	P	Gen=Z|Num=S|Cas=2|Per=3|Var=1
P	P	Gen=Z|Num=S|Cas=3|Per=3
P	P	Gen=Z|Num=S|Cas=4|Per=3|Var=2
P	P	Gen=Z|Num=S|Cas=7|Per=3
P	P	Num=P|Cas=1|Per=1
P	P	Num=P|Cas=1|Per=2
P	P	Num=P|Cas=2|Per=1
P	P	Num=P|Cas=2|Per=2
P	P	Num=P|Cas=3|Per=1
P	P	Num=P|Cas=3|Per=2
P	P	Num=P|Cas=4|Per=1
P	P	Num=P|Cas=4|Per=2
P	P	Num=P|Cas=5|Per=1
P	P	Num=P|Cas=5|Per=2
P	P	Num=P|Cas=6|Per=1
P	P	Num=P|Cas=6|Per=2
P	P	Num=P|Cas=7|Per=1
P	P	Num=P|Cas=7|Per=1|Var=6
P	P	Num=P|Cas=7|Per=2
P	P	Num=P|Cas=X|Per=1
P	P	Num=P|Cas=X|Per=2
P	P	Num=S|Cas=1|Per=1
P	P	Num=S|Cas=1|Per=2
P	P	Num=S|Cas=1|Per=2|Ten=P|Neg=A|Voi=A
P	P	Num=S|Cas=2|Per=1
P	P	Num=S|Cas=2|Per=2
P	P	Num=S|Cas=3|Per=1
P	P	Num=S|Cas=3|Per=2
P	P	Num=S|Cas=4|Per=1
P	P	Num=S|Cas=4|Per=2
P	P	Num=S|Cas=4|Per=2|Sem=m
P	P	Num=S|Cas=5|Per=1
P	P	Num=S|Cas=5|Per=2
P	P	Num=S|Cas=5|Per=2|Sem=R
P	P	Num=S|Cas=5|Per=2|Sem=m
P	P	Num=S|Cas=6|Per=1
P	P	Num=S|Cas=6|Per=2
P	P	Num=S|Cas=7|Per=1
P	P	Num=S|Cas=7|Per=2
P	P	Num=S|Cas=X|Per=1
P	P	Num=S|Cas=X|Per=2
P	P	Num=X|Cas=X|Per=2
P	Q	Cas=1
P	Q	Cas=1|Var=9
P	Q	Cas=2
P	Q	Cas=2|Var=9
P	Q	Cas=3
P	Q	Cas=3|Var=9
P	Q	Cas=4
P	Q	Cas=4|Var=9
P	Q	Cas=6
P	Q	Cas=6|Var=9
P	Q	Cas=7
P	Q	Cas=7|Var=9
P	Q	Cas=X
P	Q	Cas=X|Var=9
P	S	Gen=F|Num=D|Cas=7|PGe=F|PNu=S|Per=3
P	S	Gen=F|Num=D|Cas=7|PNu=P|Per=1
P	S	Gen=F|Num=D|Cas=7|PNu=P|Per=2
P	S	Gen=F|Num=D|Cas=7|PNu=S|Per=1
P	S	Gen=F|Num=D|Cas=7|PNu=S|Per=1|Var=6
P	S	Gen=F|Num=D|Cas=7|PNu=S|Per=2
P	S	Gen=F|Num=D|Cas=7|PNu=S|Per=2|Var=6
P	S	Gen=F|Num=P|Cas=1|PNu=S|Per=1|Var=1
P	S	Gen=F|Num=P|Cas=1|PNu=S|Per=2|Var=1
P	S	Gen=F|Num=P|Cas=4|PNu=S|Per=1|Var=1
P	S	Gen=F|Num=P|Cas=4|PNu=S|Per=2|Var=1
P	S	Gen=F|Num=P|Cas=X|PNu=P|Per=1
P	S	Gen=F|Num=P|Cas=X|PNu=S|Per=1
P	S	Gen=F|Num=P|Cas=X|PNu=X|Per=1
P	S	Gen=F|Num=S|Cas=1|PNu=S|Per=1|Var=1
P	S	Gen=F|Num=S|Cas=1|PNu=S|Per=2|Var=1
P	S	Gen=F|Num=S|Cas=2|PNu=P|Per=1
P	S	Gen=F|Num=S|Cas=2|PNu=P|Per=2
P	S	Gen=F|Num=S|Cas=2|PNu=S|Per=1
P	S	Gen=F|Num=S|Cas=2|PNu=S|Per=1|Var=1
P	S	Gen=F|Num=S|Cas=2|PNu=S|Per=1|Var=6
P	S	Gen=F|Num=S|Cas=2|PNu=S|Per=2
P	S	Gen=F|Num=S|Cas=2|PNu=S|Per=2|Var=1
P	S	Gen=F|Num=S|Cas=2|PNu=S|Per=2|Var=6
P	S	Gen=F|Num=S|Cas=3|PNu=P|Per=1
P	S	Gen=F|Num=S|Cas=3|PNu=P|Per=2
P	S	Gen=F|Num=S|Cas=3|PNu=S|Per=1
P	S	Gen=F|Num=S|Cas=3|PNu=S|Per=1|Var=1
P	S	Gen=F|Num=S|Cas=3|PNu=S|Per=1|Var=6
P	S	Gen=F|Num=S|Cas=3|PNu=S|Per=2
P	S	Gen=F|Num=S|Cas=3|PNu=S|Per=2|Var=1
P	S	Gen=F|Num=S|Cas=3|PNu=S|Per=2|Var=6
P	S	Gen=F|Num=S|Cas=4|PNu=P|Per=1
P	S	Gen=F|Num=S|Cas=4|PNu=P|Per=1|Var=6
P	S	Gen=F|Num=S|Cas=4|PNu=P|Per=2
P	S	Gen=F|Num=S|Cas=4|PNu=P|Per=2|Var=6
P	S	Gen=F|Num=S|Cas=4|PNu=S|Per=1
P	S	Gen=F|Num=S|Cas=4|PNu=S|Per=1|Var=1
P	S	Gen=F|Num=S|Cas=4|PNu=S|Per=1|Var=6
P	S	Gen=F|Num=S|Cas=4|PNu=S|Per=2
P	S	Gen=F|Num=S|Cas=4|PNu=S|Per=2|Var=1
P	S	Gen=F|Num=S|Cas=4|PNu=S|Per=2|Var=6
P	S	Gen=F|Num=S|Cas=5|PNu=S|Per=1|Var=1
P	S	Gen=F|Num=S|Cas=5|PNu=S|Per=2|Var=1
P	S	Gen=F|Num=S|Cas=6|PNu=P|Per=1
P	S	Gen=F|Num=S|Cas=6|PNu=P|Per=2
P	S	Gen=F|Num=S|Cas=6|PNu=S|Per=1
P	S	Gen=F|Num=S|Cas=6|PNu=S|Per=1|Var=1
P	S	Gen=F|Num=S|Cas=6|PNu=S|Per=1|Var=6
P	S	Gen=F|Num=S|Cas=6|PNu=S|Per=2
P	S	Gen=F|Num=S|Cas=6|PNu=S|Per=2|Var=1
P	S	Gen=F|Num=S|Cas=6|PNu=S|Per=2|Var=6
P	S	Gen=F|Num=S|Cas=7|PNu=P|Per=1
P	S	Gen=F|Num=S|Cas=7|PNu=P|Per=2
P	S	Gen=F|Num=S|Cas=7|PNu=S|Per=1
P	S	Gen=F|Num=S|Cas=7|PNu=S|Per=1|Var=1
P	S	Gen=F|Num=S|Cas=7|PNu=S|Per=2
P	S	Gen=F|Num=S|Cas=7|PNu=S|Per=2|Var=1
P	S	Gen=F|Num=S|Cas=X|PGe=F|PNu=S|Per=3
P	S	Gen=F|Num=S|Cas=X|PNu=P|Per=1
P	S	Gen=F|Num=S|Cas=X|PNu=P|Per=1|Sem=m
P	S	Gen=F|Num=S|Cas=X|PNu=S|Per=1
P	S	Gen=F|Num=S|Cas=X|PNu=X|Per=1
P	S	Gen=H|Num=P|Cas=1|PNu=P|Per=1
P	S	Gen=H|Num=P|Cas=1|PNu=P|Per=2
P	S	Gen=H|Num=P|Cas=1|PNu=S|Per=1
P	S	Gen=H|Num=P|Cas=1|PNu=S|Per=2
P	S	Gen=H|Num=P|Cas=5|PNu=S|Per=1
P	S	Gen=H|Num=P|Cas=5|PNu=S|Per=1|Var=7
P	S	Gen=H|Num=P|Cas=5|PNu=S|Per=2
P	S	Gen=H|Num=P|Cas=5|PNu=S|Per=2|Var=7
P	S	Gen=H|Num=S|Cas=1|PNu=P|Per=1
P	S	Gen=H|Num=S|Cas=1|PNu=P|Per=2
P	S	Gen=H|Num=S|Cas=1|PNu=S|Per=1
P	S	Gen=H|Num=S|Cas=1|PNu=S|Per=2
P	S	Gen=H|Num=S|Cas=5|PNu=P|Per=1
P	S	Gen=H|Num=S|Cas=5|PNu=P|Per=2
P	S	Gen=H|Num=S|Cas=5|PNu=S|Per=1
P	S	Gen=H|Num=S|Cas=5|PNu=S|Per=2
P	S	Gen=I|Num=P|Cas=1|PNu=P|Per=1
P	S	Gen=I|Num=P|Cas=1|PNu=P|Per=2
P	S	Gen=I|Num=P|Cas=1|PNu=S|Per=1
P	S	Gen=I|Num=P|Cas=1|PNu=S|Per=1|Var=1
P	S	Gen=I|Num=P|Cas=1|PNu=S|Per=1|Var=7
P	S	Gen=I|Num=P|Cas=1|PNu=S|Per=2
P	S	Gen=I|Num=P|Cas=1|PNu=S|Per=2|Var=1
P	S	Gen=I|Num=P|Cas=1|PNu=S|Per=2|Var=7
P	S	Gen=I|Num=P|Cas=5|PNu=S|Per=1
P	S	Gen=I|Num=P|Cas=5|PNu=S|Per=1|Var=1
P	S	Gen=I|Num=P|Cas=5|PNu=S|Per=1|Var=7
P	S	Gen=I|Num=P|Cas=5|PNu=S|Per=2
P	S	Gen=I|Num=P|Cas=5|PNu=S|Per=2|Var=1
P	S	Gen=I|Num=P|Cas=5|PNu=S|Per=2|Var=7
P	S	Gen=I|Num=P|Cas=X|PNu=P|Per=1
P	S	Gen=I|Num=P|Cas=X|PNu=S|Per=1
P	S	Gen=I|Num=P|Cas=X|PNu=X|Per=1
P	S	Gen=I|Num=S|Cas=4|PGe=F|PNu=S|Per=3
P	S	Gen=I|Num=S|Cas=4|PNu=P|Per=1
P	S	Gen=I|Num=S|Cas=4|PNu=P|Per=2
P	S	Gen=I|Num=S|Cas=4|PNu=S|Per=1
P	S	Gen=I|Num=S|Cas=4|PNu=S|Per=1|Var=6
P	S	Gen=I|Num=S|Cas=4|PNu=S|Per=2
P	S	Gen=I|Num=S|Cas=4|PNu=S|Per=2|Var=6
P	S	Gen=I|Num=S|Cas=X|PNu=P|Per=1
P	S	Gen=I|Num=S|Cas=X|PNu=S|Per=1
P	S	Gen=I|Num=S|Cas=X|PNu=X|Per=1
P	S	Gen=M|Num=P|Cas=1|PNu=P|Per=1
P	S	Gen=M|Num=P|Cas=1|PNu=P|Per=2
P	S	Gen=M|Num=P|Cas=1|PNu=S|Per=1
P	S	Gen=M|Num=P|Cas=1|PNu=S|Per=1|Var=1
P	S	Gen=M|Num=P|Cas=1|PNu=S|Per=1|Var=7
P	S	Gen=M|Num=P|Cas=1|PNu=S|Per=2
P	S	Gen=M|Num=P|Cas=1|PNu=S|Per=2|Var=1
P	S	Gen=M|Num=P|Cas=5|PNu=P|Per=1
P	S	Gen=M|Num=P|Cas=5|PNu=P|Per=2
P	S	Gen=M|Num=P|Cas=5|PNu=S|Per=1
P	S	Gen=M|Num=P|Cas=5|PNu=S|Per=1|Var=1
P	S	Gen=M|Num=P|Cas=5|PNu=S|Per=1|Var=7
P	S	Gen=M|Num=P|Cas=5|PNu=S|Per=2
P	S	Gen=M|Num=P|Cas=5|PNu=S|Per=2|Var=1
P	S	Gen=M|Num=P|Cas=X|PNu=P|Per=1
P	S	Gen=M|Num=P|Cas=X|PNu=S|Per=1
P	S	Gen=M|Num=P|Cas=X|PNu=X|Per=1
P	S	Gen=M|Num=S|Cas=4|PGe=F|PNu=S|Per=3
P	S	Gen=M|Num=S|Cas=4|PNu=P|Per=1
P	S	Gen=M|Num=S|Cas=4|PNu=P|Per=2
P	S	Gen=M|Num=S|Cas=4|PNu=S|Per=1
P	S	Gen=M|Num=S|Cas=4|PNu=S|Per=1|Var=6
P	S	Gen=M|Num=S|Cas=4|PNu=S|Per=2
P	S	Gen=M|Num=S|Cas=4|PNu=S|Per=2|Var=6
P	S	Gen=M|Num=S|Cas=X|PNu=P|Per=1
P	S	Gen=M|Num=S|Cas=X|PNu=S|Per=1
P	S	Gen=M|Num=S|Cas=X|PNu=X|Per=1
P	S	Gen=N|Num=P|Cas=1|PNu=S|Per=1|Var=1
P	S	Gen=N|Num=P|Cas=1|PNu=S|Per=2|Var=1
P	S	Gen=N|Num=P|Cas=4|PNu=S|Per=1|Var=1
P	S	Gen=N|Num=P|Cas=4|PNu=S|Per=2|Var=1
P	S	Gen=N|Num=P|Cas=5|PNu=S|Per=1|Var=1
P	S	Gen=N|Num=P|Cas=5|PNu=S|Per=2|Var=1
P	S	Gen=N|Num=P|Cas=X|PNu=P|Per=1
P	S	Gen=N|Num=P|Cas=X|PNu=S|Per=1
P	S	Gen=N|Num=P|Cas=X|PNu=X|Per=1
P	S	Gen=N|Num=S|Cas=1|PNu=S|Per=1|Var=1
P	S	Gen=N|Num=S|Cas=1|PNu=S|Per=2|Var=1
P	S	Gen=N|Num=S|Cas=4|PGe=F|PNu=S|Per=3
P	S	Gen=N|Num=S|Cas=4|PNu=P|Per=1
P	S	Gen=N|Num=S|Cas=4|PNu=P|Per=2
P	S	Gen=N|Num=S|Cas=4|PNu=S|Per=1
P	S	Gen=N|Num=S|Cas=4|PNu=S|Per=1|Var=1
P	S	Gen=N|Num=S|Cas=4|PNu=S|Per=2
P	S	Gen=N|Num=S|Cas=4|PNu=S|Per=2|Var=1
P	S	Gen=N|Num=S|Cas=5|PNu=S|Per=1|Var=1
P	S	Gen=N|Num=S|Cas=5|PNu=S|Per=2|Var=1
P	S	Gen=N|Num=S|Cas=X|PNu=P|Per=1
P	S	Gen=N|Num=S|Cas=X|PNu=S|Per=1
P	S	Gen=N|Num=S|Cas=X|PNu=X|Per=1
P	S	Gen=X|Num=P|Cas=1|PGe=F|PNu=S|Per=3
P	S	Gen=X|Num=P|Cas=2|PGe=F|PNu=S|Per=3
P	S	Gen=X|Num=P|Cas=2|PNu=P|Per=1
P	S	Gen=X|Num=P|Cas=2|PNu=P|Per=2
P	S	Gen=X|Num=P|Cas=2|PNu=S|Per=1
P	S	Gen=X|Num=P|Cas=2|PNu=S|Per=1|Var=6
P	S	Gen=X|Num=P|Cas=2|PNu=S|Per=2
P	S	Gen=X|Num=P|Cas=2|PNu=S|Per=2|Var=6
P	S	Gen=X|Num=P|Cas=3|PGe=F|PNu=S|Per=3
P	S	Gen=X|Num=P|Cas=3|PNu=P|Per=1
P	S	Gen=X|Num=P|Cas=3|PNu=P|Per=2
P	S	Gen=X|Num=P|Cas=3|PNu=S|Per=1
P	S	Gen=X|Num=P|Cas=3|PNu=S|Per=1|Var=6
P	S	Gen=X|Num=P|Cas=3|PNu=S|Per=2
P	S	Gen=X|Num=P|Cas=3|PNu=S|Per=2|Var=6
P	S	Gen=X|Num=P|Cas=4|PGe=F|PNu=S|Per=3
P	S	Gen=X|Num=P|Cas=4|PNu=P|Per=1
P	S	Gen=X|Num=P|Cas=4|PNu=P|Per=2
P	S	Gen=X|Num=P|Cas=4|PNu=S|Per=1
P	S	Gen=X|Num=P|Cas=4|PNu=S|Per=1|Var=7
P	S	Gen=X|Num=P|Cas=4|PNu=S|Per=2
P	S	Gen=X|Num=P|Cas=4|PNu=S|Per=2|Var=7
P	S	Gen=X|Num=P|Cas=5|PGe=F|PNu=S|Per=3
P	S	Gen=X|Num=P|Cas=6|PGe=F|PNu=S|Per=3
P	S	Gen=X|Num=P|Cas=6|PNu=P|Per=1
P	S	Gen=X|Num=P|Cas=6|PNu=P|Per=2
P	S	Gen=X|Num=P|Cas=6|PNu=S|Per=1
P	S	Gen=X|Num=P|Cas=6|PNu=S|Per=1|Var=6
P	S	Gen=X|Num=P|Cas=6|PNu=S|Per=2
P	S	Gen=X|Num=P|Cas=6|PNu=S|Per=2|Var=6
P	S	Gen=X|Num=P|Cas=7|PGe=F|PNu=S|Per=3
P	S	Gen=X|Num=P|Cas=7|PGe=F|PNu=S|Per=3|Var=6
P	S	Gen=X|Num=P|Cas=7|PNu=P|Per=1
P	S	Gen=X|Num=P|Cas=7|PNu=P|Per=1|Var=6
P	S	Gen=X|Num=P|Cas=7|PNu=P|Per=2
P	S	Gen=X|Num=P|Cas=7|PNu=P|Per=2|Var=6
P	S	Gen=X|Num=P|Cas=7|PNu=S|Per=1
P	S	Gen=X|Num=P|Cas=7|PNu=S|Per=2
P	S	Gen=X|Num=X|Cas=X|PGe=N|PNu=S|Per=3
P	S	Gen=X|Num=X|Cas=X|PGe=X|PNu=P|Per=3
P	S	Gen=X|Num=X|Cas=X|PGe=Y|PNu=S|Per=3
P	S	Gen=X|Num=X|Cas=X|PGe=Z|PNu=S|Per=3
P	S	Gen=X|Num=X|Cas=X|PNu=P|Per=1
P	S	Gen=X|Num=X|Cas=X|PNu=S|Per=1
P	S	Gen=X|Num=X|Cas=X|PNu=X|Per=2
P	S	Gen=X|Num=X|Cas=X|PNu=X|Per=2|Sem=R
P	S	Gen=X|Num=X|Cas=X|PNu=X|Per=2|Sem=m
P	S	Gen=Y|Num=P|Cas=4|PNu=S|Per=1|Var=1
P	S	Gen=Y|Num=P|Cas=4|PNu=S|Per=1|Var=7
P	S	Gen=Y|Num=P|Cas=4|PNu=S|Per=2|Var=1
P	S	Gen=Y|Num=S|Cas=1|PNu=P|Per=1
P	S	Gen=Y|Num=S|Cas=1|PNu=P|Per=2
P	S	Gen=Y|Num=S|Cas=1|PNu=S|Per=1
P	S	Gen=Y|Num=S|Cas=1|PNu=S|Per=1|Var=6
P	S	Gen=Y|Num=S|Cas=1|PNu=S|Per=2
P	S	Gen=Y|Num=S|Cas=1|PNu=S|Per=2|Var=6
P	S	Gen=Y|Num=S|Cas=5|PNu=P|Per=1
P	S	Gen=Y|Num=S|Cas=5|PNu=P|Per=2
P	S	Gen=Y|Num=S|Cas=5|PNu=S|Per=1
P	S	Gen=Y|Num=S|Cas=5|PNu=S|Per=1|Var=6
P	S	Gen=Y|Num=S|Cas=5|PNu=S|Per=2
P	S	Gen=Y|Num=S|Cas=5|PNu=S|Per=2|Var=6
P	S	Gen=Z|Num=S|Cas=1|PGe=F|PNu=S|Per=3
P	S	Gen=Z|Num=S|Cas=2|PGe=F|PNu=S|Per=3
P	S	Gen=Z|Num=S|Cas=2|PNu=P|Per=1
P	S	Gen=Z|Num=S|Cas=2|PNu=P|Per=2
P	S	Gen=Z|Num=S|Cas=2|PNu=S|Per=1
P	S	Gen=Z|Num=S|Cas=2|PNu=S|Per=1|Var=6
P	S	Gen=Z|Num=S|Cas=2|PNu=S|Per=2
P	S	Gen=Z|Num=S|Cas=2|PNu=S|Per=2|Var=6
P	S	Gen=Z|Num=S|Cas=3|PGe=F|PNu=S|Per=3
P	S	Gen=Z|Num=S|Cas=3|PNu=P|Per=1
P	S	Gen=Z|Num=S|Cas=3|PNu=P|Per=2
P	S	Gen=Z|Num=S|Cas=3|PNu=S|Per=1
P	S	Gen=Z|Num=S|Cas=3|PNu=S|Per=1|Var=6
P	S	Gen=Z|Num=S|Cas=3|PNu=S|Per=2
P	S	Gen=Z|Num=S|Cas=3|PNu=S|Per=2|Var=6
P	S	Gen=Z|Num=S|Cas=5|PGe=F|PNu=S|Per=3
P	S	Gen=Z|Num=S|Cas=6|PGe=F|PNu=S|Per=3
P	S	Gen=Z|Num=S|Cas=6|PNu=P|Per=1
P	S	Gen=Z|Num=S|Cas=6|PNu=P|Per=2
P	S	Gen=Z|Num=S|Cas=6|PNu=S|Per=1
P	S	Gen=Z|Num=S|Cas=6|PNu=S|Per=1|Var=6
P	S	Gen=Z|Num=S|Cas=6|PNu=S|Per=1|Var=7
P	S	Gen=Z|Num=S|Cas=6|PNu=S|Per=2
P	S	Gen=Z|Num=S|Cas=6|PNu=S|Per=2|Var=6
P	S	Gen=Z|Num=S|Cas=6|PNu=S|Per=2|Var=7
P	S	Gen=Z|Num=S|Cas=7|PGe=F|PNu=S|Per=3
P	S	Gen=Z|Num=S|Cas=7|PNu=P|Per=1
P	S	Gen=Z|Num=S|Cas=7|PNu=P|Per=1|Var=6
P	S	Gen=Z|Num=S|Cas=7|PNu=P|Per=1|Var=8
P	S	Gen=Z|Num=S|Cas=7|PNu=P|Per=2
P	S	Gen=Z|Num=S|Cas=7|PNu=P|Per=2|Var=6
P	S	Gen=Z|Num=S|Cas=7|PNu=S|Per=1
P	S	Gen=Z|Num=S|Cas=7|PNu=S|Per=1|Var=6
P	S	Gen=Z|Num=S|Cas=7|PNu=S|Per=2
P	S	Gen=Z|Num=S|Cas=7|PNu=S|Per=2|Var=6
P	W	Cas=1
P	W	Cas=2
P	W	Cas=2|Var=2
P	W	Cas=3
P	W	Cas=3|Var=2
P	W	Cas=4
P	W	Cas=6
P	W	Cas=6|Var=2
P	W	Cas=7
P	W	Cas=7|Var=2
P	W	Cas=X
P	W	Gen=F|Num=D|Cas=7
P	W	Gen=F|Num=D|Cas=7|Var=6
P	W	Gen=F|Num=P|Cas=1
P	W	Gen=F|Num=P|Cas=1|Var=6
P	W	Gen=F|Num=P|Cas=4
P	W	Gen=F|Num=P|Cas=4|Var=6
P	W	Gen=F|Num=P|Cas=5
P	W	Gen=F|Num=P|Cas=5|Var=6
P	W	Gen=F|Num=S|Cas=1
P	W	Gen=F|Num=S|Cas=2
P	W	Gen=F|Num=S|Cas=2|Var=6
P	W	Gen=F|Num=S|Cas=3
P	W	Gen=F|Num=S|Cas=3|Var=6
P	W	Gen=F|Num=S|Cas=4
P	W	Gen=F|Num=S|Cas=5
P	W	Gen=F|Num=S|Cas=6
P	W	Gen=F|Num=S|Cas=6|Var=6
P	W	Gen=F|Num=S|Cas=7
P	W	Gen=I|Num=P|Cas=1
P	W	Gen=I|Num=P|Cas=1|Var=6
P	W	Gen=I|Num=P|Cas=5
P	W	Gen=I|Num=P|Cas=5|Var=6
P	W	Gen=I|Num=S|Cas=4
P	W	Gen=I|Num=S|Cas=4|Var=6
P	W	Gen=M|Cas=1
P	W	Gen=M|Cas=2
P	W	Gen=M|Cas=3
P	W	Gen=M|Cas=4
P	W	Gen=M|Cas=6
P	W	Gen=M|Cas=7
P	W	Gen=M|Num=P|Cas=1
P	W	Gen=M|Num=P|Cas=1|Var=6
P	W	Gen=M|Num=P|Cas=5
P	W	Gen=M|Num=P|Cas=5|Var=6
P	W	Gen=M|Num=S|Cas=4
P	W	Gen=M|Num=S|Cas=4|Var=6
P	W	Gen=N|Num=P|Cas=1
P	W	Gen=N|Num=P|Cas=1|Var=6
P	W	Gen=N|Num=P|Cas=1|Var=7
P	W	Gen=N|Num=P|Cas=4
P	W	Gen=N|Num=P|Cas=4|Var=6
P	W	Gen=N|Num=P|Cas=5
P	W	Gen=N|Num=P|Cas=5|Var=6
P	W	Gen=N|Num=S|Cas=1
P	W	Gen=N|Num=S|Cas=1|Var=6
P	W	Gen=N|Num=S|Cas=4
P	W	Gen=N|Num=S|Cas=4|Var=6
P	W	Gen=N|Num=S|Cas=5
P	W	Gen=N|Num=S|Cas=5|Var=6
P	W	Gen=X|Num=P|Cas=2
P	W	Gen=X|Num=P|Cas=2|Var=6
P	W	Gen=X|Num=P|Cas=3
P	W	Gen=X|Num=P|Cas=3|Var=6
P	W	Gen=X|Num=P|Cas=6
P	W	Gen=X|Num=P|Cas=6|Var=6
P	W	Gen=X|Num=P|Cas=7
P	W	Gen=X|Num=P|Cas=7|Var=6
P	W	Gen=X|Num=P|Cas=7|Var=7
P	W	Gen=Y|Num=P|Cas=4
P	W	Gen=Y|Num=P|Cas=4|Var=6
P	W	Gen=Y|Num=S|Cas=1
P	W	Gen=Y|Num=S|Cas=1|Var=6
P	W	Gen=Y|Num=S|Cas=5
P	W	Gen=Y|Num=S|Cas=5|Var=6
P	W	Gen=Z|Num=S|Cas=2
P	W	Gen=Z|Num=S|Cas=2|Var=6
P	W	Gen=Z|Num=S|Cas=3
P	W	Gen=Z|Num=S|Cas=3|Var=6
P	W	Gen=Z|Num=S|Cas=6
P	W	Gen=Z|Num=S|Cas=6|Var=6
P	W	Gen=Z|Num=S|Cas=6|Var=7
P	W	Gen=Z|Num=S|Cas=7
P	W	Gen=Z|Num=S|Cas=7|Var=6
P	Y	_
P	Z	Cas=1
P	Z	Cas=1|Var=1
P	Z	Cas=1|Var=2
P	Z	Cas=1|Var=4
P	Z	Cas=2
P	Z	Cas=2|Var=1
P	Z	Cas=2|Var=2
P	Z	Cas=3
P	Z	Cas=3|Var=1
P	Z	Cas=3|Var=2
P	Z	Cas=4
P	Z	Cas=4|Var=1
P	Z	Cas=4|Var=2
P	Z	Cas=4|Var=4
P	Z	Cas=5
P	Z	Cas=5|Var=1
P	Z	Cas=5|Var=2
P	Z	Cas=6
P	Z	Cas=6|Var=1
P	Z	Cas=6|Var=2
P	Z	Cas=7
P	Z	Cas=7|Var=1
P	Z	Cas=7|Var=2
P	Z	Gen=F|Num=D|Cas=7
P	Z	Gen=F|Num=D|Cas=7|Var=1
P	Z	Gen=F|Num=D|Cas=7|Var=6
P	Z	Gen=F|Num=P|Cas=1
P	Z	Gen=F|Num=P|Cas=1|Var=1
P	Z	Gen=F|Num=P|Cas=1|Var=6
P	Z	Gen=F|Num=P|Cas=4
P	Z	Gen=F|Num=P|Cas=4|Var=1
P	Z	Gen=F|Num=P|Cas=4|Var=6
P	Z	Gen=F|Num=P|Cas=5
P	Z	Gen=F|Num=P|Cas=5|Var=1
P	Z	Gen=F|Num=P|Cas=5|Var=6
P	Z	Gen=F|Num=S|Cas=1
P	Z	Gen=F|Num=S|Cas=1|Var=1
P	Z	Gen=F|Num=S|Cas=1|Var=6
P	Z	Gen=F|Num=S|Cas=2
P	Z	Gen=F|Num=S|Cas=2|Var=1
P	Z	Gen=F|Num=S|Cas=2|Var=6
P	Z	Gen=F|Num=S|Cas=3
P	Z	Gen=F|Num=S|Cas=3|Var=1
P	Z	Gen=F|Num=S|Cas=3|Var=6
P	Z	Gen=F|Num=S|Cas=4
P	Z	Gen=F|Num=S|Cas=4|Var=1
P	Z	Gen=F|Num=S|Cas=4|Var=6
P	Z	Gen=F|Num=S|Cas=5
P	Z	Gen=F|Num=S|Cas=5|Var=1
P	Z	Gen=F|Num=S|Cas=5|Var=6
P	Z	Gen=F|Num=S|Cas=6
P	Z	Gen=F|Num=S|Cas=6|Var=1
P	Z	Gen=F|Num=S|Cas=6|Var=6
P	Z	Gen=F|Num=S|Cas=7
P	Z	Gen=F|Num=S|Cas=7|Var=1
P	Z	Gen=F|Num=S|Cas=7|Var=6
P	Z	Gen=I|Num=P|Cas=1
P	Z	Gen=I|Num=P|Cas=1|Var=1
P	Z	Gen=I|Num=P|Cas=1|Var=6
P	Z	Gen=I|Num=P|Cas=5
P	Z	Gen=I|Num=P|Cas=5|Var=1
P	Z	Gen=I|Num=P|Cas=5|Var=6
P	Z	Gen=I|Num=S|Cas=4
P	Z	Gen=I|Num=S|Cas=4|Var=1
P	Z	Gen=I|Num=S|Cas=4|Var=6
P	Z	Gen=I|Num=S|Cas=6|Var=7
P	Z	Gen=M|Cas=1
P	Z	Gen=M|Cas=1|Var=1
P	Z	Gen=M|Cas=2
P	Z	Gen=M|Cas=2|Var=1
P	Z	Gen=M|Cas=3
P	Z	Gen=M|Cas=3|Var=1
P	Z	Gen=M|Cas=4
P	Z	Gen=M|Cas=4|Var=1
P	Z	Gen=M|Cas=5
P	Z	Gen=M|Cas=5|Var=1
P	Z	Gen=M|Cas=6
P	Z	Gen=M|Cas=6|Var=1
P	Z	Gen=M|Cas=7
P	Z	Gen=M|Cas=7|Var=1
P	Z	Gen=M|Num=P|Cas=1
P	Z	Gen=M|Num=P|Cas=1|Var=1
P	Z	Gen=M|Num=P|Cas=1|Var=6
P	Z	Gen=M|Num=P|Cas=5
P	Z	Gen=M|Num=P|Cas=5|Var=1
P	Z	Gen=M|Num=P|Cas=5|Var=6
P	Z	Gen=M|Num=S|Cas=4
P	Z	Gen=M|Num=S|Cas=4|Var=1
P	Z	Gen=M|Num=S|Cas=4|Var=6
P	Z	Gen=M|Num=S|Cas=6|Var=7
P	Z	Gen=N|Num=P|Cas=1
P	Z	Gen=N|Num=P|Cas=1|Var=1
P	Z	Gen=N|Num=P|Cas=1|Var=6
P	Z	Gen=N|Num=P|Cas=4
P	Z	Gen=N|Num=P|Cas=4|Var=1
P	Z	Gen=N|Num=P|Cas=4|Var=6
P	Z	Gen=N|Num=P|Cas=5
P	Z	Gen=N|Num=P|Cas=5|Var=1
P	Z	Gen=N|Num=P|Cas=5|Var=6
P	Z	Gen=N|Num=S|Cas=1
P	Z	Gen=N|Num=S|Cas=1|Var=1
P	Z	Gen=N|Num=S|Cas=1|Var=6
P	Z	Gen=N|Num=S|Cas=4
P	Z	Gen=N|Num=S|Cas=4|Var=1
P	Z	Gen=N|Num=S|Cas=4|Var=6
P	Z	Gen=N|Num=S|Cas=5
P	Z	Gen=N|Num=S|Cas=5|Var=1
P	Z	Gen=N|Num=S|Cas=5|Var=6
P	Z	Gen=N|Num=S|Cas=6|Var=7
P	Z	Gen=X|Num=P|Cas=2
P	Z	Gen=X|Num=P|Cas=2|Var=1
P	Z	Gen=X|Num=P|Cas=2|Var=6
P	Z	Gen=X|Num=P|Cas=3
P	Z	Gen=X|Num=P|Cas=3|Var=1
P	Z	Gen=X|Num=P|Cas=3|Var=6
P	Z	Gen=X|Num=P|Cas=6
P	Z	Gen=X|Num=P|Cas=6|Var=1
P	Z	Gen=X|Num=P|Cas=6|Var=6
P	Z	Gen=X|Num=P|Cas=7
P	Z	Gen=X|Num=P|Cas=7|Var=1
P	Z	Gen=X|Num=P|Cas=7|Var=6
P	Z	Gen=X|Num=P|Cas=7|Var=7
P	Z	Gen=X|Num=X|Cas=X
P	Z	Gen=Y|Num=P|Cas=4
P	Z	Gen=Y|Num=P|Cas=4|Var=1
P	Z	Gen=Y|Num=P|Cas=4|Var=6
P	Z	Gen=Y|Num=S|Cas=1
P	Z	Gen=Y|Num=S|Cas=1|Var=1
P	Z	Gen=Y|Num=S|Cas=1|Var=6
P	Z	Gen=Y|Num=S|Cas=2
P	Z	Gen=Y|Num=S|Cas=3
P	Z	Gen=Y|Num=S|Cas=5
P	Z	Gen=Y|Num=S|Cas=5|Var=1
P	Z	Gen=Y|Num=S|Cas=5|Var=6
P	Z	Gen=Z|Num=S|Cas=2
P	Z	Gen=Z|Num=S|Cas=2|Var=1
P	Z	Gen=Z|Num=S|Cas=2|Var=6
P	Z	Gen=Z|Num=S|Cas=3
P	Z	Gen=Z|Num=S|Cas=3|Var=1
P	Z	Gen=Z|Num=S|Cas=3|Var=6
P	Z	Gen=Z|Num=S|Cas=6
P	Z	Gen=Z|Num=S|Cas=6|Var=1
P	Z	Gen=Z|Num=S|Cas=6|Var=6
P	Z	Gen=Z|Num=S|Cas=6|Var=7
P	Z	Gen=Z|Num=S|Cas=7
P	Z	Gen=Z|Num=S|Cas=7|Var=1
P	Z	Gen=Z|Num=S|Cas=7|Var=6
R	F	_
R	R	Cas=1
R	R	Cas=1|Var=8
R	R	Cas=2
R	R	Cas=2|Var=1
R	R	Cas=2|Var=3
R	R	Cas=2|Var=8
R	R	Cas=3
R	R	Cas=3|Var=8
R	R	Cas=4
R	R	Cas=4|Var=8
R	R	Cas=6
R	R	Cas=7
R	R	Cas=7|Var=8
R	R	Cas=X
R	R	Cas=X|Sem=G
R	R	Cas=X|Sem=K
R	R	Cas=X|Sem=R
R	R	Cas=X|Sem=S
R	R	Cas=X|Sem=Y
R	R	Cas=X|Sem=m
R	R	Cas=X|Var=8
R	V	Cas=1
R	V	Cas=2
R	V	Cas=2|Var=1
R	V	Cas=3
R	V	Cas=3|Var=1
R	V	Cas=4
R	V	Cas=4|Var=1
R	V	Cas=6
R	V	Cas=7
T	T	Sem=K
T	T	Sem=S
T	T	Sem=m
T	T	Var=1
T	T	Var=8
T	T	_
V	B	Num=P|Per=1|Ten=F|Neg=A|Voi=A
V	B	Num=P|Per=1|Ten=F|Neg=A|Voi=A|Var=6
V	B	Num=P|Per=1|Ten=F|Neg=A|Voi=A|Var=7
V	B	Num=P|Per=1|Ten=F|Neg=A|Voi=A|Var=8
V	B	Num=P|Per=1|Ten=F|Neg=N|Voi=A
V	B	Num=P|Per=1|Ten=F|Neg=N|Voi=A|Var=6
V	B	Num=P|Per=1|Ten=F|Neg=N|Voi=A|Var=7
V	B	Num=P|Per=1|Ten=F|Neg=N|Voi=A|Var=8
V	B	Num=P|Per=1|Ten=P|Neg=A|Voi=A
V	B	Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=1
V	B	Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=2
V	B	Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=3
V	B	Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=6
V	B	Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=7
V	B	Num=P|Per=1|Ten=P|Neg=N|Voi=A
V	B	Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=1
V	B	Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=2
V	B	Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=3
V	B	Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=6
V	B	Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=7
V	B	Num=P|Per=2|Ten=F|Neg=A|Voi=A
V	B	Num=P|Per=2|Ten=F|Neg=A|Voi=A|Var=7
V	B	Num=P|Per=2|Ten=F|Neg=N|Voi=A
V	B	Num=P|Per=2|Ten=F|Neg=N|Voi=A|Var=7
V	B	Num=P|Per=2|Ten=P|Neg=A|Voi=A
V	B	Num=P|Per=2|Ten=P|Neg=A|Voi=A|Var=1
V	B	Num=P|Per=2|Ten=P|Neg=A|Voi=A|Var=2
V	B	Num=P|Per=2|Ten=P|Neg=A|Voi=A|Var=3
V	B	Num=P|Per=2|Ten=P|Neg=A|Voi=A|Var=6
V	B	Num=P|Per=2|Ten=P|Neg=N|Voi=A
V	B	Num=P|Per=2|Ten=P|Neg=N|Voi=A|Var=1
V	B	Num=P|Per=2|Ten=P|Neg=N|Voi=A|Var=2
V	B	Num=P|Per=2|Ten=P|Neg=N|Voi=A|Var=3
V	B	Num=P|Per=2|Ten=P|Neg=N|Voi=A|Var=6
V	B	Num=P|Per=3|Ten=F|Neg=A|Voi=A
V	B	Num=P|Per=3|Ten=F|Neg=A|Voi=A|Var=1
V	B	Num=P|Per=3|Ten=F|Neg=A|Voi=A|Var=7
V	B	Num=P|Per=3|Ten=F|Neg=N|Voi=A
V	B	Num=P|Per=3|Ten=F|Neg=N|Voi=A|Var=1
V	B	Num=P|Per=3|Ten=F|Neg=N|Voi=A|Var=7
V	B	Num=P|Per=3|Ten=P|Neg=A|Voi=A
V	B	Num=P|Per=3|Ten=P|Neg=A|Voi=A|Sem=m
V	B	Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=1
V	B	Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=2
V	B	Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=3
V	B	Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=4
V	B	Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=5
V	B	Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=6
V	B	Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=7
V	B	Num=P|Per=3|Ten=P|Neg=N|Voi=A
V	B	Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=1
V	B	Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=2
V	B	Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=3
V	B	Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=4
V	B	Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=5
V	B	Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=6
V	B	Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=7
V	B	Num=S|Per=1|Ten=F|Neg=A|Voi=A
V	B	Num=S|Per=1|Ten=F|Neg=A|Voi=A|Var=1
V	B	Num=S|Per=1|Ten=F|Neg=A|Voi=A|Var=7
V	B	Num=S|Per=1|Ten=F|Neg=N|Voi=A
V	B	Num=S|Per=1|Ten=F|Neg=N|Voi=A|Var=1
V	B	Num=S|Per=1|Ten=F|Neg=N|Voi=A|Var=7
V	B	Num=S|Per=1|Ten=P|Neg=A|Voi=A
V	B	Num=S|Per=1|Ten=P|Neg=A|Voi=A|Sem=m
V	B	Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=1
V	B	Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=2
V	B	Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=3
V	B	Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=4
V	B	Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=6
V	B	Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=7
V	B	Num=S|Per=1|Ten=P|Neg=N|Voi=A
V	B	Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=1
V	B	Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=2
V	B	Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=3
V	B	Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=4
V	B	Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=6
V	B	Num=S|Per=2|Ten=F|Neg=A|Voi=A
V	B	Num=S|Per=2|Ten=F|Neg=A|Voi=A|Var=7
V	B	Num=S|Per=2|Ten=F|Neg=N|Voi=A
V	B	Num=S|Per=2|Ten=F|Neg=N|Voi=A|Var=7
V	B	Num=S|Per=2|Ten=P|Neg=A|Voi=A
V	B	Num=S|Per=2|Ten=P|Neg=A|Voi=A|Var=1
V	B	Num=S|Per=2|Ten=P|Neg=A|Voi=A|Var=2
V	B	Num=S|Per=2|Ten=P|Neg=A|Voi=A|Var=3
V	B	Num=S|Per=2|Ten=P|Neg=A|Voi=A|Var=6
V	B	Num=S|Per=2|Ten=P|Neg=A|Voi=A|Var=7
V	B	Num=S|Per=2|Ten=P|Neg=N|Voi=A
V	B	Num=S|Per=2|Ten=P|Neg=N|Voi=A|Var=1
V	B	Num=S|Per=2|Ten=P|Neg=N|Voi=A|Var=2
V	B	Num=S|Per=2|Ten=P|Neg=N|Voi=A|Var=3
V	B	Num=S|Per=2|Ten=P|Neg=N|Voi=A|Var=6
V	B	Num=S|Per=3|Ten=F|Neg=A|Voi=A
V	B	Num=S|Per=3|Ten=F|Neg=A|Voi=A|Var=7
V	B	Num=S|Per=3|Ten=F|Neg=N|Voi=A
V	B	Num=S|Per=3|Ten=F|Neg=N|Voi=A|Var=7
V	B	Num=S|Per=3|Ten=P|Neg=A|Voi=A
V	B	Num=S|Per=3|Ten=P|Neg=A|Voi=A|Sem=m
V	B	Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=1
V	B	Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=2
V	B	Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=3
V	B	Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=4
V	B	Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=5
V	B	Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=6
V	B	Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=7
V	B	Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=8
V	B	Num=S|Per=3|Ten=P|Neg=N|Voi=A
V	B	Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=1
V	B	Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=2
V	B	Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=3
V	B	Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=4
V	B	Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=5
V	B	Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=6
V	B	Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=7
V	B	Num=X|Per=X|Ten=F|Neg=A|Voi=A
V	B	Num=X|Per=X|Ten=F|Neg=N|Voi=A
V	B	Num=X|Per=X|Ten=P|Neg=A|Voi=A
V	B	Num=X|Per=X|Ten=P|Neg=A|Voi=A|Sem=m
V	B	Num=X|Per=X|Ten=P|Neg=N|Voi=A
V	c	Num=P|Per=1
V	c	Num=P|Per=1|Var=6
V	c	Num=P|Per=2
V	c	Num=S|Per=1
V	c	Num=S|Per=1|Var=6
V	c	Num=S|Per=2
V	c	Num=X|Per=3
V	e	Gen=H|Num=S|Neg=A
V	e	Gen=H|Num=S|Neg=A|Var=2
V	e	Gen=H|Num=S|Neg=N
V	e	Gen=H|Num=S|Neg=N|Var=2
V	e	Gen=X|Num=P|Neg=A
V	e	Gen=X|Num=P|Neg=A|Var=2
V	e	Gen=X|Num=P|Neg=N
V	e	Gen=X|Num=P|Neg=N|Var=2
V	e	Gen=Y|Num=S|Neg=A
V	e	Gen=Y|Num=S|Neg=N
V	f	Neg=A
V	f	Neg=A|Sem=K
V	f	Neg=A|Sem=R
V	f	Neg=A|Sem=m
V	f	Neg=A|Var=1
V	f	Neg=A|Var=2
V	f	Neg=A|Var=3
V	f	Neg=A|Var=4
V	f	Neg=A|Var=6
V	f	Neg=A|Var=8
V	f	Neg=N
V	f	Neg=N|Var=1
V	f	Neg=N|Var=2
V	f	Neg=N|Var=3
V	f	Neg=N|Var=4
V	f	Neg=N|Var=6
V	i	Num=P|Per=1|Neg=A
V	i	Num=P|Per=1|Neg=A|Var=1
V	i	Num=P|Per=1|Neg=A|Var=2
V	i	Num=P|Per=1|Neg=A|Var=3
V	i	Num=P|Per=1|Neg=A|Var=6
V	i	Num=P|Per=1|Neg=N
V	i	Num=P|Per=1|Neg=N|Var=1
V	i	Num=P|Per=1|Neg=N|Var=2
V	i	Num=P|Per=1|Neg=N|Var=3
V	i	Num=P|Per=1|Neg=N|Var=6
V	i	Num=P|Per=2|Neg=A
V	i	Num=P|Per=2|Neg=A|Var=1
V	i	Num=P|Per=2|Neg=A|Var=2
V	i	Num=P|Per=2|Neg=A|Var=3
V	i	Num=P|Per=2|Neg=A|Var=6
V	i	Num=P|Per=2|Neg=A|Var=7
V	i	Num=P|Per=2|Neg=N
V	i	Num=P|Per=2|Neg=N|Var=1
V	i	Num=P|Per=2|Neg=N|Var=2
V	i	Num=P|Per=2|Neg=N|Var=3
V	i	Num=P|Per=2|Neg=N|Var=6
V	i	Num=P|Per=2|Neg=N|Var=7
V	i	Num=P|Per=3|Neg=A|Var=1
V	i	Num=P|Per=3|Neg=A|Var=2
V	i	Num=P|Per=3|Neg=A|Var=3
V	i	Num=P|Per=3|Neg=A|Var=4
V	i	Num=P|Per=3|Neg=A|Var=9
V	i	Num=P|Per=3|Neg=N|Var=1
V	i	Num=P|Per=3|Neg=N|Var=2
V	i	Num=P|Per=3|Neg=N|Var=3
V	i	Num=P|Per=3|Neg=N|Var=4
V	i	Num=P|Per=3|Neg=N|Var=9
V	i	Num=S|Per=2|Neg=A
V	i	Num=S|Per=2|Neg=A|Var=1
V	i	Num=S|Per=2|Neg=A|Var=2
V	i	Num=S|Per=2|Neg=A|Var=6
V	i	Num=S|Per=2|Neg=A|Var=7
V	i	Num=S|Per=2|Neg=A|Var=8
V	i	Num=S|Per=2|Neg=N
V	i	Num=S|Per=2|Neg=N|Var=1
V	i	Num=S|Per=2|Neg=N|Var=2
V	i	Num=S|Per=2|Neg=N|Var=6
V	i	Num=S|Per=2|Neg=N|Var=7
V	i	Num=S|Per=3|Neg=A
V	i	Num=S|Per=3|Neg=A|Var=2
V	i	Num=S|Per=3|Neg=A|Var=3
V	i	Num=S|Per=3|Neg=A|Var=4
V	i	Num=S|Per=3|Neg=A|Var=9
V	i	Num=S|Per=3|Neg=N
V	i	Num=S|Per=3|Neg=N|Var=2
V	i	Num=S|Per=3|Neg=N|Var=3
V	i	Num=S|Per=3|Neg=N|Var=4
V	i	Num=S|Per=3|Neg=N|Var=9
V	i	Num=X|Per=2|Neg=A
V	i	Num=X|Per=2|Neg=N
V	m	Gen=H|Num=S|Neg=A
V	m	Gen=H|Num=S|Neg=A|Var=4
V	m	Gen=H|Num=S|Neg=N
V	m	Gen=H|Num=S|Neg=N|Var=4
V	m	Gen=X|Num=P|Neg=A
V	m	Gen=X|Num=P|Neg=A|Var=4
V	m	Gen=X|Num=P|Neg=N
V	m	Gen=X|Num=P|Neg=N|Var=4
V	m	Gen=Y|Num=S|Neg=A
V	m	Gen=Y|Num=S|Neg=A|Var=4
V	m	Gen=Y|Num=S|Neg=N
V	m	Gen=Y|Num=S|Neg=N|Var=4
V	p	Gen=F|Num=S|Per=2|Ten=R|Neg=A|Voi=A
V	p	Gen=F|Num=S|Per=2|Ten=R|Neg=A|Voi=A|Var=1
V	p	Gen=F|Num=S|Per=2|Ten=R|Neg=A|Voi=A|Var=6
V	p	Gen=F|Num=S|Per=2|Ten=R|Neg=N|Voi=A
V	p	Gen=F|Num=S|Per=2|Ten=R|Neg=N|Voi=A|Var=1
V	p	Gen=F|Num=S|Per=2|Ten=R|Neg=N|Voi=A|Var=6
V	p	Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A
V	p	Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=1
V	p	Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	p	Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=6
V	p	Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A
V	p	Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=1
V	p	Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	p	Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=6
V	p	Gen=N|Num=S|Per=2|Ten=R|Neg=A|Voi=A
V	p	Gen=N|Num=S|Per=2|Ten=R|Neg=A|Voi=A|Var=1
V	p	Gen=N|Num=S|Per=2|Ten=R|Neg=A|Voi=A|Var=6
V	p	Gen=N|Num=S|Per=2|Ten=R|Neg=N|Voi=A
V	p	Gen=N|Num=S|Per=2|Ten=R|Neg=N|Voi=A|Var=1
V	p	Gen=N|Num=S|Per=2|Ten=R|Neg=N|Voi=A|Var=6
V	p	Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A
V	p	Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=1
V	p	Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	p	Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=6
V	p	Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A
V	p	Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=1
V	p	Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	p	Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=6
V	p	Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A
V	p	Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A|Var=1
V	p	Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	p	Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A|Var=6
V	p	Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A
V	p	Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A|Var=1
V	p	Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	p	Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A|Var=6
V	p	Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A
V	p	Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=1
V	p	Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	p	Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=6
V	p	Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A
V	p	Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=1
V	p	Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	p	Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=6
V	p	Gen=X|Num=P|Per=X|Ten=R|Neg=A|Voi=A
V	p	Gen=X|Num=P|Per=X|Ten=R|Neg=N|Voi=A
V	p	Gen=X|Num=S|Per=X|Ten=R|Neg=A|Voi=A
V	p	Gen=X|Num=S|Per=X|Ten=R|Neg=N|Voi=A
V	p	Gen=X|Num=X|Per=X|Ten=R|Neg=A|Voi=A
V	p	Gen=X|Num=X|Per=X|Ten=R|Neg=A|Voi=A|Sem=m
V	p	Gen=X|Num=X|Per=X|Ten=R|Neg=N|Voi=A
V	p	Gen=Y|Num=S|Per=2|Ten=R|Neg=A|Voi=A
V	p	Gen=Y|Num=S|Per=2|Ten=R|Neg=A|Voi=A|Var=1
V	p	Gen=Y|Num=S|Per=2|Ten=R|Neg=A|Voi=A|Var=6
V	p	Gen=Y|Num=S|Per=2|Ten=R|Neg=N|Voi=A
V	p	Gen=Y|Num=S|Per=2|Ten=R|Neg=N|Voi=A|Var=1
V	p	Gen=Y|Num=S|Per=2|Ten=R|Neg=N|Voi=A|Var=6
V	p	Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A
V	p	Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=1
V	p	Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=6
V	p	Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=8
V	p	Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=9
V	p	Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A
V	p	Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=1
V	p	Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=6
V	p	Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=8
V	q	Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=2
V	q	Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	q	Gen=M|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=7
V	q	Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=2
V	q	Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	q	Gen=M|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=7
V	q	Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=2
V	q	Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	q	Gen=N|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=7
V	q	Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=2
V	q	Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	q	Gen=N|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=7
V	q	Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A|Var=2
V	q	Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	q	Gen=Q|Num=W|Per=X|Ten=R|Neg=A|Voi=A|Var=7
V	q	Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A|Var=2
V	q	Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	q	Gen=Q|Num=W|Per=X|Ten=R|Neg=N|Voi=A|Var=7
V	q	Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=2
V	q	Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	q	Gen=T|Num=P|Per=X|Ten=R|Neg=A|Voi=A|Var=7
V	q	Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=2
V	q	Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	q	Gen=T|Num=P|Per=X|Ten=R|Neg=N|Voi=A|Var=7
V	q	Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=2
V	q	Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=3
V	q	Gen=Y|Num=S|Per=X|Ten=R|Neg=A|Voi=A|Var=7
V	q	Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=2
V	q	Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=3
V	q	Gen=Y|Num=S|Per=X|Ten=R|Neg=N|Voi=A|Var=7
V	s	Gen=F|Num=S|Cas=4|Per=X|Ten=X|Neg=A|Voi=P
V	s	Gen=F|Num=S|Cas=4|Per=X|Ten=X|Neg=A|Voi=P|Var=2
V	s	Gen=F|Num=S|Cas=4|Per=X|Ten=X|Neg=N|Voi=P
V	s	Gen=F|Num=S|Cas=4|Per=X|Ten=X|Neg=N|Voi=P|Var=2
V	s	Gen=F|Num=S|Per=2|Ten=H|Neg=A|Voi=P
V	s	Gen=F|Num=S|Per=2|Ten=H|Neg=A|Voi=P|Var=2
V	s	Gen=F|Num=S|Per=2|Ten=H|Neg=N|Voi=P
V	s	Gen=F|Num=S|Per=2|Ten=H|Neg=N|Voi=P|Var=2
V	s	Gen=M|Num=P|Per=X|Ten=X|Neg=A|Voi=P
V	s	Gen=M|Num=P|Per=X|Ten=X|Neg=A|Voi=P|Var=2
V	s	Gen=M|Num=P|Per=X|Ten=X|Neg=N|Voi=P
V	s	Gen=M|Num=P|Per=X|Ten=X|Neg=N|Voi=P|Var=2
V	s	Gen=N|Num=S|Per=2|Ten=H|Neg=A|Voi=P
V	s	Gen=N|Num=S|Per=2|Ten=H|Neg=A|Voi=P|Var=2
V	s	Gen=N|Num=S|Per=2|Ten=H|Neg=N|Voi=P
V	s	Gen=N|Num=S|Per=2|Ten=H|Neg=N|Voi=P|Var=2
V	s	Gen=N|Num=S|Per=X|Ten=X|Neg=A|Voi=P
V	s	Gen=N|Num=S|Per=X|Ten=X|Neg=A|Voi=P|Var=2
V	s	Gen=N|Num=S|Per=X|Ten=X|Neg=N|Voi=P
V	s	Gen=N|Num=S|Per=X|Ten=X|Neg=N|Voi=P|Var=2
V	s	Gen=Q|Num=W|Per=X|Ten=X|Neg=A|Voi=P
V	s	Gen=Q|Num=W|Per=X|Ten=X|Neg=A|Voi=P|Var=2
V	s	Gen=Q|Num=W|Per=X|Ten=X|Neg=N|Voi=P
V	s	Gen=Q|Num=W|Per=X|Ten=X|Neg=N|Voi=P|Var=2
V	s	Gen=T|Num=P|Per=X|Ten=X|Neg=A|Voi=P
V	s	Gen=T|Num=P|Per=X|Ten=X|Neg=A|Voi=P|Var=2
V	s	Gen=T|Num=P|Per=X|Ten=X|Neg=N|Voi=P
V	s	Gen=T|Num=P|Per=X|Ten=X|Neg=N|Voi=P|Var=2
V	s	Gen=Y|Num=S|Per=2|Ten=H|Neg=A|Voi=P
V	s	Gen=Y|Num=S|Per=X|Ten=X|Neg=A|Voi=P
V	s	Gen=Y|Num=S|Per=X|Ten=X|Neg=A|Voi=P|Var=2
V	s	Gen=Y|Num=S|Per=X|Ten=X|Neg=N|Voi=P
V	s	Gen=Y|Num=S|Per=X|Ten=X|Neg=N|Voi=P|Var=2
V	t	Num=P|Per=1|Ten=F|Neg=A|Voi=A|Var=2
V	t	Num=P|Per=1|Ten=F|Neg=A|Voi=A|Var=3
V	t	Num=P|Per=1|Ten=F|Neg=N|Voi=A|Var=2
V	t	Num=P|Per=1|Ten=F|Neg=N|Voi=A|Var=3
V	t	Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=2
V	t	Num=P|Per=1|Ten=P|Neg=A|Voi=A|Var=3
V	t	Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=2
V	t	Num=P|Per=1|Ten=P|Neg=N|Voi=A|Var=3
V	t	Num=P|Per=2|Ten=F|Neg=A|Voi=A|Var=2
V	t	Num=P|Per=2|Ten=F|Neg=N|Voi=A|Var=2
V	t	Num=P|Per=2|Ten=P|Neg=A|Voi=A|Var=2
V	t	Num=P|Per=2|Ten=P|Neg=N|Voi=A|Var=2
V	t	Num=P|Per=3|Ten=F|Neg=A|Voi=A|Var=2
V	t	Num=P|Per=3|Ten=F|Neg=A|Voi=A|Var=3
V	t	Num=P|Per=3|Ten=F|Neg=N|Voi=A|Var=2
V	t	Num=P|Per=3|Ten=F|Neg=N|Voi=A|Var=3
V	t	Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=2
V	t	Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=3
V	t	Num=P|Per=3|Ten=P|Neg=A|Voi=A|Var=9
V	t	Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=2
V	t	Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=3
V	t	Num=P|Per=3|Ten=P|Neg=N|Voi=A|Var=9
V	t	Num=S|Per=1|Ten=F|Neg=A|Voi=A|Var=2
V	t	Num=S|Per=1|Ten=F|Neg=A|Voi=A|Var=3
V	t	Num=S|Per=1|Ten=F|Neg=N|Voi=A|Var=2
V	t	Num=S|Per=1|Ten=F|Neg=N|Voi=A|Var=3
V	t	Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=2
V	t	Num=S|Per=1|Ten=P|Neg=A|Voi=A|Var=3
V	t	Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=2
V	t	Num=S|Per=1|Ten=P|Neg=N|Voi=A|Var=3
V	t	Num=S|Per=2|Ten=F|Neg=A|Voi=A|Var=2
V	t	Num=S|Per=2|Ten=F|Neg=N|Voi=A|Var=2
V	t	Num=S|Per=2|Ten=P|Neg=A|Voi=A|Var=2
V	t	Num=S|Per=2|Ten=P|Neg=N|Voi=A|Var=2
V	t	Num=S|Per=3|Ten=F|Neg=A|Voi=A|Var=2
V	t	Num=S|Per=3|Ten=F|Neg=N|Voi=A|Var=2
V	t	Num=S|Per=3|Ten=P|Neg=A|Voi=A|Var=2
V	t	Num=S|Per=3|Ten=P|Neg=N|Voi=A|Var=2
V	~	_
X	@	Sem=G
X	@	Sem=m
X	@	Var=0
X	@	Var=1
X	@	_
X	X	Var=8
X	X	_
X	x	Sem=K
X	x	Sem=m
X	x	_
Z	#	_
Z	:	_
end_of_list
    ;
    # Protect from editors that replace tabs by spaces.
    $list =~ s/ \s+/\t/sg;
    my @list = split(/\r?\n/, $list);
    return \@list;
}



#------------------------------------------------------------------------------
# Consistency check: every CoNLL tag, after stripping the Sem feature and
# converting it to PDT, should be permitted by the PDT driver.
#------------------------------------------------------------------------------
sub consistency_test_against_pdt
{
    my $self = shift;
    my $list = $self->list();
    my $pdtlist = $self->SUPER::list();
    my %pdthash;
    map {$pdthash{$_}++} @{$pdtlist};
    foreach my $tag (@{$list})
    {
        my $pdttag = _conll_to_pdt($tag);
        if(!exists($pdthash{$pdttag}))
        {
            print STDERR ("WARNING: cs::conll tags maps to unknown cs::pdt tag.\n");
            print STDERR ("WARNING: cs::conll $tag\n");
            print STDERR ("WARNING: cs::pdt   $pdttag\n");
            print STDERR ("\n");
        }
    }
}



1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Lingua::Interset::Tagset::CS::Conll - Driver for the Czech tagset of the CoNLL 2006 and 2007 Shared Tasks.

=head1 VERSION

version 2.016

=head1 SYNOPSIS

  use Lingua::Interset::Tagset::CS::Conll;
  my $driver = Lingua::Interset::Tagset::CS::Conll->new();
  my $fs = $driver->decode("N\tN\tGen=M|Num=S|Cas=1|Neg=A");

or

  use Lingua::Interset qw(decode);
  my $fs = decode('cs::conll', "N\tN\tGen=M|Num=S|Cas=1|Neg=A");

=head1 DESCRIPTION

Interset driver for the Czech tagset of the CoNLL 2006 and 2007 Shared Tasks.
CoNLL tagsets in Interset are traditionally three values separated by tabs.
The values come from the CoNLL columns CPOS, POS and FEAT. For Czech,
these values are derived from the tagset of the Prague Dependency Treebank; however,
there is an additional surface feature C<Sem>, which is derived from PDT lemmas.
Thus this driver extends the C<cs::pdt> driver.

=head1 SEE ALSO

L<Lingua::Interset>,
L<Lingua::Interset::Tagset>,
L<Lingua::Interset::Tagset::CS::Pdt>,
L<Lingua::Interset::Tagset::CS::Conll>,
L<Lingua::Interset::FeatureStructure>

=head1 AUTHOR

Dan Zeman <zeman@ufal.mff.cuni.cz>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by Univerzita Karlova v Praze (Charles University in Prague).

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
