###############################################################################
#                                                                             #
#    Copyright © 2012-2013 -- IRB/INSERM                                      #
#                            (Institut de Recherche en Biothérapie /          #
#                             Institut National de la Santé et de la          #
#                             Recherche Médicale)                             #
#                                                                             #
#  Auteurs/Authors:  Jerôme AUDOUX <jerome.audoux@univ-montp2.fr>             #
#                    Nicolas PHILIPPE <nicolas.philippe@inserm.fr>            #
#                                                                             #
#  -------------------------------------------------------------------------  #
#                                                                             #
#  Ce fichier fait partie de la suite CracTools qui contient plusieurs pipeline# 
#  intégrés permettant de traiter les évênements biologiques présents dans du #
#  RNA-Seq. Les CracTools travaillent à partir d'un fichier SAM de CRAC et d'un# 
#  fichier d'annotation au format GFF3.                                       #
#                                                                             #
#  Ce logiciel est régi  par la licence CeCILL  soumise au droit français et  #
#  respectant les principes  de diffusion des logiciels libres.  Vous pouvez  #
#  utiliser, modifier et/ou redistribuer ce programme sous les conditions de  #
#  la licence CeCILL  telle que diffusée par le CEA,  le CNRS et l'INRIA sur  #
#  le site "http://www.cecill.info".                                          #
#                                                                             #
#  En contrepartie de l'accessibilité au code source et des droits de copie,  #
#  de modification et de redistribution accordés par cette licence, il n'est  #
#  offert aux utilisateurs qu'une garantie limitée.  Pour les mêmes raisons,  #
#  seule une responsabilité  restreinte pèse  sur l'auteur du programme,  le  #
#  titulaire des droits patrimoniaux et les concédants successifs.            #
#                                                                             #
#  À  cet égard  l'attention de  l'utilisateur est  attirée sur  les risques  #
#  associés  au chargement,  à  l'utilisation,  à  la modification  et/ou au  #
#  développement  et à la reproduction du  logiciel par  l'utilisateur étant  #
#  donné  sa spécificité  de logiciel libre,  qui peut le rendre  complexe à  #
#  manipuler et qui le réserve donc à des développeurs et des professionnels  #
#  avertis  possédant  des  connaissances  informatiques  approfondies.  Les  #
#  utilisateurs  sont donc  invités  à  charger  et  tester  l'adéquation du  #
#  logiciel  à leurs besoins  dans des conditions  permettant  d'assurer  la  #
#  sécurité de leurs systêmes et ou de leurs données et,  plus généralement,  #
#  à l'utiliser et l'exploiter dans les mêmes conditions de sécurité.         #
#                                                                             #
#  Le fait  que vous puissiez accéder  à cet en-tête signifie  que vous avez  #
#  pris connaissance  de la licence CeCILL,  et que vous en avez accepté les  #
#  termes.                                                                    #
#                                                                             #
#  -------------------------------------------------------------------------  #
#                                                                             #
#  This file is part of the CracTools which provide several integrated        #
#  pipeline to analyze biological events present in RNA-Seq data. CracTools   #
#  work on a SAM file generated by CRAC and an annotation file in GFF3 format.#
#                                                                             #
#  This software is governed by the CeCILL license under French law and       #
#  abiding by the rules of distribution of free software. You can use,        #
#  modify and/ or redistribute the software under the terms of the CeCILL     #
#  license as circulated by CEA, CNRS and INRIA at the following URL          #
#  "http://www.cecill.info".                                                  #
#                                                                             #
#  As a counterpart to the access to the source code and rights to copy,      #
#  modify and redistribute granted by the license, users are provided only    #
#  with a limited warranty and the software's author, the holder of the       #
#  economic rights, and the successive licensors have only limited            #
#  liability.                                                                 #
#                                                                             #
#  In this respect, the user's attention is drawn to the risks associated     #
#  with loading, using, modifying and/or developing or reproducing the        #
#  software by the user in light of its specific status of free software,     #
#  that may mean that it is complicated to manipulate, and that also          #
#  therefore means that it is reserved for developers and experienced         #
#  professionals having in-depth computer knowledge. Users are therefore      #
#  encouraged to load and test the software's suitability as regards their    #
#  requirements in conditions enabling the security of their systems and/or   #
#  data to be ensured and, more generally, to use and operate it in the same  #
#  conditions as regards security.                                            #
#                                                                             #
#  The fact that you are presently reading this means that you have had       #
#  knowledge of the CeCILL license and that you accept its terms.             #
#                                                                             #
###############################################################################

package CracTools::DigitagCT::Analyzer::RNASeq;

use parent 'CracTools::DigitagCT::Analyzer';

use strict;
use warnings;
use Carp;

# load Cractools-core modules
use CracTools::SAMReader;

# load Cractools-digitagCT modules
use CracTools::DigitagCT::Const;

=head1 METHODS

=head2 new

  Arg [rna_seq_sam]     : String - RNA-seq SAM file
  Arg [is_stranded] : boolean (Optional)

=cut

sub new {
  my $class = shift;

  # Creating Annotation Analyzer using the generic analyzer
  my $self  = $class->SUPER::new(@_);

  my %args = @_;

  croak "Missing rna_seq_sam in arguments" unless defined $args{rna_seq_sam};

  $self->name($args{name});
  $self->_init($args{rna_seq_sam},$args{is_stranded});

  return $self;
}

sub name {
  my $self = shift;
  my $new_name = shift;
  if(defined $new_name) {
    $self->{name} = $new_name;
  }
  return $self->{name};
}

sub _init {
  my ($self,$rna_seq_sam,$stranded) = @_;

  # Get const
  my $dge_distance = $CracTools::DigitagCT::Const::DGE_RNASEQ_DISTANCE;

  # TODO Integrate a less naive algorithm using bam to request a specific region!!!
  my $reads_number = 0;

  print STDERR "> Crossing DGE with RNA-Seq.\n";
  my %RNAtags = ();

  my $rna_sam_reader = CracTools::SAMReader->new($rna_seq_sam);

  if(!defined $stranded && $rna_sam_reader->hasCracOption('stranded')) {
    $stranded = 1;
    print STDERR "Option --stranded activated (found in crac command line arguments)\n";
  }      

  my $it = $rna_sam_reader->iterator();
  while(my $line = $it->()) {
    $reads_number++;
    if($line->isClassified('normal')) {
      $RNAtags{$line->rname.'@'.$line->pos.'@'.$line->getStrand}{'OCC'} += 1;
      $RNAtags{$line->rname.'@'.$line->pos.'@'.$line->getStrand}{'TYPE'} = 0;
    }
    if(defined $line->events('Junction')) {
      $RNAtags{$line->rname.'@'.$line->pos.'@'.$line->getStrand}{'OCC'} += 1;
      $RNAtags{$line->rname.'@'.$line->pos.'@'.$line->getStrand}{'TYPE'} = 1;
    }
  }
  my @RNASorted = sort keys %RNAtags; 

  my $tag_it = $self->digitagStruct->iterator();
  my ($total,$total_cross) = (0,0);
  while (my $tag = $tag_it->()) {
    $total++;
    my $samline = $self->digitagStruct->getSAMLine($tag);  
    my ($chr,$pos,$strand) = ($samline->rname,$samline->pos,$samline->getStrand);
    my ($minRNA,$maxRNA,$nb_occ, $nb_splice) = ($dge_distance, 0, 0, 0);
    my ($posMinRNA,$posMaxRNA,$type);
    my $index = bsearchRNA($chr.'@'.$pos.'@'.$strand,\@RNASorted);
    my ($chrRNA,$posRNA,$strandRNA) = split(/@/,$RNASorted[$index]);	
    my $currentDistance = abs($posRNA-$pos);

    while (($chr eq $chrRNA) && $currentDistance < $dge_distance){
      if (!$stranded || ($strandRNA == $strand)){
        $total_cross++;
        if ($currentDistance < $minRNA) {
          $minRNA = $currentDistance;
          $posMinRNA = $posRNA;
        }
        if ($currentDistance > $maxRNA){
          $maxRNA = $currentDistance;
          $posMaxRNA = $posRNA;
        }
        $nb_occ += $RNAtags{$RNASorted[$index]}{'OCC'};
        $type = $RNAtags{$RNASorted[$index]}{'TYPE'};
        if ($type == 1){
          $nb_splice ++;
        }
      }
      if ($strand == 1){
        $index--;
      }else{
        $index++;
      }
      if(defined $RNASorted[$index]) {
        ($chrRNA,$posRNA,$strandRNA) = split(/@/,$RNASorted[$index]);	
        $currentDistance = abs($posRNA-$pos);
      } else {
        last;
      }
    }
    my $nb_occ_rpm = int($nb_occ*$CracTools::DigitagCT::Const::RPM_NORMALIZATION/$reads_number + 0.5);

    my %rna_cross;
    if ($nb_occ_rpm > 0){
      my $nb_splice_rpm = int($nb_splice*$CracTools::DigitagCT::Const::RPM_NORMALIZATION/$reads_number + 0.5);
      $rna_cross{posMinRNA} = $posMinRNA;
      $rna_cross{posMaxRNA} = $posMaxRNA;
      $rna_cross{nbOccRpm} = $nb_occ_rpm;
      $rna_cross{nbSpliceRpm} = $nb_splice_rpm;
    }else{
      $rna_cross{nbOccRpm} = 0;
      $rna_cross{nbSpliceRpm} = 0;
    }
    $self->digitagStruct->addGenericElement($tag,'rna_cross',\%rna_cross);
  }
}

# Binary search, array passed by reference
# search array of string for a given string x
# return index where found or -1 if not found
sub bsearchRNA {
  my ($x, $a) = @_;            # search for x in array a
  my ($l, $u) = (0, @$a - 1);  # lower, upper end of search interval
  my $i;                       # index of probe
  while ($l <= $u) {
    $i = int(($l + $u)/2);
    if ($a->[$i] lt $x) {
      $l = $i+1;
    }
    elsif ($a->[$i] gt $x) {
      $u = $i-1;
    } 
    else {
      return $i; # found
    }
  }
  # Not found, get a $i according to some rules
  my ($strand) = $x =~ /@.*@(\S+)$/;
  if ($strand == 1){
    if ($i > 0 && ($x lt $a->[$i])) {
      return $i-1;
    }
  }else{
    if ($i < (@$a - 1) && ($x gt $a->[$i])) {
      return $i+1;
    }
  }
  return $i;
}

sub getHeaders {
  my $self = shift;
  return ("pos_min_RNA_Seq-".$self->name,
          "pos_max_RNA_Seq-".$self->name,
          "RPM_RNA_Seq($CracTools::DigitagCT::Const::DGE_RNASEQ_DISTANCE bp,3PRIM)-".$self->name,
          "RPM_splice_RNA_Seq($CracTools::DigitagCT::Const::DGE_RNASEQ_DISTANCE bp,3PRIM)-".$self->name,
        );
}

sub getOutput {
  my $self = shift;
  my $tag = shift;
  my @output;
  my $rna_cross = $self->digitagStruct->getGenericElement($tag,'rna_cross');
  return ($rna_cross->{posMinRNA},
          $rna_cross->{posMaxRNA},
          $rna_cross->{nbOccRpm},
          $rna_cross->{nbSpliceRpm},
        );
}

1;
